package com.instabug.bug.model;

import com.instabug.bug.Constants;
import com.instabug.bug.Constants.ReportType;
import com.instabug.bug.settings.BugSettings;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.StringUtility;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

import androidx.annotation.DrawableRes;
import androidx.annotation.Nullable;


/**
 * The Class ReportCategory to create a new ReportCategory.
 * <p>
 * To create a new ReportCategory you must use {@link #getInstance()} then
 * you should use {@link #withIcon(int)} and {@link #withLabel(String)}
 *
 * @author tarek
 * @since 5.0
 */
public class ReportCategory {

    private static final String KEY_LABEL = "name";
    private static final String KEY_SUBS = "subs";
    private static final String KEY_SLUG = "slug";
    private static final String CATEGORY_REPORT_A_PROBLEM = "report-a-problem";
    private static final String CATEGORY_SUGGEST_AN_IMPROVEMENT = "suggest-an-improvement";
    private static final String CATEGORY_ASK_A_QUESTION = "ask-a-question";
    private static final String KEY_DESCRIPTION = "description";
    @Nullable
    private String description;
    @Nullable
    private String label;
    private int icon;
    @Nullable
    private List<ReportCategory> subs;

    private ReportCategory() {

    }

    /**
     * Gets a new instance of ReportCategory model object.
     *
     * @return the instance
     */
    public static ReportCategory getInstance() {
        return new ReportCategory();
    }

    /**
     * sets the reportCategory label
     *
     * @param label the label
     * @return the report category
     */
    public ReportCategory withLabel(String label) {
        this.label = label;
        return this;
    }

    /**
     * sets the reportCategory icon.
     *
     * @param icon the icon
     * @return the report category
     */
    public ReportCategory withIcon(@DrawableRes int icon) {
        this.icon = icon;
        return this;
    }

    /**
     * Gets label.
     *
     * @return the label
     */
    @Nullable
    public String getLabel() {
        return label;
    }

    /**
     * Sets label.
     *
     * @param label the label
     */
    public void setLabel(String label) {
        this.label = StringUtility.ellipsize(label, 35);
    }

    /**
     * Gets icon.
     *
     * @return the icon
     */
    public int getIcon() {
        return icon;
    }

    /**
     * Sets icon.
     *
     * @param icon the icon
     */
    public void setIcon(@DrawableRes int icon) {
        this.icon = icon;
    }

    private static ReportCategory fromJsonObject(JSONObject jsonObject) throws JSONException {
        ReportCategory reportCategory = new ReportCategory();

        if (jsonObject.has(KEY_LABEL)) {
            reportCategory.setLabel(jsonObject.getString(KEY_LABEL));
        }

        if (jsonObject.has(KEY_DESCRIPTION)) {
            reportCategory.setDescription(jsonObject.getString(KEY_DESCRIPTION));
        }

        List<ReportCategory> subCategories = new ArrayList<>();

        if (jsonObject.has(KEY_SUBS)) {
            JSONArray subsJsonArray = jsonObject.getJSONArray(KEY_SUBS);
            for (int j = 0; j < subsJsonArray.length(); j++) {
                subCategories.add(fromJsonObject(subsJsonArray.getJSONObject(j)));
            }
        }

        reportCategory.setSubs(subCategories);
        return reportCategory;
    }

    @Nullable
    public String getDescription() {
        return description;
    }

    @Nullable
    public List<ReportCategory> getSubs() {
        return subs;
    }

    private void setSubs(@Nullable List<ReportCategory> subs) {
        this.subs = subs;
    }

    public void setDescription(String description) {
        this.description = StringUtility.ellipsize(description, 75);
    }

    @Nullable
    public static List<ReportCategory> getSubReportCategories(@ReportType String reportType) {
        List<ReportCategory> remoteSubReportCategories = getRemoteSubReportCategories(reportType);
        return remoteSubReportCategories != null ? remoteSubReportCategories : BugSettings
                .getInstance().getReportCategories();
    }

    @Nullable
    private static List<ReportCategory> getRemoteSubReportCategories(@ReportType String reportType) {

        String remoteReportCategoriesString = BugSettings.getInstance().getRemoteReportCategories();

        if (remoteReportCategoriesString != null) {
            try {

                JSONArray jsonArray = new JSONArray(remoteReportCategoriesString);
                List<ReportCategory> subs;

                switch (reportType) {
                    case ReportType.BUG:
                        subs = getSubCategories(jsonArray, CATEGORY_REPORT_A_PROBLEM);
                        if (subs != null && subs.size() > 0) {
                            return subs;
                        }
                        break;
                    case ReportType.FEEDBACK:
                        subs = getSubCategories(jsonArray, CATEGORY_SUGGEST_AN_IMPROVEMENT);
                        if (subs != null && subs.size() > 0) {
                            return subs;
                        }
                        break;
                    case ReportType.ASK_QUESTION:
                        subs = getSubCategories(jsonArray, CATEGORY_ASK_A_QUESTION);
                        if (subs != null && subs.size() > 0) {
                            return subs;
                        }
                        break;
                }

            } catch (JSONException e) {
                InstabugSDKLogger.e(Constants.LOG_TAG,"error while getRemoteSubReportCategories",e);
            }
        }

        return null;
    }

    @Nullable
    private static List<ReportCategory> getSubCategories(JSONArray jsonArray, String slug)
            throws JSONException {

        for (int i = 0; i < jsonArray.length(); i++) {
            String slugValue = jsonArray.getJSONObject(i).getString(KEY_SLUG);
            if (slug.equals(slugValue)) {
                return ReportCategory.fromJsonObject(jsonArray.getJSONObject(i)).getSubs();
            }
        }

        return null;
    }

    public static boolean hasSubCategories(@ReportType String reportType) {
        List<ReportCategory> remoteBugReportCategories =
                getRemoteSubReportCategories(ReportType.BUG);
        List<ReportCategory> remoteFeedbackReportCategories =
                getRemoteSubReportCategories(ReportType.FEEDBACK);
        List<ReportCategory> remoteAskQuestionCategories =
                getRemoteSubReportCategories(ReportType.ASK_QUESTION);

        boolean isOverridden = false;

        if (remoteBugReportCategories != null && !remoteBugReportCategories.isEmpty()) {

            if (reportType.equals(ReportType.BUG)) {
                return true;
            }
            isOverridden = true;
        }

        if (remoteFeedbackReportCategories != null && !remoteFeedbackReportCategories
                .isEmpty()) {
            if (reportType.equals(ReportType.FEEDBACK)) {
                return true;
            }
            isOverridden = true;
        }
        if (remoteAskQuestionCategories != null && !remoteAskQuestionCategories
                .isEmpty()) {
            if (reportType.equals(ReportType.ASK_QUESTION)) {
                return true;
            }
            isOverridden = true;
        }

        List<ReportCategory> reportCategories = BugSettings.getInstance().getReportCategories();

        return !isOverridden && (reportCategories != null && reportCategories.size() > 0);
    }

}
