package com.instabug.bug.internal.video;

import static android.content.Context.MEDIA_PROJECTION_SERVICE;

import android.annotation.TargetApi;
import android.content.Context;
import android.content.Intent;
import android.media.projection.MediaProjection;
import android.media.projection.MediaProjectionManager;
import android.os.Build;
import android.util.DisplayMetrics;

import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;

import com.instabug.bug.internal.video.customencoding.AudioEncodeConfig;
import com.instabug.bug.internal.video.customencoding.ScreenRecorder;
import com.instabug.bug.internal.video.customencoding.VideoEncodeConfig;
import com.instabug.library.Constants;
import com.instabug.library.Feature;
import com.instabug.library.internal.storage.AttachmentManager;
import com.instabug.library.internal.storage.AttachmentsUtility;
import com.instabug.library.internal.video.InstabugVideoUtils;
import com.instabug.library.internal.video.InternalAutoScreenRecorderHelper;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.DeviceStateProvider;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.MicUtils;
import com.instabug.library.util.threading.PoolProvider;

import java.io.File;
import java.io.IOException;

@TargetApi(Build.VERSION_CODES.LOLLIPOP)
final class ScreenRecordingSession {
    private final Context context;
    private final Listener listener;
    private final Feature.State isAUSAudioCapturingEnabled;
    private final String outputFile;
    private boolean running;
    private final boolean isManuallyRecording;
    @Nullable
    private MediaProjection projection;
    @Nullable
    private ScreenRecorder mRecorder;

    @RequiresApi(api = Build.VERSION_CODES.LOLLIPOP)
    ScreenRecordingSession(Context context, Listener listener, ScreenRecorder.Callback recordingCallBack, int resultCode, Intent data) {
        this.context = context;
        this.listener = listener;
        this.isManuallyRecording = InternalScreenRecordHelper.getInstance().isRecording();
        this.isAUSAudioCapturingEnabled = SettingsManager.getInstance().getAutoScreenRecordingAudioCapturingState();

        // setup output file directory based on the type of current recording instance (Manual/Auto)
        if (isManuallyRecording) {
            outputFile = AttachmentsUtility.getVideoFile(context).getAbsolutePath();
        } else {
            outputFile = AttachmentManager.getAutoScreenRecordingFile(context).getAbsolutePath();
        }
        MediaProjectionManager mMediaProjectionManager = (MediaProjectionManager) context.getSystemService(MEDIA_PROJECTION_SERVICE);
        if (mMediaProjectionManager != null) {
            projection = mMediaProjectionManager.getMediaProjection(resultCode, data);
        }

        VideoEncodeConfig video = createVideoConfig();

        if (isManuallyRecording || isAUSAudioCapturingEnabled == Feature.State.ENABLED) {
            AudioEncodeConfig audio = createAudioConfig();
            mRecorder = new ScreenRecorder(video, audio, projection, outputFile);
        } else {
            mRecorder = new ScreenRecorder(video, null, projection, outputFile);
        }

        startRecording(recordingCallBack);
    }

    @Nullable
    private AudioEncodeConfig createAudioConfig() {
        if (!MicUtils.isAudioPermissionGranted()) return null;

        return new AudioEncodeConfig();
    }

    private VideoEncodeConfig createVideoConfig() {
        // video size
        int[] selectedWithHeight = getSelectedWithHeight();
        int width = selectedWithHeight[0];
        int height = selectedWithHeight[1];
        int density = selectedWithHeight[2];

        return new VideoEncodeConfig(width, height, density);
    }

    private int[] getSelectedWithHeight() {

        DisplayMetrics displayMetrics = DeviceStateProvider.getDisplayMetrics(context);

        int displayWidth = displayMetrics.widthPixels;
        int displayHeight = displayMetrics.heightPixels;
        int displayDensity = displayMetrics.densityDpi;

        return new int[]{displayWidth, displayHeight, displayDensity};
    }

    private void startRecording(ScreenRecorder.Callback recordingCallBack) {
        if (mRecorder != null) {
            mRecorder.setCallback(recordingCallBack);
            mRecorder.start();
        }
        setRunning(true);
        listener.onStart();


        if (isManuallyRecording) {
            InternalScreenRecordHelper.getInstance().startTimerOnRecordingFAB();
        }
        if (isAUSAudioCapturingEnabled == Feature.State.DISABLED) {
            MicUtils.muteMic(context);
        } else {
            MicUtils.unmuteMic(context);
        }


        InstabugSDKLogger.d(Constants.LOG_TAG, "Screen recording started");
    }

    private void stopRecording(ScreenRecorder.Callback recordingCallback) {

        if (!running) {
            return;
        }
        setRunning(false);
        boolean propagate = false;
        try {
            // Stop the projection in order to flush everything to the recorder.
            if (projection != null) {
                projection.stop();
            }

            // Stop the recorder which writes the contents to the file.
            if (mRecorder != null) {
                mRecorder.setCallback(recordingCallback);
            }
            if (mRecorder != null) {
                mRecorder.quit();
            }

            mRecorder = null;

            propagate = true;
        } catch (RuntimeException e) {
            if (e.getMessage() != null) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Error while stopping screen recording");
            }
            if (mRecorder != null) {
                mRecorder.quit();
            }
            propagate = true;
        } finally {
            try {
                // Ensure the listener can tear down its resources regardless if stopping crashes.
                listener.onStop();
            } catch (RuntimeException e) {
                if (propagate) {
                    InstabugSDKLogger.e(Constants.LOG_TAG,"RuntimeException happened " + e.getMessage());
                    //noinspection ThrowFromFinallyBlock
                }
            }
        }
    }

    synchronized void destroy(ScreenRecorder.Callback recordingCallback) {
        if (running) {
            stopRecording(recordingCallback);
        } else {
            listener.onError();
            listener.onEnd();
        }
    }

    void deleteVideoFile() {
        String fileToBeDeleted = outputFile;
        PoolProvider.postIOTask(() -> {
            File file = new File(fileToBeDeleted);
            if (!file.exists()) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Screen recording file doesn't exist - couldn't be deleted");
                return;
            }

            if (!file.delete()) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Screen recording file couldn't be deleted");
            }
            listener.onEnd();
        });
    }

    public synchronized void hold() {
        File recordedVideoFile = new File(outputFile);
        InstabugSDKLogger.v(Constants.LOG_TAG, "Recorded video file size: " + recordedVideoFile.length() / 1024 + " KB");
        if (isManuallyRecording) {
            InternalScreenRecordHelper.getInstance().setAutoScreenRecordingFile(recordedVideoFile);
            InternalScreenRecordHelper.getInstance().onRecordingFinished();
        } else {
            InternalAutoScreenRecorderHelper.getInstance().setAutoScreenRecordingFile(recordedVideoFile);
        }
        listener.onEnd();
    }

    synchronized void trim(final int autoRecordingMaxDuration) {
        PoolProvider.postIOTask(() -> {
            File videoFile = new File(outputFile);
            try {
                File file = InstabugVideoUtils.startTrim(videoFile,
                        AttachmentManager.getAutoScreenRecordingFile(context),
                        autoRecordingMaxDuration);
                InstabugSDKLogger.v(Constants.LOG_TAG, "Recorded video file size after trim: " + file.length() / 1024 + " KB");
                InternalAutoScreenRecorderHelper.getInstance().setAutoScreenRecordingFile(file);
            } catch (IOException | IllegalArgumentException e) {
                e.printStackTrace();
                InternalAutoScreenRecorderHelper.getInstance().setAutoScreenRecordingFile(videoFile);
            }
            listener.onEnd();
        });
    }

    public synchronized void setRunning(boolean running) {
        this.running = running;
    }

    interface Listener {
        /**
         * Invoked immediately prior to the start of recording.
         */
        void onStart();

        /**
         * Invoked immediately after the end of recording.
         */
        void onStop();

        /**
         * Invoked immediately if there's an error with providing a recording
         */
        void onError();

        /**
         * Invoked after all work for this session has completed.
         */
        void onEnd();
    }


}
