package com.instabug.bug.view.disclaimer;

import static com.instabug.library.model.State.KEY_APP_PACKAGE_NAME;
import static com.instabug.library.model.State.KEY_APP_VERSION;
import static com.instabug.library.model.State.KEY_CARRIER;
import static com.instabug.library.model.State.KEY_CONSOLE_LOG;
import static com.instabug.library.model.State.KEY_CURRENT_VIEW;
import static com.instabug.library.model.State.KEY_DENSITY;
import static com.instabug.library.model.State.KEY_DEVICE;
import static com.instabug.library.model.State.KEY_DEVICE_ROOTED;
import static com.instabug.library.model.State.KEY_DURATION;
import static com.instabug.library.model.State.KEY_EMAIL;
import static com.instabug.library.model.State.KEY_INSTABUG_LOG;
import static com.instabug.library.model.State.KEY_LOCALE;
import static com.instabug.library.model.State.KEY_NETWORK_LOGS;
import static com.instabug.library.model.State.KEY_ORIENTATION;
import static com.instabug.library.model.State.KEY_OS;
import static com.instabug.library.model.State.KEY_REPORTED_AT;
import static com.instabug.library.model.State.KEY_SCREEN_SIZE;
import static com.instabug.library.model.State.KEY_SDK_VERSION;
import static com.instabug.library.model.State.KEY_USER_ATTRIBUTES;
import static com.instabug.library.model.State.KEY_USER_DATA;
import static com.instabug.library.model.State.KEY_USER_STEPS;
import static com.instabug.library.model.State.KEY_VISUAL_USER_STEPS;
import static com.instabug.library.model.State.KEY_WIFI_STATE;

import android.net.Uri;
import android.os.Build;
import android.text.Html;
import android.text.Spanned;

import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.bug.LiveBugManager;
import com.instabug.bug.di.ServiceLocator;
import com.instabug.bug.model.Bug;
import com.instabug.bug.utils.DisclaimersLinksUtils;
import com.instabug.library.Feature;
import com.instabug.library.IBGFeature;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.model.State;

import java.util.ArrayList;

/**
 * Created by tarek on 8/14/17.
 */

public class DisclaimerHelper {

    public static final String KEY_BATTERY = "BATTERY";
    public static final String KEY_MEMORY = "MEMORY";
    public static final String KEY_STORAGE = "STORAGE";

    public static final String SCHEME = "instabug-bug";
    public static final String HOST = "instabug-disclaimer.com";
    public static final String PATH = "/disclaimer";
    public static final String URI = SCHEME + "://" + HOST + PATH;
    public static final String LINK = "#metadata-screen";

    public static boolean isDisclaimerUri(@Nullable Uri uri) {
        return uri != null
                && SCHEME.equals(uri.getScheme())
                && HOST.equals(uri.getHost())
                && PATH.equals(uri.getPath());
    }

    public static ArrayList<Disclaimer> getDisclaimers() {
        ArrayList<Disclaimer> disclaimers = new ArrayList<>();
        Bug bug = LiveBugManager.getInstance().getBug();
        if (bug != null && bug.getState() != null) {
            State state = bug.getState();
            if (state != null) {
                if (state.getAppPackageName() != null) {
                    addDisclaimer(new Disclaimer(KEY_APP_PACKAGE_NAME, state.getAppPackageName()),
                            disclaimers);
                }
                if (state.getAppVersion() != null) {
                    addDisclaimer(new Disclaimer(KEY_APP_VERSION, state.getAppVersion()),
                            disclaimers);
                }
                if (state.getBatteryState() != null) {
                    addDisclaimer(new Disclaimer(KEY_BATTERY, state.getBatteryLevel() + "%, " + state.getBatteryState()),
                            disclaimers);
                }
                if (state.getCarrier() != null) {
                    addDisclaimer(new Disclaimer(KEY_CARRIER, state.getCarrier()),
                            disclaimers);
                }
                if (isConsoleLogsEnabled()) {
                    addDisclaimer(new Disclaimer(KEY_CONSOLE_LOG, state.getConsoleLog().toString()).setLog(true),
                            disclaimers);
                }
                if (state.getCurrentView() != null) {
                    addDisclaimer(new Disclaimer(KEY_CURRENT_VIEW, state.getCurrentView()), disclaimers);
                }
                if (state.getScreenDensity() != null) {
                    addDisclaimer(new Disclaimer(KEY_DENSITY, state.getScreenDensity()), disclaimers);
                }
                if (state.getDevice() != null) {
                    addDisclaimer(new Disclaimer(KEY_DEVICE, state.getDevice()), disclaimers);
                }
                addDisclaimer(new Disclaimer(KEY_DEVICE_ROOTED, String.valueOf(state.isDeviceRooted())), disclaimers);
                addDisclaimer(new Disclaimer(KEY_DURATION, String.valueOf(state.getDuration())), disclaimers);
                if (state.getUserEmail() != null) {
                    addDisclaimer(new Disclaimer(KEY_EMAIL, state.getUserEmail()), disclaimers);
                }
                if (state.getInstabugLog() != null) {
                    addDisclaimer(new Disclaimer(KEY_INSTABUG_LOG, state.getInstabugLog()).setLog(true), disclaimers);
                }
                if (state.getLocale() != null) {
                    addDisclaimer(new Disclaimer(KEY_LOCALE, state.getLocale()), disclaimers);
                }
                addDisclaimer(new Disclaimer(KEY_MEMORY, state.getUsedMemory() / 1000f + "/" + state.getTotalMemory()
                        / 1000f + " GB"), disclaimers);
                if (state.getNetworkLogs() != null) {
                    addDisclaimer(new Disclaimer(KEY_NETWORK_LOGS, state.getNetworkLogs()).setLog(true), disclaimers);
                }
                if (state.getScreenOrientation() != null) {
                    addDisclaimer(new Disclaimer(KEY_ORIENTATION, state.getScreenOrientation()), disclaimers);
                }
                if (state.getOS() != null) {
                    addDisclaimer(new Disclaimer(KEY_OS, state.getOS()), disclaimers);
                }
                addDisclaimer(new Disclaimer(KEY_REPORTED_AT, String.valueOf(state.getReportedAt())), disclaimers);
                if (state.getScreenSize() != null) {
                    addDisclaimer(new Disclaimer(KEY_SCREEN_SIZE, state.getScreenSize()), disclaimers);
                }
                if (state.getSdkVersion() != null) {
                    addDisclaimer(new Disclaimer(KEY_SDK_VERSION, state.getSdkVersion()), disclaimers);
                }
                addDisclaimer(new Disclaimer(KEY_STORAGE, state.getUsedStorage() / 1000f + "/" + state.getTotalStorage
                                () / 1000f + " GB"),
                        disclaimers);
                if (state.getUserAttributes() != null) {
                    addDisclaimer(new Disclaimer(KEY_USER_ATTRIBUTES, state.getUserAttributes()).setLog(true), disclaimers);
                }
                if (state.getUserData() != null) {
                    addDisclaimer(new Disclaimer(KEY_USER_DATA, state.getUserData()).setLog(true), disclaimers);
                }
                if (isUserTrackingStepsEnabled()) {
                    addDisclaimer(new Disclaimer(KEY_USER_STEPS, state.getUserSteps().toString()).setLog(true), disclaimers);
                }
                if (ServiceLocator.getConfigurationsProvider().isReproStepsEnabled()) {
                    addDisclaimer(new Disclaimer(KEY_VISUAL_USER_STEPS, state.getVisualUserSteps()).setLog(true), disclaimers);
                }
                addDisclaimer(new Disclaimer(KEY_WIFI_STATE, String.valueOf(state.isWifiEnable())), disclaimers);
            }
        }
        return disclaimers;
    }

    @VisibleForTesting
    static void addDisclaimer(Disclaimer disclaimer, ArrayList<Disclaimer> disclaimers) {
        if (disclaimer.getValue() != null && !disclaimer.getValue().isEmpty() && !disclaimer
                .getValue()
                .equals("{}") && !disclaimer.getValue().equals("[]")) {
            disclaimer.setKey(disclaimer.getKey().toUpperCase().replace('_', ' '));
            disclaimers.add(disclaimer);
        }
    }

    private static boolean isUserTrackingStepsEnabled() {
        return InstabugCore.getFeatureState(IBGFeature.TRACK_USER_STEPS) == Feature.State.ENABLED;
    }

    private static boolean isConsoleLogsEnabled() {
        return InstabugCore.getFeatureState(IBGFeature.CONSOLE_LOGS) == Feature.State.ENABLED;
    }

    public static Spanned getSpannedDisclaimer(String disclaimer) {
        String htmlDisclaimer = convertToHTML(disclaimer);
        Spanned spanned;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            spanned = Html.fromHtml(htmlDisclaimer, Html.FROM_HTML_MODE_LEGACY);
        } else {
            spanned = Html.fromHtml(htmlDisclaimer);
        }
        return spanned;
    }

    private static String convertToHTML(String disclaimer) {
        return DisclaimersLinksUtils.convertToHTML(disclaimer, LINK, URI);
    }

    public static boolean isDisclaimerAvailable(@Nullable String disclaimer) {
        return disclaimer != null && !disclaimer.isEmpty() &&
                InstabugCore.getFeatureState(IBGFeature.DISCLAIMER) == Feature.State.ENABLED;
    }
}
