package com.instabug.bug.invocation.invoker;

import static android.os.Build.VERSION_CODES.TIRAMISU;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.ContentResolver;
import android.os.Build;
import android.os.Handler;
import android.os.HandlerThread;
import android.provider.MediaStore;

import androidx.annotation.Nullable;

import com.instabug.bug.invocation.InvocationListener;
import com.instabug.bug.invocation.InvocationManager;
import com.instabug.bug.utils.PermissionsUtils;
import com.instabug.library.Constants;
import com.instabug.library.Instabug;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.eventbus.coreeventbus.IBGCoreEventSubscriber;
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent;
import com.instabug.library.invocation.InstabugInvocationEvent;
import com.instabug.library.tracking.InstabugInternalTrackingDelegate;
import com.instabug.library.util.InstabugSDKLogger;

/**
 * @author mesbah
 */
public class ScreenshotGestureInvoker implements AbstractInvoker<Void> {
    @Nullable
    private ContentResolver contentResolver;
    private ScreenshotObserver screenshotObserver;

    private HandlerThread handlerThread;
    private Handler handler;
    private boolean shouldAskForPermission = true;
    private volatile boolean active;

    @SuppressLint({"ERADICATE_FIELD_NOT_INITIALIZED", "ERADICATE_PARAMETER_NOT_NULLABLE"})
    public ScreenshotGestureInvoker(InvocationListener invocationListener) {
        if (Instabug.getApplicationContext() == null) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "ScreenshotGestureInvoker() called with null context");
            return;
        }

        contentResolver = Instabug.getApplicationContext().getContentResolver();

        handlerThread = new HandlerThread("ScreenshotObserver");
        handlerThread.start();

        handler = new Handler(handlerThread.getLooper());
        screenshotObserver = new ScreenshotObserver(handler, contentResolver, invocationListener);

        subscribeToSessionState();
    }

    private void subscribeToSessionState() {
        IBGCoreEventSubscriber.subscribe(event -> {
            if (event instanceof IBGSdkCoreEvent.Session.SessionStarted) {
                shouldAskForPermission = true;
            }
        });
    }

    @Override
    public void listen() {
        if (shouldAskForPermission && !isStoragePermissionGranted()) {
            requestExternalStorage();
        } else {
            startObserving();
        }
    }

    private void startObserving() {
        if (isStoragePermissionGranted() && contentResolver != null) {
            contentResolver.registerContentObserver(
                    MediaStore.Images.Media.EXTERNAL_CONTENT_URI,
                    true,
                    screenshotObserver
            );
            active = true;
        }
    }

    private boolean isStoragePermissionGranted() {
        Activity currentActivity
                = InstabugInternalTrackingDelegate.getInstance().getCurrentActivity();
        boolean isStoragePermissionGranted = PermissionsUtils.isMediaStoragePermissionGranted(currentActivity);
        InstabugSDKLogger.v(Constants.LOG_TAG, "isStoragePermissionGranted = [" + isStoragePermissionGranted + "]");
        return isStoragePermissionGranted;
    }

    @Override
    public void handle(Void input) {

    }

    @Override
    public void sleep() {
        if (contentResolver != null) {
            contentResolver.unregisterContentObserver(screenshotObserver);
            active = false;
        }
    }

    @Override
    public boolean isActive() {
        return active;
    }

    private void requestExternalStorage() {
        InstabugInvocationEvent[] events =
                InvocationManager.getInstance().getCurrentInstabugInvocationEvents();
        if (events == null) return;

        for (InstabugInvocationEvent event : events) {
            if (event == InstabugInvocationEvent.SCREENSHOT) {
                Activity currentActivity
                        = InstabugInternalTrackingDelegate.getInstance().getCurrentActivity();
                if (currentActivity != null) {
                    // Storage Permissions
                    requestPermission(currentActivity);
                    shouldAskForPermission = false;
                }
                break;
            }
        }
    }

    private static void requestPermission(Activity currentActivity) {
        int REQUEST_EXTERNAL_STORAGE = 1;
        try {
            PermissionsUtils.requestPermission(currentActivity, PermissionsUtils.mediaStoragePermission(), REQUEST_EXTERNAL_STORAGE, null, null);
        } catch (Exception exception) {
            if (Build.VERSION.SDK_INT == TIRAMISU) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "trying to request READ_MEDIA_IMAGES Without adding it to the manifest ", exception);
            } else {
                InstabugCore.reportError(exception, "Something Went Wrong while requesting Storage Permission");
            }
        }
    }
}