package com.instabug.chat.annotation;

import android.animation.Animator;
import android.annotation.SuppressLint;
import android.content.Context;
import android.content.res.Configuration;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Matrix;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.RectF;
import android.graphics.drawable.GradientDrawable;
import android.os.Build;
import android.os.Handler;
import android.util.AttributeSet;
import android.util.DisplayMetrics;
import android.util.TypedValue;
import android.view.Gravity;
import android.view.View;
import android.widget.LinearLayout;

import androidx.annotation.ColorInt;
import androidx.annotation.Nullable;
import androidx.annotation.StringRes;
import androidx.core.view.AccessibilityDelegateCompat;
import androidx.core.view.ViewCompat;
import androidx.core.view.accessibility.AccessibilityNodeInfoCompat;

import com.instabug.bug.R;
import com.instabug.library.util.AccessibilityUtils;
import com.instabug.library.util.AttrResolver;

/**
 * Created by tarek on 7/11/17.
 */

public class ShapeSuggestionsLayout extends LinearLayout {

    private final static long SHOW_TIME = 3000L;
    private final static float CORNER_RADIUS = 4.0f;
    private float cornerRadius;
    private final static float BORDER = 1.5f;
    @Nullable
    private OnShapeSuggestionSelectedListener onShapeSuggestionSelectedListener;
    private boolean hidden;

    private final Runnable hideRunnable = new Runnable() {
        @Override
        public void run() {
            hide();
        }
    };

    @Nullable
    private Handler hideHandler = new Handler();

    public ShapeSuggestionsLayout(Context context) {
        this(context, null);
    }

    public ShapeSuggestionsLayout(Context context, @Nullable AttributeSet attrs) {
        this(context, attrs, 0);
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public ShapeSuggestionsLayout(Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init(context);
    }

    @Override
    protected void onConfigurationChanged(Configuration newConfig) {
        super.onConfigurationChanged(newConfig);
        if (!hidden) {
            hide();
            show();
        }
    }

    private void init(Context context) {

        setVisibility(INVISIBLE);

        setGravity(Gravity.CENTER);

        DisplayMetrics metrics = context.getResources().getDisplayMetrics();

        cornerRadius = TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, CORNER_RADIUS,
                metrics);

        int border = (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, BORDER,
                metrics);

        GradientDrawable shapeDrawable = new GradientDrawable();
        shapeDrawable.setColor(AttrResolver.getBackgroundColor(getContext()));
        shapeDrawable.setCornerRadius(cornerRadius);
        shapeDrawable.setStroke(border, AttrResolver.getDividerColor(getContext()));

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {

            setBackground(shapeDrawable);
        } else {
            setBackgroundDrawable(shapeDrawable);
        }
    }


    public void setOnShapeSelectedListener(OnShapeSuggestionSelectedListener
                                                   onShapeSuggestionSelectedListener) {
        this.onShapeSuggestionSelectedListener = onShapeSuggestionSelectedListener;
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public void show() {
        if (!hidden) {
            return;
        }
        hidden = false;
        setVisibility(VISIBLE);

        animate().alpha(1).setDuration(400).setListener(null);

        // Only hides if accessibility services is off
        // If on, will not hide until a shape is selected
        if (hideHandler != null && !AccessibilityUtils.isTalkbackEnabled()) {
            hideHandler.postDelayed(hideRunnable, SHOW_TIME);
        }
    }

    @Override
    protected void onDetachedFromWindow() {
        super.onDetachedFromWindow();
        if (hideHandler != null) {
            hideHandler.removeCallbacks(hideRunnable);
            hideHandler = null;
        }
    }

    void hide() {
        if (hideHandler != null) {
            hideHandler.removeCallbacks(hideRunnable);
        }

        if (hidden) {
            return;
        }

        hidden = true;
        animate().alpha(0).setDuration(400).setListener(new Animator.AnimatorListener() {
            @Override
            public void onAnimationStart(Animator animation) {

            }

            @Override
            public void onAnimationEnd(Animator animation) {
                setVisibility(INVISIBLE);
                removeAllViews();

            }

            @Override
            public void onAnimationCancel(Animator animation) {

            }

            @Override
            public void onAnimationRepeat(Animator animation) {

            }
        });
    }

    interface OnShapeSuggestionSelectedListener {
        void onShapeSelected(int index);
    }

    public void addShapeSuggestion(@StringRes int shapeAccessibilityContentDescription, Path path) {
        PathView pathView = new PathView(getContext(), path, AttrResolver.getTintingColor
                (getContext()));
        pathView.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View view) {
                int selectedChildIndex = indexOfChild(view);
                selectChild(selectedChildIndex);

                if (onShapeSuggestionSelectedListener != null) {
                    onShapeSuggestionSelectedListener.onShapeSelected(selectedChildIndex);
                }

                hide();
            }
        });

        pathView.setFocusable(true);
        pathView.setClickable(true);
        pathView.setContentDescription(getContext().getString(shapeAccessibilityContentDescription));
        ViewCompat.setAccessibilityDelegate(pathView, new AccessibilityDelegateCompat() {
            @Override
            public void onInitializeAccessibilityNodeInfo(View host, AccessibilityNodeInfoCompat info) {
                super.onInitializeAccessibilityNodeInfo(host, info);
                info.setRoleDescription("Button");
            }
        });

        addView(pathView);

        // Select the first child
        selectChild(0);
    }

    private void selectChild(int viewIndex) {
        //Clear children background
        for (int i = 0; i < getChildCount(); i++) {
            getChildAt(i).setBackgroundColor(Color.TRANSPARENT);
        }
        getChildAt(viewIndex).setBackgroundColor(AttrResolver.getDividerColor(getContext()));
    }

    private final class PathView extends View {

        private Path path;
        @Nullable
        private Path resizedPath;
        private Paint paint;
        @Nullable
        private RectF viewBounds;

        @ColorInt
        private int backgroundColor;

        @ColorInt
        private int pathColor;

        public PathView(Context context, Path path, int pathColor) {
            super(context);
            this.path = path;
            paint = new Paint(Paint.ANTI_ALIAS_FLAG);
            paint.setStrokeWidth(4);
            this.pathColor = pathColor;
        }

        @Override
        protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
            super.onMeasure(widthMeasureSpec, heightMeasureSpec);

            int margin = (int) getResources().getDimension(R.dimen.instabug_path_view_margin);
            LayoutParams layoutParams = (LayoutParams) getLayoutParams();
            layoutParams.setMargins(margin, margin, margin, margin);
            setLayoutParams(layoutParams);

            int dimen = (int) getResources().getDimension(R.dimen.instabug_path_view_dimen);
            setMeasuredDimension(dimen, dimen);

            viewBounds = new RectF(0, 0, dimen, dimen);
            RectF pathBounds = new RectF(viewBounds);
            pathBounds.inset(8, 8);
            resizedPath = resizePath(path, pathBounds);
        }


        @Override
        protected void onDraw(Canvas canvas) {
            super.onDraw(canvas);

            paint.setColor(backgroundColor);
            paint.setStyle(Paint.Style.FILL);
            if (viewBounds != null) {
                canvas.drawRoundRect(viewBounds, cornerRadius, cornerRadius, paint);
            }

            paint.setColor(pathColor);
            paint.setStyle(Paint.Style.STROKE);
            if (resizedPath != null) {
                canvas.drawPath(resizedPath, paint);
            }
        }

        @Override
        public void setBackgroundColor(@ColorInt int backgroundColor) {
            this.backgroundColor = backgroundColor;
            invalidate();
        }

        public Path resizePath(Path path, RectF bounds) {
            Path resizedPath = new Path(path);
            RectF src = new RectF();
            resizedPath.computeBounds(src, true);

            Matrix resizeMatrix = new Matrix();
            resizeMatrix.setRectToRect(src, bounds, Matrix.ScaleToFit.CENTER);
            resizedPath.transform(resizeMatrix);

            return resizedPath;
        }
    }
}