package com.instabug.apm.util

import androidx.annotation.VisibleForTesting
import com.instabug.apm.fragment.model.*

typealias EventTime = Pair<Long, Long>

val EventTime.startTime
    get() = first

val EventTime.duration
    get() = second

fun List<FragmentSpansEvent>.toValidEvents(): MutableList<FragmentSpansEvent> =
    mutableListOf(
        getEvent(ON_ATTACH), getEvent(ON_CREATE),
        getEvent(ON_CREATE_VIEW), getEvent(ON_VIEW_CREATED),
        getEvent(ON_ACTIVITY_CREATED), getEvent(ON_VIEW_STATE_RESTORED),
        getEvent(ON_START), getEvent(ON_RESUME)
    ).filterNotNull().toMutableList()

@VisibleForTesting
fun List<FragmentSpansEvent>.getEvent(eventName: @FragmentSpansEvents String): FragmentSpansEvent? {
    return when (eventName) {
        ON_ATTACH -> getEventData(PRE_ATTACH, POST_ATTACH)?.toFragmentSpansEvent(eventName)
        ON_CREATE -> getEventData(PRE_CREATE, POST_CREATE)?.toFragmentSpansEvent(eventName)
        ON_CREATE_VIEW -> getEventData(PRE_CREATE_VIEW, POST_CREATE_VIEW)?.toFragmentSpansEvent(
            eventName
        )
        ON_VIEW_CREATED -> getEventData(PRE_VIEW_CREATED, POST_VIEW_CREATED)?.toFragmentSpansEvent(
            eventName
        )
        ON_ACTIVITY_CREATED -> getEventData(
            PRE_ACTIVITY_CREATED,
            POST_ACTIVITY_CREATED
        )?.toFragmentSpansEvent(eventName)
        ON_VIEW_STATE_RESTORED -> getEventData(
            PRE_VIEW_STATE_RESTORED,
            POST_VIEW_STATE_RESTORED
        )?.toFragmentSpansEvent(eventName)
        ON_START -> getEventData(PRE_START, POST_START)?.toFragmentSpansEvent(eventName)
        ON_RESUME -> getEventData(PRE_RESUME, POST_RESUME)?.toFragmentSpansEvent(eventName)
        else -> null
    }
}

@VisibleForTesting
fun List<FragmentSpansEvent>.getEventData(
    preEventName: @FragmentSpansEvents String,
    postEventName: @FragmentSpansEvents String
): EventTime? {
    find { it.name == preEventName }?.let { preEvent ->
        val startTime = preEvent.startTime
        val startTimeNano = preEvent.startTimeNano

        find { it.name == postEventName }?.let { postEvent ->
            val endTimeNano = postEvent.startTimeNano

            return startTime to (endTimeNano - startTimeNano) / 1000L
        }

        return null
    }

    return null
}

@VisibleForTesting
fun EventTime?.toFragmentSpansEvent(name: @FragmentSpansEvents String): FragmentSpansEvent? =
    if (this != null) {
        FragmentSpansEvent(name = name, startTime = startTime, duration = duration)
    } else {
        null
    }
