package com.instabug.apm.networkinterception

import androidx.annotation.VisibleForTesting
import com.instabug.apm.APMPlugin
import com.instabug.apm.attributes.listeners.OnNetworkTraceListener
import com.instabug.apm.constants.ErrorMessages
import com.instabug.apm.di.ServiceLocator
import com.instabug.apm.handler.attributes.NetworkTraceAttributesHandler
import com.instabug.apm.handler.networklog.NetworkLogHandler
import com.instabug.apm.logger.internal.Logger
import com.instabug.apm.model.APMNetworkLog
import com.instabug.apm.model.DefaultAPMNetworkLog
import com.instabug.apm.networkinterception.map.applyFromSnapshot
import com.instabug.apm.networkinterception.map.toSnapshot
import com.instabug.apm.sanitization.AsyncSanitizer
import com.instabug.apm.sanitization.OnCompleteCallback
import com.instabug.apm.sanitization.Sanitizer
import com.instabug.apm.util.connection.ConnectionUtils
import com.instabug.library.logging.listeners.networklogs.NetworkLogSnapshot
import org.json.JSONObject

open class APMNetworkLogWrapper(private val networkLog: APMNetworkLog = DefaultAPMNetworkLog()) :
    APMNetworkLog by networkLog, OnCompleteCallback<NetworkLogSnapshot> {

    private val asyncSanitizer: AsyncSanitizer<NetworkLogSnapshot>?
        get() = ServiceLocator.getNetworkInterceptionRepository().asyncSanitizer

    private val networkTraceAttributesHandler: NetworkTraceAttributesHandler?
        get() = ServiceLocator.getNetworkTraceAttributesHandler()

    private val handler: NetworkLogHandler by lazy {
        ServiceLocator.getNetworkLogHandler()
    }

    private val logger: Logger by lazy {
        ServiceLocator.getApmLogger()
    }

    private var addedAttributes = false

    init {
        ServiceLocator.postNetworkLoggingTask { populateNetworkLog(networkLog) }
    }

    private fun populateNetworkLog(networkLog: APMNetworkLog) = networkLog.run {
        updateSessionIdIfPossible()
        radio = ConnectionUtils.getRadio()
        carrier = ConnectionUtils.getCarrier()
        executedInBackground = sessionId.isNullOrBlank()
    }

    private fun updateSessionIdIfPossible() {
        if (sessionId.isNullOrBlank()) {
            val currentSession = ServiceLocator.getSessionHandler()?.currentSession
            sessionId = currentSession?.id
        }
    }

    fun insert(exception: Exception?, sanitizer: Sanitizer<APMNetworkLog>) {
        if (!isValid()) return
        ServiceLocator.postNetworkLoggingTask {
            adjustW3CAttributes()
            if (id == -1L) insertOrAsyncSanitize(sanitizer) else updateNetworkLog(exception)
        }
    }

    private fun insertOrAsyncSanitize(sanitizer: Sanitizer<APMNetworkLog>) = sanitizer.runCatching {
        sanitize(this@APMNetworkLogWrapper)?.takeIf { it.isValid() }?.let {
            asyncSanitizer?.sanitize(it.toSnapshot, this@APMNetworkLogWrapper)
                ?: insertNetworkLog()
        }
    }.onFailure {
        logger.logSDKErrorWithStackTrace(ErrorMessages.SANITIZATION_FAILED_WITH_EXCEPTION, it)
    }

    override fun onComplete(item: NetworkLogSnapshot?) = ServiceLocator.postNetworkLoggingTask {
        item?.runCatching {
            if (url == null) {
                logger.logSDKError(ErrorMessages.REMOVING_NETWORK_LOG_URL_IS_NOT_ALLOWED)
                return@runCatching
            }
            takeUnless { it.url.isNullOrBlank() }
                ?.also(::applyFromSnapshot)
                ?.let { insertNetworkLog() }
        }?.onFailure {
            logger.logSDKErrorWithStackTrace(ErrorMessages.SANITIZATION_FAILED_WITH_EXCEPTION, it)
        }
    }

    private fun insertNetworkLog() = cacheThenAddAttributes {
        updateSessionIdIfPossible()
        id = handler.insertNetworkLog(this)
    }

    private fun updateNetworkLog(exception: Exception?) = cacheThenAddAttributes {
        handler.updateNetworkLog(networkLog)
        logger.d(getLogMessage(exception, handler))
    }

    private inline fun cacheThenAddAttributes(operation: () -> Unit) =
        synchronized(APMPlugin.lock) {
            operation()
            if (!addedAttributes) addAttributes()
        }

    private fun adjustW3CAttributes() {
        adjustW3CAttributesForClientErrors()
        adjustW3CAttributesForLateHeadersInjection()
    }

    private fun adjustW3CAttributesForClientErrors() {
        if (errorMessage.isNullOrBlank()) return
        isW3CTraceIdCaptured = null
        generatedW3CPid = null
        generatedW3CTimestampSeconds = null
        fullyGeneratedW3CTraceId = null
        syncableGeneratedW3CTraceId = null
        syncableCapturedW3CTraceId = null
    }

    private fun adjustW3CAttributesForLateHeadersInjection() {
        // If fullyGeneratedW3CTraceId is null that means we didn't generate anything to begin with.
        if (fullyGeneratedW3CTraceId == null) return
        val traceIdValue = requestHeaders?.let(::JSONObject)
            ?.optString("traceparent")
            ?.takeUnless(String::isEmpty) ?: return
        // If the fully generated trace id is equal to the final headers value, then there was
        // no overriding to the header value due to late headers injection
        if (fullyGeneratedW3CTraceId == traceIdValue) return
        // Invoking the factory again with the captured trace id value to adjust model data.
        getInjectableHeader(traceIdValue)
    }

    @VisibleForTesting
    fun addAttributes() {
        networkTraceAttributesHandler?.let {
            val name = "[$method] $url"
            val trace = ServiceLocator.getNetworkLogToTraceMapper().map(networkLog)
            val listeners = it.all ?: return
            listeners.forEach { onNetworkTraceListener ->
                onNetworkTraceListener.addAttributesOnFinish(trace)
                    .takeUnless { map -> skipAddingAttribute(onNetworkTraceListener, map) }
                    ?.forEach { entry ->
                        addSingleAttribute(name, entry.key, entry.value, handler)
                    }
            }
        }
        addedAttributes = true
    }

    // Skip adding attributes if url is null OR
    // Predicates exists and not matching OR
    // No attributes map has returned from the API
    private fun skipAddingAttribute(
        onNetworkTraceListener: OnNetworkTraceListener,
        stringMap: Map<String, String>?
    ) = url == null
            || (onNetworkTraceListener.predicate?.run { !check(url!!) } ?: false)
            || stringMap == null

    private fun addSingleAttribute(
        name: String,
        key: String?,
        value: String?,
        networkLogHandler: NetworkLogHandler
    ) {
        if (!networkLogHandler.isValidAttribute(name, key, value)) return
        val trimmedKey = key!!.trim()
        val trimmedValue = value?.trim()
        networkLogHandler.addAttribute(id, name, executedInBackground, trimmedKey, trimmedValue)
    }

    private fun getLogMessage(exception: Exception?, networkLogHandler: NetworkLogHandler): String {
        return if (exception != null || !errorMessage.isNullOrEmpty()) {
            ErrorMessages.NETWORK_REQUEST_FAILED_CLIENT_SIDE.prepareLogMessage(networkLogHandler) {
                replace("\$error", exception?.toString() ?: errorMessage!!)
            }
        } else if (responseCode >= 400) {
            ErrorMessages.NETWORK_REQUEST_FAILED_SERVER_SIDE.prepareLogMessage(networkLogHandler) {
                replace("\$code", responseCode.toString())
            }
        } else {
            ErrorMessages.NETWORK_REQUEST_ENDED.prepareLogMessage(networkLogHandler) {
                replace("\$code", responseCode.toString())
            }
        }
    }

    private fun String.prepareLogMessage(
        networkLogHandler: NetworkLogHandler,
        customReplacement: String.() -> String,
    ) =
        replace("\$method", method.toString())
            .replace("\$url", url ?: "")
            .replace("\$duration", totalDuration.toString())
            .customReplacement()
            .replace(
                "\$attr",
                JSONObject(
                    ((networkLogHandler.getTraceAttributes(id) ?: mutableMapOf()) as Map<*, *>)
                ).toString()
            )

    override fun toString(): String = networkLog.toString()

    fun getInjectableHeader(capturedId: String?): List<Pair<String, String>>? =
        ServiceLocator.getExternalNetworkTraceIdFactory()
            .create(capturedId)?.run {
                isW3CTraceIdCaptured = isCaptured
                generatedW3CPid = pid
                generatedW3CTimestampSeconds = timestampSeconds
                fullyGeneratedW3CTraceId = fullyGeneratedId
                syncableGeneratedW3CTraceId = syncableGeneratedTraceId
                syncableCapturedW3CTraceId = syncableCapturedTraceId
                headers
            }
}