package com.instabug.apm.networking.handler;

import static com.instabug.apm.networking.mapping.sessions.SessionMappingKeys.DROPPED_SESSION_COUNT_STORE_LIMIT;
import static com.instabug.apm.networking.mapping.sessions.SessionMappingKeys.SESSIONS_LIST;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.apm.cache.model.SessionCacheModel;
import com.instabug.apm.configuration.APMStateProvider;
import com.instabug.apm.constants.Constants;
import com.instabug.apm.constants.ErrorMessages;
import com.instabug.apm.di.ServiceLocator;
import com.instabug.apm.logger.internal.Logger;
import com.instabug.apm.networking.APMSyncException;
import com.instabug.apm.networking.Endpoints;
import com.instabug.apm.networking.mapping.sessions.SessionMapper;
import com.instabug.apm.util.debug.DebugUtils;
import com.instabug.apm.util.device.APMDeviceStateProvider;
import com.instabug.library.IBGNetworkWorker;
import com.instabug.library.internal.device.InstabugDeviceProperties;
import com.instabug.library.networkv2.NetworkManager;
import com.instabug.library.networkv2.RequestResponse;
import com.instabug.library.networkv2.request.Header;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.networkv2.request.RequestMethod;
import com.instabug.library.networkv2.request.RequestParameter;
import com.instabug.library.networkv2.request.RequestType;
import com.instabug.library.settings.SettingsManager;

import org.json.JSONArray;

import java.util.List;
import java.util.Locale;

public class SyncManagerNetworkHandlerImpl implements SyncManagerNetworkHandler {

    private final SessionMapper sessionMapper = ServiceLocator.getSessionMapper();
    @VisibleForTesting
    @Nullable
    Request request;

    @Override
    public void syncSessions(@NonNull List<SessionCacheModel> sessions,
                             Request.Callbacks<RequestResponse, Throwable> callback) {
        try {
            JSONArray jsonArray = sessionMapper.toJsonArray(sessions);
            request = constructRequest(jsonArray);
            doRequest(request, callback);
        } catch (Exception e) {
            callback.onFailed(e);
        }
    }

    private void doRequest(@Nullable Request request, Request.Callbacks<RequestResponse, Throwable> callback) {
        if (request != null) {
            NetworkManager networkManager = ServiceLocator.getNetworkManager();
            networkManager.doRequest(IBGNetworkWorker.CORE, RequestType.NORMAL, request, callback);
        } else {
            callback.onFailed(new APMSyncException("Request object can't be null"));
        }
    }

    @VisibleForTesting
    public Request constructRequest(JSONArray sessionArray) {
        DebugUtils debugUtils = ServiceLocator.getDebugUtils();
        APMDeviceStateProvider deviceStateProvider = ServiceLocator.getDeviceStateProvider();

        Request.Builder builder = new Request.Builder()
                .url(Endpoints.SYNC_SESSIONS)
                .method(RequestMethod.POST)
                .addParameter(new RequestParameter<>(SESSIONS_LIST,
                        sessionArray))
                .shorten(true)
                .tokenProvider(() -> SettingsManager.getInstance().getAppToken())
                .hasUuid(false);

        String appToken = ServiceLocator.getAppToken();
        if (appToken != null) {
            builder.addHeader(new RequestParameter<>(Header.APP_TOKEN,
                            appToken))
                    .addParameter(new RequestParameter<>(Constants.BaseRequestKeys.APPLICATION_TOKEN_PARAM,
                            appToken));
        }
        if (deviceStateProvider.isProbablyAnEmulator()) {
            builder.addParameter(new RequestParameter<>(Constants.BaseRequestKeys.DEVICE_NAME,
                    "Emulator - " + InstabugDeviceProperties.getDeviceType()));
        } else {
            builder.addParameter(new RequestParameter<>(Constants.BaseRequestKeys.DEVICE_NAME,
                    InstabugDeviceProperties.getDeviceType()));
        }
        if (debugUtils.isInDebugMode()) {
            builder.addHeader(new RequestParameter<>(Constants.BaseRequestKeys.DEBUG_MODE_HEADER, "true"));
            builder.addParameter(new RequestParameter<>(Constants.BaseRequestKeys.DEBUG_MODE_PARAM, true));
        }
        APMStateProvider stateProvider = ServiceLocator.getApmStateProvider();
        if (stateProvider != null) {
            int droppedSessionsCount = stateProvider.getStoreLimitDroppedSessionsCount();
            if (droppedSessionsCount > 0) {
                builder.addParameter(new RequestParameter<>(DROPPED_SESSION_COUNT_STORE_LIMIT, droppedSessionsCount));
                Logger apmLogger = ServiceLocator.getApmLogger();
                if (apmLogger != null) {
                    apmLogger.logSDKWarning(String.format(
                            Locale.ENGLISH,
                            ErrorMessages.SESSIONS_DROPPED_STORE_LIMIT,
                            droppedSessionsCount)
                    );
                }
            }
        }
        return builder.build();
    }

}
