package com.instabug.apm.networking;

import androidx.annotation.Nullable;

import com.instabug.apm.configuration.APMConfigurationProvider;
import com.instabug.apm.di.InterceptorsServiceLocator;
import com.instabug.apm.di.ServiceLocator;
import com.instabug.apm.logger.internal.Logger;
import com.instabug.apm.networkinterception.APMNetworkLogWrapper;
import com.instabug.apm.networkinterception.cp.APMCPNetworkLog;
import com.instabug.apm.networkinterception.cp.ExtensionsKt;
import com.instabug.library.apichecker.APIChecker;
import com.instabug.library.util.threading.ThreadUtils;

public class APMNetworkLogger {

    @Nullable
    private APMNetworkLogWrapper networkLogWrapper;
    private final Logger apmLogger = ServiceLocator.getApmLogger();

    public APMNetworkLogger() {
        networkLogWrapper = new APMNetworkLogWrapper();
    }

    /**
     * Logs HTTP connection properties.
     * Used by cross-platform code
     *
     * @param requestStartTime   the Start time when the connection was opened
     * @param requestDuration    the time between the start time and the end of the input stream read
     * @param requestHeaders     the request headers
     * @param requestBody        the request Body
     * @param requestBodySize    the request Body size
     * @param requestMethod      the request Method (One of those values: "GET", "PUT", "POST", "DELETE", "PATCH")
     * @param requestUrl         the request Url
     * @param requestContentType the request content types
     * @param responseHeaders    the response headers
     * @param responseBody       the response Body
     * @param responseBodySize   the response Body size
     * @param statusCode         the response status code
     * @param errorMessage       the Client side error message
     * @param gqlQueryName       GraphQL query name if it was a graph QL request
     * @param serverErrorMessage Server side error message
     */
    private void log(final long requestStartTime,
                     final long requestDuration,
                     final String requestHeaders,
                     final String requestBody,
                     final long requestBodySize,
                     final String requestMethod,
                     final String requestUrl,
                     final String requestContentType,
                     final String responseHeaders,
                     final String responseBody,
                     final long responseBodySize,
                     final int statusCode,
                     final String responseContentType,
                     final String errorMessage,
                     @Nullable final String gqlQueryName,
                     @Nullable final String serverErrorMessage,
                     @Nullable final APMCPNetworkLog.W3CExternalTraceAttributes w3cAttributes) {
        final StackTraceElement[] currentThreadStackTrace = ThreadUtils.getCurrentThreadStackTrace();
        APIChecker.checkAndRunInExecutor("APMNetworkLogger.log", () -> {
            // Use a local allocation to overcome SpotBugs false-positive possible NPE warnings
            APMNetworkLogWrapper networkLogWrapper = APMNetworkLogger.this.networkLogWrapper;
            if (networkLogWrapper == null) {
                networkLogWrapper = new APMNetworkLogWrapper();
            }
            boolean isGraphQlRequest = gqlQueryName != null;
            boolean graphQlEnabled = false;
            APMConfigurationProvider apmConfigurationProvider = ServiceLocator.getApmConfigurationProvider();
            if (apmConfigurationProvider != null) {
                graphQlEnabled = apmConfigurationProvider.isNetworkGraphQlFeatureEnabled();
            }
            String nonEmptyServerSideErrorMessage = null;
            if (serverErrorMessage != null && !serverErrorMessage.isEmpty()) {
                nonEmptyServerSideErrorMessage = serverErrorMessage;
            }
            networkLogWrapper.setStartTime(requestStartTime);
            networkLogWrapper.setTotalDuration(requestDuration);
            networkLogWrapper.setRequestHeaders(requestHeaders);
            networkLogWrapper.setRequestBody(requestBody);
            networkLogWrapper.setRequestBodySize(requestBodySize);
            networkLogWrapper.setMethod(requestMethod);
            networkLogWrapper.setUrl(requestUrl);
            networkLogWrapper.setRequestContentType(requestContentType);
            networkLogWrapper.setResponseHeaders(responseHeaders);
            networkLogWrapper.setResponseBody(responseBody);
            networkLogWrapper.setResponseBodySize(responseBodySize);
            networkLogWrapper.setResponseCode(statusCode);
            networkLogWrapper.setResponseContentType(responseContentType);
            networkLogWrapper.setErrorMessage(errorMessage);
            if (graphQlEnabled && isGraphQlRequest) {
                networkLogWrapper.setGraphQlQueryName(gqlQueryName);
                networkLogWrapper.setServerSideErrorMessage(nonEmptyServerSideErrorMessage);
            } else if (!isGraphQlRequest) {
                networkLogWrapper.setServerSideErrorMessage(nonEmptyServerSideErrorMessage);
            }
            ExtensionsKt.populateExternalTraceAttributes(networkLogWrapper, w3cAttributes, apmConfigurationProvider);
            APMNetworkLogger.this.networkLogWrapper = networkLogWrapper;
            APMNetworkLogger.this.networkLogWrapper.insert(null, InterceptorsServiceLocator.getApmNetworkLoggerSanitizer());
            apmLogger.logSDKDebug("adding network log: " + APMNetworkLogger.this.networkLogWrapper.toString() + "\n");
            APMNetworkLogger.this.networkLogWrapper = null;
        });
    }
}