package com.instabug.apm.compose.compose_spans

import android.content.ContentValues
import android.database.Cursor
import com.instabug.apm.common.concurrent.OrderedExecutor
import com.instabug.apm.compose.ComposeEventListener
import com.instabug.apm.compose.compose_spans.configuration.ComposeSpansConfigurationHandler
import com.instabug.apm.compose.compose_spans.configuration.ComposeSpansConfigurationProvider
import com.instabug.apm.compose.compose_spans.configuration.ComposeSpansConfigurationProviderImpl
import com.instabug.apm.compose.compose_spans.configuration.MAX_ALLOWED_COMPOSABLE_NAME_LENGTH
import com.instabug.apm.compose.compose_spans.handler.ComposeSpansCacheHandler
import com.instabug.apm.compose.compose_spans.handler.ComposeSpansCacheHandlerImpl
import com.instabug.apm.compose.compose_spans.handler.ComposeSpansHandler
import com.instabug.apm.compose.compose_spans.handler.ComposeSpansHandlerImpl
import com.instabug.apm.compose.compose_spans.model.ComposeSpansCacheModel
import com.instabug.apm.compose.compose_spans.model.ComposeSpansModel
import com.instabug.apm.compose.compose_spans.model.SpansCacheModelCursorParser
import com.instabug.apm.compose.compose_spans.model.transform.ComposableNameLengthSanitizer
import com.instabug.apm.compose.compose_spans.model.transform.ComposeSpansListToJsonArrayMapper
import com.instabug.apm.compose.compose_spans.model.transform.ComposeSpansModelToCacheMapper
import com.instabug.apm.compose.compose_spans.model.transform.SpansCacheContentValueMapper
import com.instabug.apm.configuration.ConfigurationHandler
import com.instabug.apm.di.ServiceLocator
import com.instabug.apm.networking.mapping.sessions.SessionFeatureJsonFiller
import com.instabug.apm.networking.mapping.sessions.SessionModelFiller
import com.instabug.library.factory.Factory
import com.instabug.library.map.Mapper
import com.instabug.library.parse.Parser
import com.instabug.library.util.threading.PoolProvider
import org.json.JSONArray
import java.lang.ref.WeakReference
import java.util.concurrent.Executor


object ComposeSpansServiceLocator {

    private const val composeEventsExecutorId = "ComposeEventDispatcher"

    @Volatile
    private var _configurationProvider: ComposeSpansConfigurationProvider? = null

    @Volatile
    private var _handlerWeakReference: WeakReference<ComposeSpansHandler>? = null

    @Volatile
    private var _composeSpansManager: ComposeSpansManager? = null

    private fun createConfigurationProvider() =
        ServiceLocator.getAPMPreferencePropertyFactory()?.let { preferencePropertyFactory ->
            ServiceLocator.getReadOnceAPMPreferencePropertyFactory()
                ?.let { readOncPreferenceFactory ->
                    ServiceLocator.getApmConfigurationProvider().let { apmConfigurationProvider ->
                        ComposeSpansConfigurationProviderImpl(
                            apmConfigurationProvider,
                            ServiceLocator.getLimitConstraintApplier(),
                            preferencePropertyFactory,
                            readOncPreferenceFactory
                        ).also { _configurationProvider = it }
                    }
                }
        }

    private fun getComposeSpansCacheModelContentValuesMapper():
            Mapper<Pair<ComposeSpansCacheModel, String>, ContentValues> =
        SpansCacheContentValueMapper()

    private fun getComposeSpansCacheModelCursorParser(): Parser<Cursor?, List<ComposeSpansCacheModel>?> =
        SpansCacheModelCursorParser()

    private fun createComposeSpansHandler() =
        ServiceLocator.getSessionMetaDataCacheHandler()?.let { metaDataHandler ->
            cacheHandler?.let { cache ->
                configurationProvider?.let { config ->
                    ComposeSpansHandlerImpl(
                        cache,
                        metaDataHandler,
                        config,
                        ServiceLocator.getApmLogger()
                    ).also { _handlerWeakReference = WeakReference(it) }
                }
            }
        }

    private fun getComposeModelToCacheModelMapper():
            Mapper<ComposeSpansModel, ComposeSpansCacheModel?> = ComposeSpansModelToCacheMapper()

    private fun getComposableLengthNameSanitizer(): Mapper<String, String?> =
        ComposableNameLengthSanitizer(
            MAX_ALLOWED_COMPOSABLE_NAME_LENGTH,
            ServiceLocator.getApmLogger()
        )

    private fun createComposeSpansManager(): ComposeSpansManager? =
        composeSpansEventListenerFactory?.let { listenerFactory ->
            configurationProvider?.let { configurations ->
                handler?.let { spansHandler ->
                    ComposeSpansManagerImpl(
                        ServiceLocator.getApmConfigurationProvider(),
                        eventsDispatcherExecutor,
                        listenerFactory,
                        configurations,
                        spansHandler
                    ).also { _composeSpansManager = it }
                }
            }
        }

    val configurationProvider: ComposeSpansConfigurationProvider?
        get() = _configurationProvider ?: synchronized(ServiceLocator.getServiceLocatorLock()) {
            _configurationProvider ?: createConfigurationProvider()
        }

    val eventsDispatcherExecutor: Executor
        get() = OrderedExecutor(
            composeEventsExecutorId,
            PoolProvider.getInstance().orderedExecutor
        )

    val cacheHandler: ComposeSpansCacheHandler?
        get() =
            ServiceLocator.getDatabaseManager()?.let { databaseManager ->
                ComposeSpansCacheHandlerImpl(
                    databaseManager,
                    getComposeSpansCacheModelContentValuesMapper(),
                    getComposeSpansCacheModelCursorParser(),
                    ServiceLocator.getApmLogger()
                )
            }

    val handler: ComposeSpansHandler?
        get() = _handlerWeakReference?.get()
            ?: synchronized(ServiceLocator.getServiceLocatorLock()) {
                _handlerWeakReference?.get() ?: createComposeSpansHandler()
            }

    val repository: ComposeSpansEventsRepository?
        get() = handler?.let { composeSpansHandler ->
            configurationProvider?.let { composeConfigurationProvider ->
                ServiceLocator.getSessionHandler()?.let { sessionHandler ->
                    ComposeSpansEventsRepositoryImpl(
                        composeSpansHandler,
                        sessionHandler,
                        getComposeModelToCacheModelMapper(),
                        getComposableLengthNameSanitizer(),
                        composeConfigurationProvider
                    )
                }
            }
        }

    val composeSpansEventListenerFactory: Factory<ComposeEventListener>?
        get() = repository?.let { composeSpansRepository ->
            object : Factory<ComposeEventListener> {
                override fun create(): ComposeEventListener =
                    ComposeSpansEventListener(composeSpansRepository)
            }
        }

    val composeSpansManager: ComposeSpansManager?
        get() = _composeSpansManager ?: synchronized(ServiceLocator.getServiceLocatorLock()) {
            _composeSpansManager ?: createComposeSpansManager()
        }

    val cacheModelToJsonMapper: Mapper<List<ComposeSpansCacheModel>?, JSONArray?>
        get() = ComposeSpansListToJsonArrayMapper(ComposeSpansProviderDependencyProvider())

    val sessionFeatureJsonFiller: SessionFeatureJsonFiller
        get() = SessionComposeSpansJsonFiller(cacheModelToJsonMapper)

    val composeSpansConfigurationHandler: ConfigurationHandler?
        get() = configurationProvider?.let(::ComposeSpansConfigurationHandler)

    val composeSpansSessionModelFiller: SessionModelFiller?
        get() = handler?.let(::ComposeSpansSessionModelFiller)

}