package com.instabug.apm.compose.compose_spans

import androidx.annotation.VisibleForTesting
import com.instabug.apm.compose.compose_spans.configuration.ComposeSpansConfigurationProvider
import com.instabug.apm.compose.compose_spans.handler.ComposeSpansHandler
import com.instabug.apm.compose.compose_spans.model.ComposeSpanEvent
import com.instabug.apm.compose.compose_spans.model.ComposeSpansCacheModel
import com.instabug.apm.compose.compose_spans.model.ComposeSpansModel
import com.instabug.apm.handler.session.SessionHandler
import com.instabug.library.map.Mapper
import com.instabug.library.util.collections.LimitedLinkedHashmap

class ComposeSpansEventsRepositoryImpl(
    private val handler: ComposeSpansHandler,
    private val sessionHandler: SessionHandler,
    private val mapper: Mapper<ComposeSpansModel, ComposeSpansCacheModel?>,
    private val nameSanitizer: Mapper<String, String?>,
    private val configurationProvider: ComposeSpansConfigurationProvider
) : ComposeSpansEventsRepository {

    @VisibleForTesting
    val eventsMap: MutableMap<String, ComposeSpansModel> = LimitedLinkedHashmap(50)

    override fun addEvent(event: ComposeSpanEvent) {
        val key = event.getKey()
        when (event) {
            is ComposeSpanEvent.CompositionStarted -> {
                saveModelToDiskAndDeleteItFromMemory(key)
                createNewModel(key, event)
                addEventToMemory(key, event)
            }

            is ComposeSpanEvent.RenderingEnded -> {
                addEventToMemory(key, event)
                saveModelToDiskAndDeleteItFromMemory(key)
            }

            is ComposeSpanEvent.Dispose -> {
                saveModelToDiskAndDeleteItFromMemory(key)
            }

            else -> {
                addEventToMemory(key, event)
            }
        }
    }

    private fun createNewModel(key: String, event: ComposeSpanEvent.CompositionStarted) {
        nameSanitizer.map(event.composableName)?.let { sanitizedName ->
            eventsMap[key] = ComposeSpansModel(
                sanitizedName,
                event.showAsScreen && configurationProvider.screenLoadingSegmentationEnabled,
                event.timeCapture.getTimeStampMicro()
            )
        }
    }

    private fun removeModelFromMemory(key: String): ComposeSpansModel? = eventsMap.remove(key)

    private fun saveModelToDiskAndDeleteItFromMemory(key: String) {
        removeModelFromMemory(key)?.saveToDisk()
    }

    private fun addEventToMemory(key: String, event: ComposeSpanEvent) {
        eventsMap[key]?.set(event.eventId, event.timeCapture)
    }

    private fun ComposeSpansModel.saveToDisk() {
        mapper.map(this)?.let { cacheModel ->
            handler.save(cacheModel, sessionHandler.currentSession?.id)
        }
    }

    private fun ComposeSpanEvent.getKey(): String =
        "$id-$composableName"
}