package com.instabug.apm.appflow.usecases

import com.instabug.apm.appflow.handler.AppFlowHandler
import com.instabug.apm.appflow.log.logForceAbandonedFlow
import com.instabug.apm.appflow.model.AppFlowEndReason
import com.instabug.apm.appflow.validate.isBoundModelValid
import com.instabug.apm.appflow.validate.sanitizeBoundModel
import com.instabug.apm.di.Provider
import com.instabug.apm.logger.internal.Logger
import com.instabug.apm.model.TimeCaptureBoundModel
import com.instabug.apm.model.takeIfModelNotNull
import com.instabug.apm.sanitization.Sanitizer
import com.instabug.apm.sanitization.Validator
import com.instabug.library.core.eventbus.AppStateEvent

class StartAppFlowUseCase(
    private val handler: AppFlowHandler,
    private val logger: Logger,
    private val configurationsValidator: Validator<Unit>,
    private val flowNameValidator: Validator<String?>,
    private val flowNameSanitizer: Sanitizer<String?>,
    private val appStateProvider: Provider<AppStateEvent?>,
    private val refreshBackgroundFlowUseCase: UseCase<Long, Boolean?>
) : UseCase<TimeCaptureBoundModel<String?>, Unit> {

    override fun invoke(param: TimeCaptureBoundModel<String?>) {
        takeIf { validateConfigurations() }
            ?.let { param }
            ?.takeIf(flowNameValidator::isBoundModelValid)
            ?.let(flowNameSanitizer::sanitizeBoundModel)
            ?.let(TimeCaptureBoundModel<String?>::takeIfModelNotNull)
            ?.also { refreshBackgroundFlowUseCase(it.timeCapture.getTimeStampMillis()) }
            ?.also(::forceAbandonFlowAndLogIfActive)
            ?.also(::startNewAppFlow)
    }

    private fun validateConfigurations() =
        configurationsValidator.isValid(Unit)

    private fun forceAbandonFlowAndLogIfActive(nameModel: TimeCaptureBoundModel<String>) =
        nameModel.model.let { flowName ->
            handler.endWithReason(flowName, AppFlowEndReason.FORCE_ABANDONMENT)
                ?.takeIf { it }
                ?.let { flowName }
                ?.let(logger::logForceAbandonedFlow)
        }

    private fun startNewAppFlow(
        nameModel: TimeCaptureBoundModel<String>
    ) =
        handler.start(
            nameModel.model,
            nameModel.timeCapture.getTimeStampMicro(),
            nameModel.timeCapture.getMicroTime(),
            appStateProvider.isBackground()
        )

    private fun Provider<AppStateEvent?>.isBackground(): Boolean =
        invoke()?.let {
            it is AppStateEvent.BackgroundAppStateEvent
        } ?: true
}