package com.instabug.apm.appflow.usecases

import com.instabug.apm.appflow.handler.AppFlowHandler
import com.instabug.apm.appflow.log.logFlowNotEndedNoMatchingActiveFlow
import com.instabug.apm.appflow.validate.isBoundModelValid
import com.instabug.apm.appflow.validate.sanitizeBoundModel
import com.instabug.apm.logger.internal.Logger
import com.instabug.apm.model.TimeCaptureBoundModel
import com.instabug.apm.model.takeIfModelNotNull
import com.instabug.apm.sanitization.Sanitizer
import com.instabug.apm.sanitization.Validator

class EndAppFlowUseCase(
    private val handler: AppFlowHandler,
    private val logger: Logger,
    private val configurationsValidator: Validator<Unit>,
    private val flowNameValidator: Validator<String?>,
    private val flowNameSanitizer: Sanitizer<String?>,
    private val refreshBackgroundFlowUseCase: UseCase<Long, Boolean?>
) : UseCase<TimeCaptureBoundModel<String?>, Unit> {

    override fun invoke(param: TimeCaptureBoundModel<String?>) {
        takeIf { validateConfigurations() }
            ?.let { param }
            ?.takeIf(flowNameValidator::isBoundModelValid)
            ?.let(flowNameSanitizer::sanitizeBoundModel)
            ?.let(TimeCaptureBoundModel<String?>::takeIfModelNotNull)
            ?.takeUnless { passedAbandonmentThreshold(it, param.model) }
            ?.let(::endAppFlow)
            ?.takeUnless { ended -> ended }
            ?.let { param.model }
            ?.let(::logNoFlowFoundWithTheProvidedName)
    }

    private fun passedAbandonmentThreshold(
        timeBoundModel: TimeCaptureBoundModel<String>,
        originalName: String?
    ) = (refreshBackgroundFlowUseCase(timeBoundModel.timeCapture.getTimeStampMillis()) == true)
        .also { if (it) logNoFlowFoundWithTheProvidedName(originalName) }

    private fun logNoFlowFoundWithTheProvidedName(name: String?) =
        name?.let(logger::logFlowNotEndedNoMatchingActiveFlow)

    private fun validateConfigurations() =
        configurationsValidator.isValid(Unit)

    private fun endAppFlow(
        nameModel: TimeCaptureBoundModel<String>
    ) = handler.end(nameModel.model, nameModel.timeCapture.getMicroTime())
}