package com.instabug.apm.webview.webview_trace.handler

import com.instabug.apm.di.Provider
import com.instabug.apm.webview.webview_trace.configuration.WebViewTraceConfigurationProvider
import com.instabug.apm.webview.webview_trace.model.WebViewCacheModel
import com.instabug.apm.webview.webview_trace.model.WebViewTraceModel
import com.instabug.apm.webview.webview_trace.model.event.WebViewEvent
import com.instabug.apm.webview.webview_trace.model.event.WebViewTraceInitiationEvent
import com.instabug.library.map.Mapper

class WebViewTraceRepositoryImpl(
    private val configurations: WebViewTraceConfigurationProvider,
    private val handler: WebViewTraceHandler,
    private val modelWrapperProvider: Provider<WebViewTraceModelWrapper?>,
    private val mapper: Mapper<WebViewTraceModel?, WebViewCacheModel?>,
    private val traces: MutableMap<Long, WebViewTraceModelWrapper>
) : WebViewTraceRepository {

    companion object {
        const val EVENT_COPY_COUNTS_CUT_OFF = 5
    }

    override fun handleEvent(webViewTraceId: Long, event: WebViewEvent) {
        if (enabled() && event.copyCount < EVENT_COPY_COUNTS_CUT_OFF) {
            checkAndHandleTerminationEvent(event, webViewTraceId)
            checkAndHandleCreationEvent(event, webViewTraceId)
            traces.handleTraceEvent(webViewTraceId, event)
        }
    }

    private fun checkAndHandleTerminationEvent(
        event: WebViewEvent,
        webViewTraceId: Long
    ) {
        if (event.shouldTerminateCurrentTrace()) {
            traces.terminateTrace(webViewTraceId)
        }
    }

    private fun checkAndHandleCreationEvent(
        event: WebViewEvent,
        webViewTraceId: Long
    ) {
        if (event is WebViewTraceInitiationEvent && event.shouldInitiateNewTrace) {
            traces.createTrace(webViewTraceId)
        }
    }

    private fun MutableMap<Long, WebViewTraceModelWrapper>.terminateTrace(webViewTraceId: Long) {
        get(webViewTraceId)?.traceModel
            ?.takeIf(WebViewTraceModel::isValid)
            ?.takeIf(::webViewSizeIsSupported)
            ?.let(::mapTraceModeToCacheModelAndUITraceIdPair)
            ?.also(::save)
        remove(webViewTraceId)
    }

    private fun webViewSizeIsSupported(it: WebViewTraceModel) =
        configurations.partialViewEnabled || it.isFullScreen == true

    private fun mapTraceModeToCacheModelAndUITraceIdPair(
        webViewTraceModel: WebViewTraceModel
    ): Pair<WebViewCacheModel, Long>? =
        mapper.map(webViewTraceModel)
            ?.let { cacheModel -> Pair(cacheModel, webViewTraceModel.uiTraceId) }

    private fun save(modelPair: Pair<WebViewCacheModel, Long>) =
        handler.save(modelPair.first, modelPair.second)

    private fun MutableMap<Long, WebViewTraceModelWrapper>.createTrace(webViewTraceId: Long) {
        if (!containsKey(webViewTraceId)) {
            modelWrapperProvider.invoke()?.let { wrapper ->
                put(webViewTraceId, wrapper)
            }
        }
    }

    private fun MutableMap<Long, WebViewTraceModelWrapper>.handleTraceEvent(
        webViewTraceId: Long,
        event: WebViewEvent
    ) {
        get(webViewTraceId)
            ?.handleEvent(event)
            ?.also {
                handleEvent(webViewTraceId, it)
            }
    }

    override fun onUiTraceEnded(traceId: Long) {
        if (enabled()) {
            traces.entries
                .filter { it.value.traceModel?.uiTraceId == traceId }
                .map(Map.Entry<Long, WebViewTraceModelWrapper>::key)
                .forEach {
                    traces.terminateTrace(it)
                }
        }
    }

    override fun isWebViewSizeResolved(webViewTraceId: Long): Boolean =
        traces[webViewTraceId]?.isSizeResolved() ?: true

    override fun shouldCollectVitals(webViewTraceId: Long): Boolean =
        traces[webViewTraceId]?.shouldRegisterWebVitalsListener() ?: false

    private fun enabled(): Boolean {
        return configurations.enabled.also {
            if (!it) {
                traces.clear()
            }
        }
    }
}