package com.instabug.apm.webview.dispatch

import android.graphics.Bitmap
import android.webkit.WebResourceError
import android.webkit.WebResourceRequest
import android.webkit.WebResourceResponse
import android.webkit.WebView
import com.instabug.apm.di.ServiceLocator
import com.instabug.apm.model.EventTimeMetricCapture
import java.util.Collections
import java.util.WeakHashMap
import java.util.concurrent.Executor

object WebViewEventDispatcher {

    @JvmStatic
    private val listeners: MutableSet<WebViewEventListener> =
        Collections.synchronizedSet(
            Collections.newSetFromMap(WeakHashMap())
        )

    @JvmStatic
    private val executor: Executor by lazy { ServiceLocator.getWebViewExecutor() }

    @JvmStatic
    fun addListener(listener: WebViewEventListener) {
        listeners += listener
    }

    @JvmStatic
    fun removeListener(listener: WebViewEventListener) {
        listeners -= listener
    }

    operator fun plusAssign(listener: WebViewEventListener) = addListener(listener)

    operator fun minusAssign(listener: WebViewEventListener) = removeListener(listener)

    @JvmStatic
    fun onLoadUrl(id: Long, webView: WebView, url: String?) {
        dispatchEvent { listener, timeCapture ->
            listener.onLoadUrl(id, webView, url, timeCapture)
        }
    }

    @JvmStatic
    fun onLoadUrl(
        id: Long,
        webView: WebView,
        url: String?,
        headers: Map<String, String>?
    ) {
        dispatchEvent { listener, timeCapture ->
            listener.onLoadUrl(id, webView, url, headers, timeCapture)
        }
    }

    @JvmStatic
    fun onPostUrl(id: Long, webView: WebView, url: String?, postData: ByteArray?) {
        dispatchEvent { listener, timeCapture ->
            listener.onPostUrl(id, webView, url, postData, timeCapture)
        }
    }

    @JvmStatic
    fun onLoadData(
        id: Long,
        webView: WebView,
    ) {
        dispatchEvent { listener, timeCapture ->
            listener.onLoadData(id, webView, timeCapture)
        }
    }

    @JvmStatic
    fun onLoadDataWithBaseUrl(
        id: Long,
        webView: WebView
    ) {
        dispatchEvent { listener, timeCapture ->
            listener.onLoadDataWithBaseUrl(
                id,
                webView,
                timeCapture
            )
        }
    }

    @JvmStatic
    fun shouldOverrideUrlLoadingStarted(
        id: Long,
        view: WebView?,
        request: WebResourceRequest?
    ) {
        dispatchEvent { listener, timeCapture ->
            listener.shouldOverrideUrlLoadingStarted(id, view, request, timeCapture)
        }
    }

    @JvmStatic
    fun shouldOverrideUrlLoadingEnded(
        id: Long,
        view: WebView?,
        request: WebResourceRequest?,
        override: Boolean
    ) {
        dispatchEvent { listener, timeCapture ->
            listener.shouldOverrideUrlLoadingEnded(id, view, request, override, timeCapture)
        }
    }

    @JvmStatic
    fun onPageStarted(
        id: Long,
        view: WebView?,
        url: String?,
        favicon: Bitmap?
    ) {
        dispatchEvent { listener, timeCapture ->
            listener.onPageStarted(id, view, url, favicon, timeCapture)
        }
    }

    @JvmStatic
    fun onPageFinished(
        id: Long,
        view: WebView?,
        url: String?
    ) {
        dispatchEvent { listener, timeCapture ->
            listener.onPageFinished(id, view, url, timeCapture)
        }
    }

    @JvmStatic
    fun onReceivedError(
        id: Long,
        view: WebView?,
        request: WebResourceRequest?,
        error: WebResourceError?
    ) {
        dispatchEvent { listener, timeCapture ->
            listener.onReceivedError(id, view, request, error, timeCapture)
        }
    }

    @JvmStatic
    fun onReceivedHttpError(
        id: Long,
        view: WebView?,
        request: WebResourceRequest?,
        errorResponse: WebResourceResponse?
    ) {
        dispatchEvent { listener, timeCapture ->
            listener.onReceivedHttpError(id, view, request, errorResponse, timeCapture)
        }
    }

    @JvmStatic
    private inline fun dispatchEvent(
        crossinline event: (WebViewEventListener, EventTimeMetricCapture) -> Unit
    ) {
        val timeCapture = EventTimeMetricCapture()
        executor.execute {
            synchronized(listeners) {
                listeners.forEach { event.invoke(it, timeCapture) }
            }
        }
    }
}