package com.instabug.apm.networkinterception.utils;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.io.IOException;
import java.io.InputStream;

/**
 * Wrapper of InputStream which counts number of bytes being read
 * to calculate response size
 */
public class APMCountableInputStream extends InputStream {

    private long count = 0;
    private int mark = -1;
    private boolean callbackCompleted = false;
    private final InputStream stream;
    private final Callback callback;

    public APMCountableInputStream(InputStream stream, Callback callback) {
        this.stream = stream;
        this.callback = callback;
    }

    @Override
    public int read() throws IOException {
        int result = stream.read();
        if (result != -1) {
            count++;
        } else {
            if (!callbackCompleted) {
                callbackCompleted = true;
                callback.onReadCompleted(count);
            }
        }
        return result;
    }

    @Override
    public int read(@NonNull byte[] b, int off, int len) throws IOException {
        int result = stream.read(b, off, len);
        if (result != -1) {
            count += result;
        } else {
            if (!callbackCompleted) {
                callbackCompleted = true;
                callback.onReadCompleted(count);
            }
        }
        return result;
    }

    @Override
    public long skip(long n) throws IOException {
        long result = stream.skip(n);
        count += result;
        return result;
    }

    @Override
    public void mark(int readLimit) {
        stream.mark(readLimit);
        mark = (int) count;

    }

    @Override
    public void reset() throws IOException {

        if (!stream.markSupported()) {
            throw new IOException("Mark not supported");
        }
        if (mark == -1) {
            throw new IOException("Mark not set");
        }

        stream.reset();
        count = mark;
    }

    @Nullable
    public String getBody() {
        return null;
    }

    public interface Callback {
        void onReadCompleted(long count) throws IOException;
    }
}