package com.instabug.apm.handler.fragment

import com.instabug.apm.cache.handler.fragments.FragmentSpansCacheHandler
import com.instabug.apm.cache.handler.fragments.FragmentSpansEventsCacheHandler
import com.instabug.apm.cache.handler.session.SessionMetaDataCacheHandler
import com.instabug.apm.cache.model.FragmentSpansCacheModel
import com.instabug.apm.configuration.APMConfigurationProvider
import com.instabug.apm.fragment.model.FragmentSpans
import com.instabug.apm.logger.internal.Logger


interface FragmentSpansHandler {
    fun saveFragment(fragmentSpans: FragmentSpans): Boolean
    fun getFragmentsForSession(sessionId: String): List<FragmentSpansCacheModel>
    fun clearFragmentsForSession(sessionId: String)
    fun clearCache()
    fun onNewSession(sessionId: String)
}

class FragmentSpansHandlerImpl(
    private val fragmentSpansCacheHandler: FragmentSpansCacheHandler,
    private val fragmentSpansEventsCacheHandler: FragmentSpansEventsCacheHandler,
    private val sessionMetaDataCacheHandler: SessionMetaDataCacheHandler,
    private val apmConfigurationProvider: APMConfigurationProvider,
    private val apmLogger: Logger
) : FragmentSpansHandler {

    override fun saveFragment(fragmentSpans: FragmentSpans): Boolean =
        synchronized(this) {
            fragmentSpansCacheHandler
                .takeIf { canSaveFragment() }
                ?.saveFragment(fragmentSpans)
                ?.takeIf { id -> id != -1L }
                ?.also { fragmentId ->
                    fragmentSpansEventsCacheHandler.saveEvents(fragmentSpans.events, fragmentId)
                    fragmentSpans.sessionId
                        ?.let { sessionId -> updateCountsAndTrim(sessionId, 1) }
                } != null
        }

    override fun onNewSession(sessionId: String) {
        fragmentSpansCacheHandler.run {
            dropInvalidDanglingOccurrences()
            migrateDanglingOccurrencesIfPossible(sessionId)
                .takeIf { it > 0 }
                ?.let { updatedCount -> updateCountsAndTrim(sessionId, updatedCount) }
        }
    }

    private fun updateCountsAndTrim(sessionId: String, addedCount: Int) {
        sessionMetaDataCacheHandler.addToFragmentSpansTotalCount(sessionId, addedCount)
        fragmentSpansCacheHandler
            .trimToLimit(sessionId, apmConfigurationProvider.fragmentSpansLimitPerRequest)
            ?.takeIf { it > 0 }
            ?.let { deletedRowsCount ->
                sessionMetaDataCacheHandler.addToFragmentSpansDroppedCount(
                    sessionId,
                    deletedRowsCount
                )
                apmLogger.d("Fragment spans dropped count: $deletedRowsCount")
            }
        fragmentSpansCacheHandler.trimToLimit(apmConfigurationProvider.fragmentSpansStoreLimit)
    }

    override fun getFragmentsForSession(sessionId: String): List<FragmentSpansCacheModel> {
        synchronized(this) {
            return fragmentSpansCacheHandler.getFragmentsForSession(sessionId)
                .takeIf {
                    it.isNotEmpty()
                }?.onEach { fragment ->
                    fragment.events.addAll(
                        fragmentSpansEventsCacheHandler.getEventsForFragment(
                            fragment.id
                        )
                    )
                } ?: listOf()
        }
    }

    override fun clearFragmentsForSession(sessionId: String) {
        synchronized(this) {
            fragmentSpansCacheHandler.deleteFragmentsForSession(sessionId)
        }
    }

    override fun clearCache() {
        synchronized(this) {
            fragmentSpansCacheHandler.clearAll()
            sessionMetaDataCacheHandler.resetFragmentSpansCounts()
        }
    }

    private fun canSaveFragment() =
        apmConfigurationProvider.isFragmentSpansEnabled && apmConfigurationProvider.isAPMEnabled
}