package com.instabug.apm.fragment

import androidx.annotation.VisibleForTesting
import androidx.fragment.app.Fragment
import com.instabug.apm.di.Provider
import com.instabug.apm.fragment.model.FragmentSpans
import com.instabug.apm.fragment.model.FragmentSpansEvent
import com.instabug.apm.fragment.model.FragmentSpansEvents
import com.instabug.apm.fragment.model.POST_ACTIVITY_CREATED
import com.instabug.apm.fragment.model.POST_ATTACH
import com.instabug.apm.fragment.model.POST_CREATE
import com.instabug.apm.fragment.model.POST_CREATE_VIEW
import com.instabug.apm.fragment.model.POST_RESUME
import com.instabug.apm.fragment.model.POST_START
import com.instabug.apm.fragment.model.POST_VIEW_CREATED
import com.instabug.apm.fragment.model.POST_VIEW_STATE_RESTORED
import com.instabug.apm.fragment.model.PRE_ACTIVITY_CREATED
import com.instabug.apm.fragment.model.PRE_ATTACH
import com.instabug.apm.fragment.model.PRE_CREATE
import com.instabug.apm.fragment.model.PRE_CREATE_VIEW
import com.instabug.apm.fragment.model.PRE_RESUME
import com.instabug.apm.fragment.model.PRE_START
import com.instabug.apm.fragment.model.PRE_VIEW_CREATED
import com.instabug.apm.fragment.model.PRE_VIEW_STATE_RESTORED
import com.instabug.apm.handler.fragment.FragmentSpansHandler
import com.instabug.apm.handler.session.SessionHandler
import com.instabug.apm.model.EventTimeMetricCapture
import com.instabug.apm.util.toValidEvents
import com.instabug.library.diagnostics.IBGDiagnostics

class FragmentLifecycleEventListenerImpl(
    private val fragmentSpansHandlerProvider: Provider<FragmentSpansHandler?>,
    private val sessionHandler: SessionHandler
) : FragmentLifecycleEventListener {

    @VisibleForTesting
    val fragments: HashMap<Int, FragmentSpans> = hashMapOf()

    override fun onFragmentPreAttach(fragment: Fragment, timeCapture: EventTimeMetricCapture) {
        synchronized(this) {
            addEventToFragment(fragment, PRE_ATTACH, timeCapture)
        }
    }

    override fun onFragmentPostAttach(fragment: Fragment, timeCapture: EventTimeMetricCapture) {
        synchronized(this) {
            addEventToFragment(fragment, POST_ATTACH, timeCapture)
        }
    }

    override fun onFragmentPreCreate(fragment: Fragment, timeCapture: EventTimeMetricCapture) {
        synchronized(this) {
            addEventToFragment(fragment, PRE_CREATE, timeCapture)
        }
    }

    override fun onFragmentPostCreate(fragment: Fragment, timeCapture: EventTimeMetricCapture) {
        synchronized(this) {
            addEventToFragment(fragment, POST_CREATE, timeCapture)
        }
    }

    override fun onFragmentPreCreateView(fragment: Fragment, timeCapture: EventTimeMetricCapture) {
        synchronized(this) {
            addEventToFragment(fragment, PRE_CREATE_VIEW, timeCapture)
        }
    }

    override fun onFragmentPostCreateView(fragment: Fragment, timeCapture: EventTimeMetricCapture) {
        synchronized(this) {
            addEventToFragment(fragment, POST_CREATE_VIEW, timeCapture)
        }
    }

    override fun onFragmentPreViewCreated(fragment: Fragment, timeCapture: EventTimeMetricCapture) {
        synchronized(this) {
            addEventToFragment(fragment, PRE_VIEW_CREATED, timeCapture)
        }
    }

    override fun onFragmentPostViewCreated(
        fragment: Fragment,
        timeCapture: EventTimeMetricCapture
    ) {
        synchronized(this) {
            addEventToFragment(fragment, POST_VIEW_CREATED, timeCapture)
        }
    }

    override fun onFragmentPreActivityCreated(
        fragment: Fragment,
        timeCapture: EventTimeMetricCapture
    ) {
        synchronized(this) {
            addEventToFragment(fragment, PRE_ACTIVITY_CREATED, timeCapture)
        }
    }

    override fun onFragmentPostActivityCreated(
        fragment: Fragment,
        timeCapture: EventTimeMetricCapture
    ) {
        synchronized(this) {
            addEventToFragment(fragment, POST_ACTIVITY_CREATED, timeCapture)
        }
    }

    override fun onFragmentPreViewStateRestore(
        fragment: Fragment,
        timeCapture: EventTimeMetricCapture
    ) {
        synchronized(this) {
            addEventToFragment(fragment, PRE_VIEW_STATE_RESTORED, timeCapture)
        }
    }

    override fun onFragmentPostViewStateRestore(
        fragment: Fragment,
        timeCapture: EventTimeMetricCapture
    ) {
        synchronized(this) {
            addEventToFragment(fragment, POST_VIEW_STATE_RESTORED, timeCapture)
        }
    }

    override fun onFragmentPreStart(fragment: Fragment, timeCapture: EventTimeMetricCapture) {
        synchronized(this) {
            addEventToFragment(fragment, PRE_START, timeCapture)
        }
    }

    override fun onFragmentPostStart(fragment: Fragment, timeCapture: EventTimeMetricCapture) {
        synchronized(this) {
            addEventToFragment(fragment, POST_START, timeCapture)
        }
    }

    override fun onFragmentPreResume(fragment: Fragment, timeCapture: EventTimeMetricCapture) {
        synchronized(this) {
            addEventToFragment(fragment, PRE_RESUME, timeCapture)
        }
    }

    override fun onFragmentPostResume(fragment: Fragment, timeCapture: EventTimeMetricCapture) {
        synchronized(this) {
            addEventToFragment(fragment, POST_RESUME, timeCapture)
            takeIf { saveFragment(fragment.hashCode()) }
                ?.also { fragments.remove(fragment.hashCode()) }
        }
    }

    override fun onFragmentPreDeAttach(fragment: Fragment, timeCapture: EventTimeMetricCapture) {
        synchronized(this) {
            takeIf { saveFragment(fragment.hashCode()) }
                ?.also { fragments.remove(fragment.hashCode()) }
        }
    }

    override fun onFragmentPostDeAttach(fragment: Fragment, timeCapture: EventTimeMetricCapture) {
        synchronized(this) {
            takeIf { saveFragment(fragment.hashCode()) }
                ?.also { fragments.remove(fragment.hashCode()) }
        }
    }

    override fun cleanup() {
        synchronized(this) {
            fragments.clear()
        }
    }


    @VisibleForTesting
    fun addEventToFragment(
        fragment: Fragment,
        eventName: @FragmentSpansEvents String,
        timeCapture: EventTimeMetricCapture
    ) {
        val event = FragmentSpansEvent(
            name = eventName,
            startTime = timeCapture.getTimeStampMicro(),
            startTimeNano = timeCapture.getNanoTime()
        )

        addFragmentToMap(fragment)
        fragments[fragment.hashCode()]?.events?.add(event)
    }

    @VisibleForTesting
    fun addFragmentToMap(fragment: Fragment) {
        if (fragment is InstabugSpannableFragment) {
            takeUnless { fragments.containsKey(fragment.hashCode()) }
                ?.let {
                    fragments[fragment.hashCode()] =
                        FragmentSpans(fragment.getInstabugName())
                }
        } else {
            IBGDiagnostics.reportNonFatal(
                Throwable("Couldn't get fragment name, fragment is not instance of InstabugSpannableFragment"),
                "Error while capturing fragment events"
            )
        }
    }

    @VisibleForTesting
    fun saveFragment(hashCode: Int): Boolean = fragments[hashCode]?.let { fragmentSpans ->
        fragmentSpans.events.toValidEvents()
            .takeIf { validEvents -> validEvents.isNotEmpty() }
            ?.let { validEvents ->
                fragmentSpansHandlerProvider()?.saveFragment(
                    FragmentSpans(
                        name = fragmentSpans.name,
                        sessionId = sessionHandler.currentSession?.id,
                        events = validEvents
                    )
                )
            }
    } == true
}