package com.instabug.apm.cache.handler.networklog;

import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMNetworkLogEntry.COLUMN_CAPTURED_W3C_EXTERNAL_NETWORK_TRACE_ID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMNetworkLogEntry.COLUMN_CARRIER;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMNetworkLogEntry.COLUMN_CLIENT_SIDE_ERROR_CODE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMNetworkLogEntry.COLUMN_DURATION;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMNetworkLogEntry.COLUMN_ERROR_MESSAGE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMNetworkLogEntry.COLUMN_EXECUTED_ON_BACKGROUND;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMNetworkLogEntry.COLUMN_EXTERNAL_TRACE_ID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMNetworkLogEntry.COLUMN_EXTERNAL_TRACE_START_TIME_MILLIS;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMNetworkLogEntry.COLUMN_GENERATED_W3C_EXTERNAL_NETWORK_TRACE_ID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMNetworkLogEntry.COLUMN_GRAPH_QL_QUERY_NAME;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMNetworkLogEntry.COLUMN_GRPC_METHOD_NAME;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMNetworkLogEntry.COLUMN_ID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMNetworkLogEntry.COLUMN_IS_W3C_EXTERNAL_NETWORK_TRACE_ID_CAPTURED;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMNetworkLogEntry.COLUMN_METHOD;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMNetworkLogEntry.COLUMN_NETWORK_LATENCY_SPANS;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMNetworkLogEntry.COLUMN_RADIO;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMNetworkLogEntry.COLUMN_REQUEST_BODY_SIZE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMNetworkLogEntry.COLUMN_REQUEST_CONTENT_TYPE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMNetworkLogEntry.COLUMN_RESPONSE_BODY_SIZE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMNetworkLogEntry.COLUMN_RESPONSE_CODE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMNetworkLogEntry.COLUMN_RESPONSE_CONTENT_TYPE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMNetworkLogEntry.COLUMN_SERVER_ERROR_MESSAGE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMNetworkLogEntry.COLUMN_SESSION_ID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMNetworkLogEntry.COLUMN_START_TIME;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMNetworkLogEntry.COLUMN_URL;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMNetworkLogEntry.COLUMN_USER_MODIFIED;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMNetworkLogEntry.COLUMN_W3C_EXTERNAL_NETWORK_TRACE_ID_PID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMNetworkLogEntry.COLUMN_W3C_EXTERNAL_NETWORK_TRACE_ID_TIMESTAMP;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMNetworkLogEntry.TABLE_NAME;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.COMMA_SEP;

import android.annotation.SuppressLint;
import android.content.ContentValues;
import android.database.Cursor;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.annotation.WorkerThread;
import androidx.collection.ArrayMap;

import com.instabug.apm.configuration.APMConfigurationProvider;
import com.instabug.apm.constants.Constants;
import com.instabug.apm.constants.ErrorMessages;
import com.instabug.apm.di.Provider;
import com.instabug.apm.di.ServiceLocator;
import com.instabug.apm.logger.internal.Logger;
import com.instabug.apm.model.APMNetworkLog;
import com.instabug.apm.model.DefaultAPMNetworkLog;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.storage.cache.db.DatabaseManager;
import com.instabug.library.internal.storage.cache.db.InstabugDbContract;
import com.instabug.library.internal.storage.cache.db.SQLiteDatabaseWrapper;
import com.instabug.library.util.InstabugSDKLogger;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;


@WorkerThread
public class NetworkLogCacheHandlerImpl implements NetworkLogCacheHandler {

    @NonNull
    private final Provider<DatabaseManager> databaseManagerProvider;

    public NetworkLogCacheHandlerImpl(@NonNull Provider<DatabaseManager> databaseManagerProvider) {
        this.databaseManagerProvider = databaseManagerProvider;
    }

    @Nullable
    private Logger apmLogger = ServiceLocator.getApmLogger();

    @Nullable
    private DatabaseManager getDatabaseManager() {
        DatabaseManager databaseManager = databaseManagerProvider.invoke();
        if (databaseManager == null) {
            InstabugSDKLogger.i(Constants.LOG_TAG, "error while getting the database manager database is null");
        }
        return databaseManager;
    }

    @Override
    public long insertNetworkLog(String sessionID, APMNetworkLog networkLog) {
        DatabaseManager databaseManager = getDatabaseManager();
        if (databaseManager != null) {
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            ContentValues values = getContentValuesFromNetworkLog(networkLog, true);
            values.put(COLUMN_SESSION_ID, sessionID);
            long rowId = sqLiteDatabaseWrapper.insert(TABLE_NAME, null, values);
            sqLiteDatabaseWrapper.close();
            return rowId;
        }
        return -1;
    }


    @Override
    public void updateNetworkLog(APMNetworkLog networkLog) {
        DatabaseManager databaseManager = getDatabaseManager();
        if (databaseManager != null) {
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            ContentValues values = getContentValuesFromNetworkLog(networkLog, false);
            String whereClause = COLUMN_ID + " = ?";
            String[] whereArgs = {String.valueOf(networkLog.getId())};
            sqLiteDatabaseWrapper.update(TABLE_NAME, values, whereClause, whereArgs);
            sqLiteDatabaseWrapper.close();
        }
    }

    private ContentValues getContentValuesFromNetworkLog(APMNetworkLog networkLog, boolean includeUrl) {
        ContentValues values = new ContentValues();
        if (networkLog.getStartTime() != null) {
            values.put(COLUMN_START_TIME, networkLog.getStartTime());
        }
        if (networkLog.getUrl() != null && includeUrl) {
            values.put(COLUMN_URL, networkLog.getUrl());
        }
        if (networkLog.getMethod() != null) {
            values.put(COLUMN_METHOD, networkLog.getMethod());
        }
        if (networkLog.getRequestContentType() != null) {
            values.put(COLUMN_REQUEST_CONTENT_TYPE, networkLog.getRequestContentType());
        }
        if (networkLog.getResponseContentType() != null) {
            values.put(COLUMN_RESPONSE_CONTENT_TYPE, networkLog.getResponseContentType());
        }
        if (networkLog.getErrorMessage() != null) {
            values.put(COLUMN_ERROR_MESSAGE, networkLog.getErrorMessage());
        }
        if (networkLog.getRadio() != null) {
            values.put(COLUMN_RADIO, networkLog.getRadio());
        }
        if (networkLog.getCarrier() != null) {
            values.put(COLUMN_CARRIER, networkLog.getCarrier());
        }
        String graphQlQueryName = networkLog.getGraphQlQueryName();
        if (graphQlQueryName != null) {
            values.put(COLUMN_GRAPH_QL_QUERY_NAME, graphQlQueryName);
        }
        String grpcMethodName = networkLog.getGrpcMethodName();
        if (grpcMethodName != null) {
            values.put(COLUMN_GRPC_METHOD_NAME, grpcMethodName);
        }
        String serverSideErrorMessage = networkLog.getServerSideErrorMessage();
        if (serverSideErrorMessage != null) {
            values.put(COLUMN_SERVER_ERROR_MESSAGE, serverSideErrorMessage);
        }
        values.put(COLUMN_DURATION, networkLog.getTotalDuration());
        values.put(COLUMN_RESPONSE_CODE, networkLog.getResponseCode());
        values.put(COLUMN_CLIENT_SIDE_ERROR_CODE, networkLog.getClientErrorCode());
        values.put(COLUMN_REQUEST_BODY_SIZE, networkLog.getRequestBodySize());
        values.put(COLUMN_RESPONSE_BODY_SIZE, networkLog.getResponseBodySize());
        if (networkLog.getErrorMessage() != null) {
            values.put(COLUMN_ERROR_MESSAGE, networkLog.getErrorMessage());
        }
        if (networkLog.getRadio() != null) {
            values.put(COLUMN_RADIO, networkLog.getRadio());
        }
        if (networkLog.getCarrier() != null) {
            values.put(COLUMN_CARRIER, networkLog.getCarrier());
        }
        values.put(COLUMN_EXECUTED_ON_BACKGROUND, networkLog.getExecutedInBackground() ? 1 : 0);
        values.put(COLUMN_USER_MODIFIED, networkLog.isModified());
        if (networkLog.getLatencySpansJsonString() != null)
            values.put(COLUMN_NETWORK_LATENCY_SPANS, networkLog.getLatencySpansJsonString());
        putW3CExternalTraceAttributes(values, networkLog);
        return values;
    }

    private void putW3CExternalTraceAttributes(ContentValues values, APMNetworkLog networkLog) {
        if (networkLog.isW3CTraceIdCaptured() == null) {
            values.putNull(COLUMN_IS_W3C_EXTERNAL_NETWORK_TRACE_ID_CAPTURED);
        } else {
            values.put(COLUMN_IS_W3C_EXTERNAL_NETWORK_TRACE_ID_CAPTURED, networkLog.isW3CTraceIdCaptured());
        }

        if (networkLog.getGeneratedW3CPid() == null) {
            values.putNull(COLUMN_W3C_EXTERNAL_NETWORK_TRACE_ID_PID);
        } else {
            values.put(COLUMN_W3C_EXTERNAL_NETWORK_TRACE_ID_PID, networkLog.getGeneratedW3CPid());
        }

        if (networkLog.getGeneratedW3CTimestampSeconds() == null) {
            values.putNull(COLUMN_W3C_EXTERNAL_NETWORK_TRACE_ID_TIMESTAMP);
        } else {
            values.put(COLUMN_W3C_EXTERNAL_NETWORK_TRACE_ID_TIMESTAMP, networkLog.getGeneratedW3CTimestampSeconds());
        }

        if (networkLog.getSyncableGeneratedW3CTraceId() == null) {
            values.putNull(COLUMN_GENERATED_W3C_EXTERNAL_NETWORK_TRACE_ID);
        } else {
            values.put(COLUMN_GENERATED_W3C_EXTERNAL_NETWORK_TRACE_ID, networkLog.getSyncableGeneratedW3CTraceId());
        }

        if (networkLog.getSyncableCapturedW3CTraceId() == null) {
            values.putNull(COLUMN_CAPTURED_W3C_EXTERNAL_NETWORK_TRACE_ID);
        } else {
            values.put(COLUMN_CAPTURED_W3C_EXTERNAL_NETWORK_TRACE_ID, networkLog.getSyncableCapturedW3CTraceId());
        }
    }

    @Override
    public void cleanUp() {
        DatabaseManager databaseManager = getDatabaseManager();
        if (databaseManager != null) {
            String query = "delete from " + TABLE_NAME
                    + " where " + COLUMN_RESPONSE_CODE + " = 0"
                    + " and " + COLUMN_GRPC_METHOD_NAME + " is NULL"
                    + " and " + COLUMN_ERROR_MESSAGE + " is NULL";
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            sqLiteDatabaseWrapper.execSQL(query);
            sqLiteDatabaseWrapper.close();
        }
    }

    @Override
    public void trimToLimit(long limit) {
        DatabaseManager databaseManager = getDatabaseManager();
        if (databaseManager != null) {
            String query = "delete from " + TABLE_NAME
                    + " where " + COLUMN_ID + " not in ("
                    + " select " + COLUMN_ID
                    + " from " + TABLE_NAME
                    + " order by " + COLUMN_ID + " desc"
                    + " limit " + limit + " )";
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            sqLiteDatabaseWrapper.execSQL(query);
            sqLiteDatabaseWrapper.close();
        }
    }

    @Override
    public int trimToLimit(String sessionID, long limit) {
        DatabaseManager databaseManager = getDatabaseManager();
        if (databaseManager != null) {
            String selectByLimitDescendingQuery = "SELECT " + COLUMN_ID
                    + " FROM " + TABLE_NAME
                    + " where " + COLUMN_SESSION_ID + " = ?"
                    + " ORDER BY " + COLUMN_ID + " DESC"
                    + " LIMIT ?";

            String whereClause = COLUMN_SESSION_ID + " = ? AND "
                    + COLUMN_ID + " NOT IN (" + selectByLimitDescendingQuery + ")";
            String[] whereArgs = {sessionID, sessionID, String.valueOf(limit)};
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            int deletedTracesCount = sqLiteDatabaseWrapper.delete(TABLE_NAME, whereClause, whereArgs);
            sqLiteDatabaseWrapper.close();
            return deletedTracesCount;
        }
        return -1;
    }

    @Nullable
    @Override
    public List<APMNetworkLog> getNetworkLogsByLimit(long limit) {
        DatabaseManager databaseManager = getDatabaseManager();
        if (databaseManager != null) {
            List<APMNetworkLog> apmNetworkLogs = new ArrayList<>();
            String query = "select * from " + TABLE_NAME
                    + " limit " + limit;
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            Cursor cursor = null;
            try {
                cursor = sqLiteDatabaseWrapper.rawQuery(query, null);
                if (cursor != null) {
                    while (cursor.moveToNext()) {
                        APMNetworkLog apmNetworkLog = readApmNetworkLogFromCursor(cursor);
                        apmNetworkLogs.add(apmNetworkLog);
                    }
                }
                sqLiteDatabaseWrapper.close();
                return apmNetworkLogs;
            } catch (Exception ex) {
                apmLogger.logSDKError("DB execution a sql failed: " + ex.getMessage(), ex);
                IBGDiagnostics.reportNonFatal(ex, "DB execution a sql failed: " + ex.getMessage());
            } finally {
                if (cursor != null) {
                    cursor.close();
                }
            }
        }
        return null;
    }

    @Override
    public void deleteNetworkLogsByLimit(long limit) {
        DatabaseManager databaseManager = getDatabaseManager();
        if (databaseManager != null) {
            String deleteQuery = "delete from " + InstabugDbContract.APMNetworkLogEntry.TABLE_NAME
                    + " where " + InstabugDbContract.APMNetworkLogEntry.COLUMN_ID
                    + " in (select " + InstabugDbContract.APMNetworkLogEntry.COLUMN_ID
                    + " from " + InstabugDbContract.APMNetworkLogEntry.TABLE_NAME
                    + " limit " + limit + ")";
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            sqLiteDatabaseWrapper.execSQL(deleteQuery);
            sqLiteDatabaseWrapper.close();
        }
    }

    @Override
    public void removeAll() {
        DatabaseManager databaseManager = getDatabaseManager();
        if (databaseManager != null) {
            String deleteQuery = "delete from " + InstabugDbContract.APMNetworkLogEntry.TABLE_NAME;
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            sqLiteDatabaseWrapper.execSQL(deleteQuery);
            sqLiteDatabaseWrapper.close();
        }
    }

    @Override
    public void removeGraphQlData() {
        DatabaseManager databaseManager = getDatabaseManager();
        if (databaseManager != null) {
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            String deleteQuery = "UPDATE " + TABLE_NAME + " SET "
                    + COLUMN_GRAPH_QL_QUERY_NAME + " = NULL" + COMMA_SEP
                    + COLUMN_SERVER_ERROR_MESSAGE + " = NULL"
                    + " WHERE " + COLUMN_GRAPH_QL_QUERY_NAME + " IS NOT NULL";
            if (sqLiteDatabaseWrapper != null) {
                sqLiteDatabaseWrapper.execSQL(deleteQuery);
                sqLiteDatabaseWrapper.close();
            }
        }
    }

    @Override
    public void removeGrpcData() {
        DatabaseManager databaseManager = getDatabaseManager();
        if (databaseManager != null) {
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            if (sqLiteDatabaseWrapper != null) {
                String whereClause = COLUMN_GRPC_METHOD_NAME + " IS NOT NULL";
                sqLiteDatabaseWrapper.delete(TABLE_NAME, whereClause, null);
                sqLiteDatabaseWrapper.close();
            }
        }
    }

    @Nullable
    @Override
    public List<APMNetworkLog> getEndedNetworkLogsForSession(String sessionID) {
        DatabaseManager databaseManager = getDatabaseManager();
        if (databaseManager != null) {
            List<APMNetworkLog> cacheModels = new ArrayList<>();
            SQLiteDatabaseWrapper databaseWrapper = databaseManager.openDatabase();
            String selectionClause = COLUMN_SESSION_ID + " = ? AND "
                    + COLUMN_DURATION + " > ?";
            String[] selectionArgs = {sessionID, "0"};
            Cursor cursor = null;
            try {
                cursor = databaseWrapper.query(TABLE_NAME,
                        null, selectionClause, selectionArgs,
                        null, null, null);
                if (cursor != null) {
                    while (cursor.moveToNext()) {
                        cacheModels.add(readApmNetworkLogFromCursor(cursor));
                    }
                }
                databaseWrapper.close();
                return cacheModels;
            } catch (Exception ex) {
                apmLogger.logSDKError("DB execution a sql failed: " + ex.getMessage(), ex);
                IBGDiagnostics.reportNonFatal(ex, "DB execution a sql failed: " + ex.getMessage());
            } finally {
                if (cursor != null) {
                    cursor.close();
                }
            }
        }
        return null;
    }

    @SuppressLint("RCN_REDUNDANT_NULLCHECK_OF_NONNULL_VALUE")
    @Override
    public void addAttribute(long traceId, @NonNull String traceName, @NonNull String key, @Nullable String value) {
        if (value == null) {
            removeAttribute(traceId, key);
            return;
        }
        Map<String, String> traceAttributes = getTraceAttributes(traceId);
        if (traceAttributes != null && traceAttributes.get(key) != null)
            updateAttribute(traceId, key, value);
        else {
            APMConfigurationProvider apmConfigurationProvider = ServiceLocator.getApmConfigurationProvider();
            int maxAttributesCount = apmConfigurationProvider.getNetworkLogsStoreAttributesLimit();
            if (traceAttributes != null && traceAttributes.size() == maxAttributesCount) {
                String logMessage = ErrorMessages.MAX_ATTRIBUTES_COUNT_REACHED
                        .replace("$s1", key)
                        .replace("$s2", traceName)
                        .replace("$s3", maxAttributesCount + "");
                apmLogger.logSDKError(logMessage);
            } else {
                addAttribute(traceId, key, value);
            }
        }
    }

    @Override
    public void addAttributes(long traceId, String traceName, @Nullable Map<String, String> attributes) {
        if (attributes == null) {
            return;
        }
        for (Map.Entry<String, String> entry : attributes.entrySet()) {
            addAttribute(traceId, traceName, entry.getKey(), entry.getValue());
        }
    }

    private void addAttribute(long traceId, @NonNull String key, @NonNull String value) {
        DatabaseManager databaseManager = getDatabaseManager();
        if (databaseManager != null) {
            ContentValues contentValues = new ContentValues();
            contentValues.put(InstabugDbContract.NetworkTracesAttributesEntry.COLUMN_TRACE_ID, traceId);
            contentValues.put(InstabugDbContract.NetworkTracesAttributesEntry.COLUMN_KEY, key);
            if (value != null) {
                contentValues.put(InstabugDbContract.NetworkTracesAttributesEntry.COLUMN_VALUE, value);
            }
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            sqLiteDatabaseWrapper.insert(InstabugDbContract.NetworkTracesAttributesEntry.TABLE_NAME, null, contentValues);
        }
    }

    private void updateAttribute(long traceId, String key, String value) {
        DatabaseManager databaseManager = getDatabaseManager();
        if (databaseManager != null) {
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            ContentValues contentValues = new ContentValues();
            contentValues.put(InstabugDbContract.NetworkTracesAttributesEntry.COLUMN_VALUE, value);
            String whereClause = InstabugDbContract.NetworkTracesAttributesEntry.COLUMN_TRACE_ID + " = ? AND "
                    + InstabugDbContract.NetworkTracesAttributesEntry.COLUMN_KEY + "= ?";
            String[] whereValues = {traceId + "", key};
            sqLiteDatabaseWrapper.update(InstabugDbContract.NetworkTracesAttributesEntry.TABLE_NAME, contentValues, whereClause, whereValues);
        }
    }

    @SuppressLint("Range")
    @Override
    public Map<String, String> getTraceAttributes(long traceId) {
        DatabaseManager databaseManager = getDatabaseManager();
        if (databaseManager != null) {
            Map<String, String> attributes = new ArrayMap<>();
            String whereClause = InstabugDbContract.NetworkTracesAttributesEntry.COLUMN_TRACE_ID + " = ?";
            String[] whereValues = {traceId + ""};
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            Cursor cursor = null;
            try {
                cursor = sqLiteDatabaseWrapper.query(InstabugDbContract.NetworkTracesAttributesEntry.TABLE_NAME, null
                        , whereClause, whereValues, null, null, null);
                if (cursor != null) {
                    while (cursor.moveToNext()) {
                        attributes.put(cursor.getString(cursor.getColumnIndex(InstabugDbContract.NetworkTracesAttributesEntry.COLUMN_KEY)),
                                cursor.getString(cursor.getColumnIndex(InstabugDbContract.NetworkTracesAttributesEntry.COLUMN_VALUE)));
                    }
                    cursor.close();
                }
            } catch (RuntimeException ex) {
                apmLogger.logSDKError("Failed to get attributes", ex);
                IBGDiagnostics.reportNonFatal(ex, "DB execution a sql failed: " + ex.getMessage());
            } finally {
                if (cursor != null) {
                    cursor.close();
                }
            }
            return attributes;
        }
        return new ArrayMap<>();
    }

    private void removeAttribute(long traceId, String key) {
        DatabaseManager databaseManager = getDatabaseManager();
        if (databaseManager != null) {
            String query = "delete from " + InstabugDbContract.NetworkTracesAttributesEntry.TABLE_NAME + " where "
                    + InstabugDbContract.NetworkTracesAttributesEntry.COLUMN_TRACE_ID + " = " + traceId
                    + " and " + InstabugDbContract.NetworkTracesAttributesEntry.COLUMN_KEY + " = \"" + key + "\"";
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            sqLiteDatabaseWrapper.execSQL(query);
        }
    }

    @Override
    public void clearW3CExternalTraceIdCache() {
        DatabaseManager databaseManager = getDatabaseManager();
        if (databaseManager == null) return;
        SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
        ContentValues values = new ContentValues();
        values.putNull(COLUMN_IS_W3C_EXTERNAL_NETWORK_TRACE_ID_CAPTURED);
        values.putNull(COLUMN_W3C_EXTERNAL_NETWORK_TRACE_ID_PID);
        values.putNull(COLUMN_W3C_EXTERNAL_NETWORK_TRACE_ID_TIMESTAMP);
        values.putNull(COLUMN_GENERATED_W3C_EXTERNAL_NETWORK_TRACE_ID);
        values.putNull(COLUMN_CAPTURED_W3C_EXTERNAL_NETWORK_TRACE_ID);
        try {
            sqLiteDatabaseWrapper.update(TABLE_NAME, values, null, null);
        } catch (Throwable t) {
            IBGDiagnostics.reportNonFatal(t, "Error occurred while clearing w3c external trace data From APMNetworkLog table: " + t.getMessage());
        }
    }

    @Override
    public void clearGeneratedW3CExternalTraceIdCache() {
        DatabaseManager databaseManager = getDatabaseManager();
        if (databaseManager == null) return;
        SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
        ContentValues values = new ContentValues();
        values.putNull(COLUMN_GENERATED_W3C_EXTERNAL_NETWORK_TRACE_ID);
        try {
            sqLiteDatabaseWrapper.update(TABLE_NAME, values, null, null);
        } catch (Throwable t) {
            IBGDiagnostics.reportNonFatal(t, "Error occurred while clearing generated w3c external trace id From APMNetworkLog table: " + t.getMessage());
        }
    }

    @Override
    public void clearCapturedW3CExternalTraceIdCache() {
        DatabaseManager databaseManager = getDatabaseManager();
        if (databaseManager == null) return;
        SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
        ContentValues values = new ContentValues();
        values.putNull(COLUMN_CAPTURED_W3C_EXTERNAL_NETWORK_TRACE_ID);
        try {
            sqLiteDatabaseWrapper.update(TABLE_NAME, values, null, null);
        } catch (Throwable t) {
            IBGDiagnostics.reportNonFatal(t, "Error occurred while clearing generated captured w3c external trace id From DanglingAPMNetworkLog table: " + t.getMessage());
        }
    }

    @SuppressLint("Range")
    @VisibleForTesting
    public APMNetworkLog readApmNetworkLogFromCursor(Cursor cursor) {
        APMNetworkLog apmNetworkLog = new DefaultAPMNetworkLog();
        apmNetworkLog.setId(cursor.getInt(
                cursor.getColumnIndex(
                        InstabugDbContract.APMNetworkLogEntry.COLUMN_ID)));
        apmNetworkLog.setMethod(cursor.getString(
                cursor.getColumnIndex(
                        InstabugDbContract.APMNetworkLogEntry.COLUMN_METHOD)));
        apmNetworkLog.setCarrier(cursor.getString(
                cursor.getColumnIndex(
                        InstabugDbContract.APMNetworkLogEntry.COLUMN_CARRIER)));
        apmNetworkLog.setErrorMessage(cursor.getString(
                cursor.getColumnIndex(
                        InstabugDbContract.APMNetworkLogEntry.COLUMN_ERROR_MESSAGE)));
        apmNetworkLog.setRadio(cursor.getString(
                cursor.getColumnIndex(
                        InstabugDbContract.APMNetworkLogEntry.COLUMN_RADIO)));
        apmNetworkLog.setRequestBodySize(cursor.getInt(
                cursor.getColumnIndex(
                        InstabugDbContract.APMNetworkLogEntry.COLUMN_REQUEST_BODY_SIZE)));
        apmNetworkLog.setRequestContentType(cursor.getString(
                cursor.getColumnIndex(
                        InstabugDbContract.APMNetworkLogEntry.COLUMN_REQUEST_CONTENT_TYPE)));
        apmNetworkLog.setRequestHeaders(cursor.getString(
                cursor.getColumnIndex(
                        InstabugDbContract.APMNetworkLogEntry.COLUMN_REQUEST_HEADERS)));
        apmNetworkLog.setResponseBodySize(cursor.getInt(
                cursor.getColumnIndex(
                        InstabugDbContract.APMNetworkLogEntry.COLUMN_RESPONSE_BODY_SIZE)));
        apmNetworkLog.setResponseCode(cursor.getInt(
                cursor.getColumnIndex(
                        InstabugDbContract.APMNetworkLogEntry.COLUMN_RESPONSE_CODE)));
        apmNetworkLog.setClientErrorCode(cursor.getInt(
                cursor.getColumnIndex(
                        COLUMN_CLIENT_SIDE_ERROR_CODE
                )
        ));
        apmNetworkLog.setResponseContentType(cursor.getString(
                cursor.getColumnIndex(
                        InstabugDbContract.APMNetworkLogEntry.COLUMN_RESPONSE_CONTENT_TYPE)));
        apmNetworkLog.setResponseHeaders(cursor.getString(
                cursor.getColumnIndex(
                        InstabugDbContract.APMNetworkLogEntry.COLUMN_RESPONSE_HEADERS)));
        apmNetworkLog.setStartTime(cursor.getLong(
                cursor.getColumnIndex(
                        InstabugDbContract.APMNetworkLogEntry.COLUMN_START_TIME)));
        apmNetworkLog.setTotalDuration(cursor.getInt(
                cursor.getColumnIndex(
                        InstabugDbContract.APMNetworkLogEntry.COLUMN_DURATION)));
        apmNetworkLog.setUrl(cursor.getString(
                cursor.getColumnIndex(
                        InstabugDbContract.APMNetworkLogEntry.COLUMN_URL)));
        apmNetworkLog.setExecutedInBackground(cursor.getInt(
                cursor.getColumnIndex(
                        InstabugDbContract.APMNetworkLogEntry.COLUMN_EXECUTED_ON_BACKGROUND)) == 1);
        apmNetworkLog.setModified(cursor.getInt(cursor.getColumnIndex(COLUMN_USER_MODIFIED)) == 1);
        apmNetworkLog.setGraphQlQueryName(cursor.getString(
                cursor.getColumnIndex(
                        COLUMN_GRAPH_QL_QUERY_NAME)));
        apmNetworkLog.setGrpcMethodName(cursor.getString(
                cursor.getColumnIndex(
                        COLUMN_GRPC_METHOD_NAME
                )
        ));
        apmNetworkLog.setServerSideErrorMessage(cursor.getString(
                cursor.getColumnIndex(
                        COLUMN_SERVER_ERROR_MESSAGE)));
        apmNetworkLog.setLatencySpansJsonString(
                cursor.getString(cursor.getColumnIndex(COLUMN_NETWORK_LATENCY_SPANS))
        );
        extractExternalTraceId(cursor, apmNetworkLog);
        extractExternalTraceStartTimeMillis(cursor, apmNetworkLog);
        extractW3CExternalTraceAttributes(cursor, apmNetworkLog);
        String attributesQuery = "select* from " + InstabugDbContract.NetworkTracesAttributesEntry.TABLE_NAME
                + " where " + InstabugDbContract.NetworkTracesAttributesEntry.COLUMN_TRACE_ID + " = " + apmNetworkLog.getId();
        DatabaseManager databaseManager = getDatabaseManager();

        if (databaseManager != null) {
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            Cursor attributesCursor = null;
            try {
                attributesCursor = sqLiteDatabaseWrapper.rawQuery(attributesQuery, null);
                if (attributesCursor != null) {
                    Map<String, String> attrsMap = null;
                    while (attributesCursor.moveToNext()) {
                        if (attrsMap == null) {
                            attrsMap = new ArrayMap<>();
                        }
                        attrsMap.put(attributesCursor.getString(attributesCursor.getColumnIndex(InstabugDbContract.NetworkTracesAttributesEntry.COLUMN_KEY)),
                                attributesCursor.getString(attributesCursor.getColumnIndex(InstabugDbContract.NetworkTracesAttributesEntry.COLUMN_VALUE)));
                    }
                    apmNetworkLog.setAttributes(attrsMap);
                }
            } catch (Throwable ex) {
                apmLogger.logSDKError("Failed to read attributes of a trace", ex);
                IBGDiagnostics.reportNonFatal(ex, "DB execution a sql failed: " + ex.getMessage());
            } finally {
                if (attributesCursor != null) {
                    attributesCursor.close();
                }
            }
        }
        return apmNetworkLog;
    }

    private static void extractExternalTraceId(Cursor cursor, APMNetworkLog APMNetworkLog) {
        int externalTraceIdColumnIndex = cursor.getColumnIndex(COLUMN_EXTERNAL_TRACE_ID);
        if (externalTraceIdColumnIndex == -1 || cursor.isNull(externalTraceIdColumnIndex)) {
            APMNetworkLog.setExternalTraceId(null);
        } else {
            long externalTraceId = cursor.getLong(externalTraceIdColumnIndex);
            APMNetworkLog.setExternalTraceId(externalTraceId == 0L ? null : externalTraceId);
        }
    }

    private static void extractExternalTraceStartTimeMillis(Cursor cursor, APMNetworkLog APMNetworkLog) {
        int externalTraceStartTimeColumnIndex = cursor.getColumnIndex(COLUMN_EXTERNAL_TRACE_START_TIME_MILLIS);
        if (externalTraceStartTimeColumnIndex == -1 || cursor.isNull(externalTraceStartTimeColumnIndex)) {
            APMNetworkLog.setExternalTraceStartTimestampMillis(null);
        } else {
            long externalTraceStartTime = cursor.getLong(externalTraceStartTimeColumnIndex);
            APMNetworkLog.setExternalTraceStartTimestampMillis(
                    externalTraceStartTime == 0 ? null : externalTraceStartTime
            );
        }
    }

    private static void extractW3CExternalTraceAttributes(Cursor cursor, APMNetworkLog APMNetworkLog) {
        int isW3CTraceIdCapturedColumnIndex = cursor.getColumnIndexOrThrow(COLUMN_IS_W3C_EXTERNAL_NETWORK_TRACE_ID_CAPTURED);
        if (cursor.isNull(isW3CTraceIdCapturedColumnIndex)) {
            APMNetworkLog.setW3CTraceIdCaptured(null);
        } else {
            int isW3CTraceIdCaptured = cursor.getInt(isW3CTraceIdCapturedColumnIndex);
            APMNetworkLog.setW3CTraceIdCaptured(isW3CTraceIdCaptured != 0);
        }

        int w3cTraceIdPidColumnIndex = cursor.getColumnIndexOrThrow(COLUMN_W3C_EXTERNAL_NETWORK_TRACE_ID_PID);
        if (cursor.isNull(w3cTraceIdPidColumnIndex)) {
            APMNetworkLog.setGeneratedW3CPid(null);
        } else {
            long w3cTraceIdPid = cursor.getLong(w3cTraceIdPidColumnIndex);
            APMNetworkLog.setGeneratedW3CPid(w3cTraceIdPid);
        }

        int w3cTraceIdTimestampColumnIndex = cursor.getColumnIndexOrThrow(COLUMN_W3C_EXTERNAL_NETWORK_TRACE_ID_TIMESTAMP);
        if (cursor.isNull(w3cTraceIdTimestampColumnIndex)) {
            APMNetworkLog.setGeneratedW3CTimestampSeconds(null);
        } else {
            long w3cTraceIdTimestamp = cursor.getLong(w3cTraceIdTimestampColumnIndex);
            APMNetworkLog.setGeneratedW3CTimestampSeconds(w3cTraceIdTimestamp);
        }

        int generatedW3CTraceIdColumnIndex = cursor.getColumnIndexOrThrow(COLUMN_GENERATED_W3C_EXTERNAL_NETWORK_TRACE_ID);
        if (cursor.isNull(generatedW3CTraceIdColumnIndex)) {
            APMNetworkLog.setSyncableGeneratedW3CTraceId(null);
        } else {
            String generatedW3CTraceId = cursor.getString(generatedW3CTraceIdColumnIndex);
            APMNetworkLog.setSyncableGeneratedW3CTraceId(generatedW3CTraceId);
        }

        int capturedW3CTraceIdColumnIndex = cursor.getColumnIndexOrThrow(COLUMN_CAPTURED_W3C_EXTERNAL_NETWORK_TRACE_ID);
        if (cursor.isNull(capturedW3CTraceIdColumnIndex)) {
            APMNetworkLog.setSyncableCapturedW3CTraceId(null);
        } else {
            String capturedW3CTraceId = cursor.getString(capturedW3CTraceIdColumnIndex);
            APMNetworkLog.setSyncableCapturedW3CTraceId(capturedW3CTraceId);
        }
    }

    @Override
    public void clearNetworkSpansData() {
        DatabaseManager databaseManager = getDatabaseManager();
        if (databaseManager != null) {
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            if (sqLiteDatabaseWrapper != null) {
                ContentValues values = new ContentValues();
                values.putNull(COLUMN_NETWORK_LATENCY_SPANS);
                sqLiteDatabaseWrapper.update(TABLE_NAME,values,null,null);
            }
        }
    }
}
