package com.instabug.apm.util.powermanagement;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.os.BatteryManager;
import android.os.Build;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;

import com.instabug.apm.di.ServiceLocator;
import com.instabug.apm.logger.internal.Logger;
import com.instabug.library.core.InstabugCore;

import java.util.ArrayList;
import java.util.List;

/**
 * Receives and pass the changes of the battery power level changes
 * Send a {@link PowerManagementCallback} with tha changed level of the battery.
 */
@RequiresApi(api = Build.VERSION_CODES.JELLY_BEAN)
public class BatteryLevelChangeBroadcast extends BroadcastReceiver {

    private final Logger apmLogger = ServiceLocator.getApmLogger();
    private final List<PowerManagementCallback> callbacks;
    private final Context appContext;

    public BatteryLevelChangeBroadcast(@NonNull Context appContext) {
        callbacks = new ArrayList<>();
        this.appContext = appContext;
    }

    @Override
    public synchronized void onReceive(Context context, Intent intent) {
        if (intent != null) {
            int level = intent.getIntExtra(BatteryManager.EXTRA_LEVEL, 0);
            for (PowerManagementCallback callback : callbacks)
                callback.onBatteryLevelChanged(level);
        }
    }

    public synchronized void register(@Nullable PowerManagementCallback callback) {
        try {
            if (callback == null) {
                return;
            }
            if (callbacks.size() > 0) {
                apmLogger.logSDKProtected("BatteryLevelChangeBroadcast is already registered. Skipping re-registering");
            } else {
                appContext.registerReceiver(this, new IntentFilter(Intent.ACTION_BATTERY_CHANGED));
                apmLogger.logSDKProtected("batteryLevelReceiver registered on " + callback.getClass().getSimpleName());
            }
            callbacks.add(callback);
        } catch (Exception e) {
            apmLogger.logSDKProtected("Error: " + e.getMessage() + "While registering battery level receiver");
        }
    }

    public synchronized void unregister(@Nullable PowerManagementCallback callback) {
        try {
            if (callback == null) {
                return;
            }
            if (callbacks.remove(callback) && callbacks.isEmpty()) {
                appContext.unregisterReceiver(this);
                apmLogger.logSDKProtected("batteryLevelReceiver unregistered from " + callback.getClass().getSimpleName());
            }
        } catch (Exception e) {
            InstabugCore.reportError(e, "unregister BatteryLevelChangeBroadcast got error");
        }
    }
}
