package com.instabug.apm.compose.compose_spans.handler

import com.instabug.apm.cache.handler.session.SessionMetaDataCacheHandler
import com.instabug.apm.compose.compose_spans.configuration.ComposeSpansConfigurationProvider
import com.instabug.apm.compose.compose_spans.model.ComposeSpansCacheModel
import com.instabug.apm.logger.internal.Logger

class ComposeSpansHandlerImpl(
    private val composeSpansCacheHandler: ComposeSpansCacheHandler,
    private val metaDataCacheHandler: SessionMetaDataCacheHandler,
    private val configurations: ComposeSpansConfigurationProvider,
    private val logger: Logger
) : ComposeSpansHandler {

    override fun save(model: ComposeSpansCacheModel, sessionId: String?) {
        sessionId?.also { apmSessionId ->
            takeIf { configurations.enabled }
                ?.let { composeSpansCacheHandler.insert(model, apmSessionId) }
                .takeIf { it != -1L }
                ?.let {
                    metaDataCacheHandler.addToComposeSpansTotalCount(apmSessionId, 1)
                    val droppedRequestLimit =
                        composeSpansCacheHandler.trim(apmSessionId, configurations.requestLimit)
                    composeSpansCacheHandler.trimAll(configurations.storeLimit)
                    droppedRequestLimit
                }?.takeIf { it > 0 }
                ?.also {
                    metaDataCacheHandler.addToComposeSpansDroppedCount(apmSessionId, it)
                    logger.d("Composable spans dropped count: $it")
                }
        } ?: let { logger.e("No sessions found, skipping saving compose spans") }
    }

    override fun getForSession(sessionId: String): List<ComposeSpansCacheModel> =
        composeSpansCacheHandler.retrieve(sessionId) ?: emptyList()

    override fun clear(sessionId: String) = composeSpansCacheHandler.clear(sessionId)

    override fun clearAll() {
        composeSpansCacheHandler.clearAll()
        metaDataCacheHandler.resetComposeSpansCounts()
    }

    override fun clearScreenLoadingSegmentationData() {
        composeSpansCacheHandler.clearScreenLoadingSegmentationData()
    }
}
