package com.instabug.apm.compose.compose_spans

import com.instabug.apm.compose.ComposeEventDispatcher
import com.instabug.apm.compose.ComposeEventListener
import com.instabug.apm.compose.compose_spans.configuration.ComposeSpansConfigurationProvider
import com.instabug.apm.compose.compose_spans.handler.ComposeSpansHandler
import com.instabug.apm.configuration.APMConfigurationProvider
import com.instabug.apm.constants.ErrorMessages
import com.instabug.apm.logger.APMLogger
import com.instabug.library.factory.Factory
import java.util.concurrent.Executor

class ComposeSpansManagerImpl(
    private val apmConfigurations: APMConfigurationProvider,
    private val executor: Executor,
    private val listenerFactory: Factory<ComposeEventListener>,
    private val configurations: ComposeSpansConfigurationProvider,
    private val handler: ComposeSpansHandler
) : ComposeSpansManager {

    private var listener: ComposeEventListener? = null

    private fun startSynchronous() {
        synchronized(this) {
            if (listener == null && configurations.enabled) {
                val localListener = listenerFactory.create()
                listener = localListener
                ComposeEventDispatcher.addListener(localListener)
            }
        }
    }

    private fun stopSynchronous() {
        synchronized(this) {
            listener?.let {
                ComposeEventDispatcher.removeListener(it)
                listener = null
            }
        }
    }

    override fun start() {
        executor.execute { startSynchronous() }
    }

    override fun stop() {
        executor.execute { stopSynchronous() }
    }

    override fun onStateChanged() {
        executor.execute {
            if (configurations.enabled) {
                startSynchronous()
                clearSegmentationDataIfNotEnabled()
            } else {
                stopSynchronous()
                handler.clearAll()
            }
        }
    }

    private fun clearSegmentationDataIfNotEnabled() {
        if (!configurations.screenLoadingSegmentationBeEnabled) {
            handler.clearScreenLoadingSegmentationData()
        }
    }

    override fun setSDKEnabled(isEnabled: Boolean) = when {
        !apmConfigurations.isAPMFeatureAvailable -> APMLogger.e(ErrorMessages.COMPOSE_APM_BE_DISABLED)
        !apmConfigurations.isAPMSdkEnabled -> APMLogger.e(ErrorMessages.COMPOSE_APM_SDK_DISABLED)
        !configurations.featureEnabled -> APMLogger.e(ErrorMessages.COMPOSE_FEATURE_BE_DISABLED)
        else -> setComposeSdkEnabledValue(isEnabled)
    }

    private fun setComposeSdkEnabledValue(isEnabled: Boolean) {
        configurations.sdkEnabled = isEnabled
        onStateChanged()
    }
}
