package com.instabug.apm.appflow

import com.instabug.apm.appflow.model.AppFlowCacheModel
import com.instabug.apm.cache.model.SessionCacheModel
import com.instabug.apm.cache.model.SessionMetaData
import com.instabug.apm.networking.mapping.sessions.SessionFeatureJsonFiller
import com.instabug.apm.networking.mapping.sessions.SessionFeatureJsonFiller.Companion.getDropCount
import com.instabug.library.map.Mapper
import org.json.JSONArray
import org.json.JSONObject

class SessionAppFlowJsonFiller(
    private val appFlowMapper: Mapper<List<AppFlowCacheModel>?, JSONArray?>,
) : SessionFeatureJsonFiller {

    companion object {
        const val TRACE_OBJECT_KEY = "ts"
        const val TRACE_LIST_KEY = "tl"
        const val DROP_COUNT_STORE_LIMIT_KEY = "dcsl"
        const val DROP_COUNT_REQUEST_LIMIT_KEY = "dcrl"
    }

    override fun addToJsonObject(model: SessionCacheModel, sessionJsonObject: JSONObject) {
        model.sessionMetaData
            ?.takeIf { hasAppFlows(it, model.appFlows?.size ?: 0) }
            ?.let { metaData ->
                val (appFlowDropStoreLimit, appFlowDropRequestLimit) = getDropCount(
                    model::getAppFlows,
                    metaData::getAppFlowTotalCount,
                    metaData::getAppFlowDroppedCountRequestLimit
                )

                val tracesJsonObject = JSONObject().apply {
                    appFlowDropStoreLimit.takeIf { it > 0 }
                        ?.let { put(DROP_COUNT_STORE_LIMIT_KEY, it) }
                    appFlowDropRequestLimit.takeIf { it > 0 }
                        ?.let { put(DROP_COUNT_REQUEST_LIMIT_KEY, it) }

                    // Add app flow data to the traces JSON object
                    appFlowMapper.map(model.appFlows)?.takeIf { appFlowsJsonArray ->
                        appFlowsJsonArray.length() > 0
                    }?.let { appFlowsJsonArray ->
                        put(TRACE_LIST_KEY, appFlowsJsonArray)
                    }
                }
                sessionJsonObject.put(TRACE_OBJECT_KEY, tracesJsonObject)
            }
    }

    private fun hasAppFlows(metaData: SessionMetaData, cachedAppFlowsCount: Int) =
        cachedAppFlowsCount + metaData.appFlowTotalCount  > 0
}