package com.instabug.apm;

import static com.instabug.apm.constants.ErrorMessages.UI_TRACE_ENDED_FROM_BACKGROUND_THREAD;

import android.app.Activity;
import android.os.Build;
import android.os.Looper;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;

import com.instabug.apm.appflow.model.AppFlowAttribute;
import com.instabug.apm.attributes.listeners.OnNetworkTraceListener;
import com.instabug.apm.di.ServiceLocator;
import com.instabug.apm.logger.internal.Logger;
import com.instabug.apm.model.EventTimeMetricCapture;
import com.instabug.apm.model.TimeCaptureBoundModel;
import com.instabug.library.apichecker.APIChecker;
import com.instabug.library.apichecker.VoidRunnable;
import com.instabug.library.logging.listeners.networklogs.NetworkLogListener;


public class APM {
    private static final APMImplementation apmImplementation;
    private static final Logger apmLogger;

    static {
        apmImplementation = ServiceLocator.getApmImplementation();
        apmLogger = ServiceLocator.getApmLogger();
    }

    /**
     * Disables/Enables APM.
     * <br/>
     * Defaults to true if APM is included in your Instabug account's plan.
     *
     * @param enabled <b>true</b> to enable APM, <b>false</b> to disable APM.
     */
    public static void setEnabled(final boolean enabled) {
        APIChecker.checkAndRunInExecutor("APM.setEnabled", new VoidRunnable() {
            @Override
            public void run() {
                apmImplementation.setEnabled(enabled);
            }
        });

    }

    /**
     * Disables/Enables Cold App Launch tracking.
     * <br/>
     * Defaults to true if APM is enabled. If APM is disabled, App Launch time will not be captured.
     * <br/>
     * More information can be found <a href="https://docs.instabug.com/docs/android-apm-app-launch">here</a>
     *
     * @param enabled <b>true</b> to enable App Launch tracking, <b>false</b> for disabling it.
     */
    public static void setColdAppLaunchEnabled(final boolean enabled) {
        APIChecker.checkAndRunInExecutor("APM.setColdAppLaunchEnabled", new VoidRunnable() {
            @Override
            public void run() {
                apmImplementation.setColdAppLaunchEnabled(enabled);
            }
        });
    }

    /**
     * Disables/Enables Hot App Launch tracking.
     * <br/>
     * Defaults to true if APM is enabled. If APM is disabled, App Launch time will not be captured.
     * <br/>
     * More information can be found <a href="https://docs.instabug.com/docs/android-apm-app-launch">here</a>
     *
     * @param enabled <b>true</b> to enable App Launch tracking, <b>false</b> for disabling it.
     */
    public static void setHotAppLaunchEnabled(final boolean enabled) {
        APIChecker.checkAndRunInExecutor("APM.setHotAppLaunchEnabled", new VoidRunnable() {
            @Override
            public void run() {
                apmImplementation.setHotAppLaunchEnabled(enabled);
            }
        });
    }

    /**
     * Disables/Enables Warm App Launch tracking
     * <br>
     * Defaults to true if APM is enabled, if APM is disabled, Warm App Launch time will not be captured.
     * </>
     *
     * @param enabled <b>true</b> to enable Warm App Launch tracking, <b>false</b> for disabling it.
     */
    public static void setWarmAppLaunchEnabled(final boolean enabled) {
        APIChecker.checkAndRunInExecutor(
                "APM.setWarmAppLaunchEnabled",
                () -> apmImplementation.setWarmAppLaunchEnabled(enabled)
        );
    }

    /**
     * Disables/Enables Automatic UI Traces.
     * <br/>
     * Defaults to true if APM is enabled. If APM is disabled, no Automatic UI Traces will be captured.
     * <br/>
     * More information can be found <a href="https://docs.instabug.com/docs/android-apm-ui-hangs">here</a>
     *
     * @param enabled <b>true</b> to enable Automatic UI traces, <b>false</b> for disabling it.
     */
    public static void setAutoUITraceEnabled(final boolean enabled) {
        APIChecker.checkAndRunInExecutor("APM.setAutoUITraceEnabled", new VoidRunnable() {
            @Override
            public void run() {
                apmImplementation.setAutoUITraceEnabled(enabled);
            }
        });
    }

    /**
     * Disables/Enables UI Hangs.
     * <br/>
     * Defaults to true if APM is enabled. If APM is disabled, no UI Hangs will be captured.
     * <br/>
     *
     * @param enabled <b>true</b> to enable UI Hangs, <b>false</b> for disabling it.
     */
    public static void setUIHangEnabled(final boolean enabled) {
        APIChecker.checkAndRunInExecutor("APM.setUIHangEnabled", new VoidRunnable() {
            @Override
            public void run() {
                apmImplementation.setUIHangEnabled(enabled);
            }
        });
    }

    /**
     * Disables/Enables UI Loading metrics.
     * <br/>
     * Defaults to true if APM is enabled. If APM is disabled, no UI Loading metrics will be captured.
     * <br/>
     *
     * @param enabled <b>true</b> to enable UI metrics, <b>false</b> for disabling it.
     */
    public static void setScreenLoadingEnabled(final boolean enabled) {
        APIChecker.checkAndRunInExecutor("APM.setScreenLoadingEnabled", new VoidRunnable() {
            @Override
            public void run() {
                apmImplementation.setScreenLoadingEnabled(enabled);
            }
        });
    }

    /**
     * Starts a Custom UI trace with the specified name.
     * <br/>
     * If APM is disabled, Custom UI Traces are not started
     * <br/>
     * Custom UI Traces cannot run in parallel, one must be ended before the other is started.
     * <br/>
     * Custom UI Trace name cannot exceed 150 characters, otherwise it's trimmed,
     * leading and trailing whitespaces are also ignored.
     * <br/>
     * This API should be called from main thread.
     * <br/>
     * More information can be found <a href="https://docs.instabug.com/docs/android-apm-ui-hangs">here</a>
     *
     * @param name Custom UI Trace name.
     */
    @RequiresApi(api = Build.VERSION_CODES.JELLY_BEAN)
    public static void startUITrace(@NonNull final String name) {
        final Looper callerThreadLooper = Looper.myLooper();
        APIChecker.checkAndRunInExecutor("APM.startUITrace", new VoidRunnable() {
            @Override
            public void run() {
                apmImplementation.startUiTrace(name, callerThreadLooper);
            }
        });
    }

    /**
     * Ends the current running Custom UI Trace.
     * <br/>
     * More information can be found <a href="https://docs.instabug.com/docs/android-apm-ui-hangs">here</a>
     */
    @RequiresApi(api = Build.VERSION_CODES.JELLY_BEAN)
    public static void endUITrace() {
        final Looper callerThreadLooper = Looper.myLooper();
        if (callerThreadLooper != Looper.getMainLooper()) {
            apmLogger.w(UI_TRACE_ENDED_FROM_BACKGROUND_THREAD);
        }
        APIChecker.checkAndRunInExecutor("APM.endUITrace", new VoidRunnable() {
            @Override
            public void run() {
                apmImplementation.endUiTrace(callerThreadLooper);
            }
        });
    }

    /**
     * Adds {@link OnNetworkTraceListener} that fires after a network trace finishes.
     */
    public static void addOnNetworkTraceListener(final OnNetworkTraceListener listener) {
        APIChecker.checkAndRunInExecutor("APM.addOnNetworkTraceListener",
                new VoidRunnable() {
                    @Override
                    public void run() {
                        apmImplementation.addOnNetworkTraceListener(listener);
                    }
                });
    }

    /**
     * Removes {@link OnNetworkTraceListener}.
     */
    public static void removeOnNetworkTraceListener(final OnNetworkTraceListener listener) {
        APIChecker.checkAndRunInExecutor("APM.removeOnNetworkTraceListener",
                new VoidRunnable() {
                    @Override
                    public void run() {
                        apmImplementation.removeOnNetworkTraceListener(listener);
                    }
                });
    }

    public static void endAppLaunch() {
        ServiceLocator.getAppLaunchDataRepository().setAppLaunchEndMicro(System.nanoTime() / 1000);
        APIChecker.checkAndRunInExecutor("APM.endAppLaunch", apmImplementation::endAppLaunch);
    }

    /**
     * Mark loading for an activity as finished
     *
     * @param activityClass class of the activity to be marked finished loading
     */
    public static <ActivityType extends Activity> void endScreenLoading(Class<ActivityType> activityClass) {
        final EventTimeMetricCapture timeMetric = new EventTimeMetricCapture();
        APIChecker.checkAndRunInExecutor("APM.endScreenLoading", () -> {
            apmImplementation.endScreenLoading(activityClass, timeMetric);
        });
    }

    /**
     * Disables/Enables Fragment spans metric.
     * <br/>
     * Defaults to true if APM is enabled. If APM is disabled, no Fragment spans will be captured.
     * <br/>
     *
     * @param enabled <b>true</b> to enable Fragment spans, <b>false</b> for disabling it.
     */
    public static void setFragmentSpansEnabled(boolean enabled) {
        APIChecker.checkAndRunInExecutor("APM.setFragmentSpansEnabled", () -> {
            apmImplementation.setFragmentSpansEnabled(enabled);
        });
    }

    /**
     * Registers a listener that is triggered when a network log is captured, which allows the caller
     * to modify the captured network log
     *
     * @param listener {@link NetworkLogListener} that will be called, null to remove the current listener
     */
    public static void registerNetworkLogsListener(@Nullable NetworkLogListener listener) {
        APIChecker.checkAndRunInExecutor(
                "APM.registerNetworkLogsListener",
                () -> apmImplementation.registerNetworkLogsListener(listener)
        );
    }

    /**
     * Disables/Enables Compose spans metric.
     * <br/>
     * Defaults to true if APM is enabled. If APM is disabled, no Compose spans will be captured.
     * <br/>
     *
     * @param enabled <b>true</b> to enable Compose spans, <b>false</b> for disabling it.
     */
    public static void setComposeSpansEnabled(boolean enabled) {
        APIChecker.checkAndRunInExecutor(
                "APM.setComposeSpansEnabled",
                () -> apmImplementation.setComposeSpansEnabled(enabled)
        );
    }

    /**
     * Disables/Enables WebView traces.
     * <br/>
     * Defaults to true if APM is enabled. If APM is disabled, no WebView traces will be captured.
     * <br/>
     *
     * @param enabled <b>true</b> to enable WebView traces, <b>false</b> for disabling it.
     */
    public static void setWebViewsTrackingEnabled(boolean enabled) {
        APIChecker.checkAndRunInExecutor(
                "APM.setWebViewsEnabled",
                () -> apmImplementation.setWebViewsTrackingEnabled(enabled)
        );
    }

    /**
     * Starts an AppFlow with the specified name.
     * <br/>
     * On starting two flows with the same name the older flow will end with force abandon end reason.
     * AppFlow name cannot exceed 150 characters otherwise it's truncated,
     * leading and trailing whitespaces are also ignored.
     * <br/>
     * This API is thread safe.
     * <br/>
     *
     * @param name AppFlow name. It can not be empty string or null.
     *             Starts a new AppFlow, if APM is enabled, feature is enabled
     *             and Instabug SDK is initialised.
     */
    public static void startFlow(@NonNull String name) {
        TimeCaptureBoundModel<String> boundName =
                new TimeCaptureBoundModel<>(name);
        APIChecker.checkAndRunInExecutor(
                "APM.startFlow",
                () -> apmImplementation.startFlow(boundName)
        );
    }

    /**
     * Ends AppFlow with a given name.
     *
     * @param name AppFlow name to be ended. It can not be empty string or null
     */
    public static void endFlow(@NonNull String name) {
        TimeCaptureBoundModel<String> boundName =
                new TimeCaptureBoundModel<>(name);
        APIChecker.checkAndRunInExecutor(
                "APM.endFlow",
                () -> apmImplementation.endFlow(boundName)
        );
    }

    /**
     * Sets custom attributes for AppFlow with a given name.
     * <br/>
     * Setting an attribute value to null will remove its corresponding key if it already exists.
     * <br/>
     * Attribute key name cannot exceed 30 characters.
     * Leading and trailing whitespaces are also ignored.
     * Does not accept empty strings or null.
     * <br/>
     * Attribute value name cannot exceed 60 characters,
     * leading and trailing whitespaces are also ignored.
     * Does not accept empty strings.
     * <br/>
     * If a trace is ended, attributes will not be added and existing ones will not be updated.
     * <br/>
     *
     * @param name  AppFlow name. It can not be empty string or null
     * @param key   AppFlow attribute value. It can not be empty string or null
     * @param value AppFlow attribute key. It can not be empty string. Null to remove attribute
     */
    public static void setFlowAttribute(@NonNull String name, @NonNull String key, String value) {
        AppFlowAttribute attribute = new AppFlowAttribute(name, key, value);
        TimeCaptureBoundModel<AppFlowAttribute> timeBoundAttribute = new TimeCaptureBoundModel<>(attribute);
        APIChecker.checkAndRunInExecutor(
                "APM.setFlowAttribute",
                () -> apmImplementation.setFlowAttribute(timeBoundAttribute)
        );
    }
}