package com.instabug.apm.networkinterception.utils;

import com.instabug.apm.constants.Constants;
import com.instabug.library.networkv2.BodyBufferHelper;
import com.instabug.library.util.InstabugSDKLogger;

import java.io.IOException;
import java.io.OutputStream;

/**
 * Wrapper of OutputStream which counts number of bytes being written
 * to calculate request size
 */
public class APMCountableOutputStream extends OutputStream {

    private final OutputStream bytesOut;
    private Long count = 0L;

    private boolean disableBodyBuffer;

    public APMCountableOutputStream(OutputStream bytesOut) {
        this.bytesOut = bytesOut;
    }

    public void setDisableBodyBuffer(final boolean disableBodyBuffer) {
        this.disableBodyBuffer = disableBodyBuffer;
    }

    @Override
    public void write(int b) throws IOException {
        try {
            bytesOut.write(b);
            count++;
        } catch (OutOfMemoryError e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, e.getMessage() != null ? e.getMessage() : "Couldn't allocate enough memory to write " + b + " bytes", e);
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, e.getMessage() != null ? e.getMessage() : "Couldn't write body bytes", e);
        }

    }

    @Override
    public void write(byte[] b, int off, int len) throws IOException {
        try {
            bytesOut.write(b, off, len);
            count += len;
        } catch (OutOfMemoryError e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, e.getMessage() != null ? e.getMessage() : "Couldn't allocate enough memory to write " + b.length + " bytes", e);
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, e.getMessage() != null ? e.getMessage() : "Couldn't write body byte array", e);
        }
    }

    @Override
    public void close() throws IOException {
        try {
            bytesOut.close();
        } catch (OutOfMemoryError e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, e.getMessage() != null ? e.getMessage() : "Couldn't allocate enough memory to close the outputStream", e);
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, e.getMessage() != null ? e.getMessage() : "Couldn't close the outputStream", e);
        }

    }

    public Long getCount() {
        return count;
    }

    private boolean shouldDisableBodyBuffer() {
        return disableBodyBuffer || BodyBufferHelper.isBodySizeAllowed(count);
    }
}