package com.instabug.apm.cache.handler.executiontraces;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.apm.cache.model.ExecutionTraceCacheModel;

import java.util.List;
import java.util.Map;

public interface ExecutionTracesCacheHandler {

    /**
     * Inserts trace in cache with the given name and start time
     *
     * @param sessionID id of the session which this trace belongs to
     * @param trace     the trace to insert
     * @return id of created trace
     */
    boolean insertTrace(@NonNull String sessionID, @NonNull ExecutionTraceCacheModel trace);

    /**
     * Adds new attribute to specific trace
     *
     * @param traceId   id of the associated trace
     * @param traceName name of the associated trace
     * @param key       attribute key
     * @param value     attribute value
     */
    void setAttribute(long traceId, @NonNull String traceName, @NonNull String key, @Nullable String value);

    /**
     * Adds new attribute to specific trace
     *
     * @param traceId   id of the associated trace
     * @param traceName name of the associated trace
     * @param attrs the map of attributes
     */
    void setAttributes(long traceId, @NonNull String traceName, @NonNull Map<String, String> attrs);

    /**
     * Removes all custom traces with its attributes
     */
    void removeAll();

    /**
     * Removes all un-ended custom traces along side with there attributes
     */
    void removeUnEndedTraces();

    /**
     * Gets list of cached ended custom traces for passed session id
     *
     * @param sessionID id of the session to get cached custom
     *                  traces for
     * @return List of {@link ExecutionTraceCacheModel}
     */
    @Nullable
    List<ExecutionTraceCacheModel> getExecutionTracesForSession(String sessionID);

    /**
     * Removes oldest ExecutionTraces with its attributes till the table reach the limit passed
     *
     * @param limit number of traces to be remained
     */
    void trimTracesToLimit(long limit);

    /**
     * Removes oldest ExecutionTraces with its attributes till
     * total traces count for the passed sessionID reaches the limit passed
     *
     * @param sessionID Session id to limit its traces
     * @param limit     number of traces to be remained
     */
    int trimTracesToLimit(String sessionID, long limit);

    /**
     * Checks whether or not a trace exists given its ID
     *
     * @param traceId the id of the trace under test
     * @return true if this trace exists. False otherwise.
     */
    boolean exists(long traceId);
}
