package com.instabug.apm.appflow.di

import android.content.ContentValues
import android.database.Cursor
import com.instabug.apm.appStateDispacher.AppStateEventListener
import com.instabug.apm.appflow.AppFlowAppStateEventListener
import com.instabug.apm.appflow.AppFlowSessionModelFiller
import com.instabug.apm.appflow.AppFlowSessionObserver
import com.instabug.apm.appflow.AppFlowSessionReadinessHandler
import com.instabug.apm.appflow.SessionAppFlowAndExecutionTraceJsonFiller
import com.instabug.apm.appflow.configuration.AppFlowConfigurationHandler
import com.instabug.apm.appflow.configuration.AppFlowConfigurationProvider
import com.instabug.apm.appflow.configuration.AppFlowConfigurationProviderImpl
import com.instabug.apm.appflow.handler.AppFlowCacheHandler
import com.instabug.apm.appflow.handler.AppFlowCacheHandlerImpl
import com.instabug.apm.appflow.handler.AppFlowHandler
import com.instabug.apm.appflow.handler.AppFlowHandlerImpl
import com.instabug.apm.appflow.manager.AppFlowManager
import com.instabug.apm.appflow.manager.AppFlowManagerImpl
import com.instabug.apm.appflow.map.AppFlowAttributeContentValuesMapper
import com.instabug.apm.appflow.map.AppFlowContentValuesMapper
import com.instabug.apm.appflow.map.AppFlowCursorParser
import com.instabug.apm.appflow.map.AppFlowJsonMapper
import com.instabug.apm.appflow.model.AppFlowAttribute
import com.instabug.apm.appflow.model.AppFlowAttributeInsertionCacheModel
import com.instabug.apm.appflow.model.AppFlowCacheModel
import com.instabug.apm.appflow.model.AppFlowInsertionCacheModel
import com.instabug.apm.appflow.usecases.AppFLowAppLaunchUseCase
import com.instabug.apm.appflow.usecases.AppFlowBackgroundUseCase
import com.instabug.apm.appflow.usecases.AppFlowForegroundUseCase
import com.instabug.apm.appflow.usecases.AppFlowNewSessionUseCase
import com.instabug.apm.appflow.usecases.DisabledUseCase
import com.instabug.apm.appflow.usecases.EndAppFlowUseCase
import com.instabug.apm.appflow.usecases.RefreshBackgroundFlowUseCase
import com.instabug.apm.appflow.usecases.SetFlowAttributeUseCase
import com.instabug.apm.appflow.usecases.StartAppFlowUseCase
import com.instabug.apm.appflow.usecases.UseCase
import com.instabug.apm.appflow.validate.AppFlowAttributeConfigurationValidator
import com.instabug.apm.appflow.validate.AppFlowAttributeSanitizer
import com.instabug.apm.appflow.validate.AppFlowAttributeValidator
import com.instabug.apm.appflow.validate.AppFlowNameSanitizer
import com.instabug.apm.appflow.validate.AppFlowNameValidator
import com.instabug.apm.appflow.validate.ConfigurationsValidator
import com.instabug.apm.common.concurrent.OrderedExecutor
import com.instabug.apm.configuration.ConfigurationHandler
import com.instabug.apm.di.Provider
import com.instabug.apm.di.ServiceLocator
import com.instabug.apm.di.getOrCreateSingleInstance
import com.instabug.apm.di.getOrCreateSingleton
import com.instabug.apm.handler.session.APMSessionObserver
import com.instabug.apm.model.TimeCaptureBoundModel
import com.instabug.apm.networking.mapping.sessions.SessionFeatureJsonFiller
import com.instabug.apm.networking.mapping.sessions.SessionModelFiller
import com.instabug.apm.sanitization.Sanitizer
import com.instabug.apm.sanitization.Validator
import com.instabug.apm.v3_session_data_readiness.APMSessionReadinessHandler
import com.instabug.apm.v3_session_data_readiness.APMSessionReadinessManager
import com.instabug.library.core.eventbus.AppStateEvent
import com.instabug.library.core.eventbus.AppStateEvent.BackgroundAppStateEvent
import com.instabug.library.core.eventbus.AppStateEvent.ForegroundAppStateEvent
import com.instabug.library.map.Mapper
import com.instabug.library.model.common.Session
import com.instabug.library.parse.Parser
import com.instabug.library.util.threading.PoolProvider
import org.json.JSONArray
import java.lang.ref.WeakReference
import java.util.concurrent.ExecutorService

object AppFlowServiceLocator {

    private const val APP_FLOW_EXECUTOR_ID = "app_flow_executor"
    private const val startFlowApiName = "startFlow"
    private const val endFlowApiName = "endFlow"
    private const val setFlowAttributeApiName = "setFlowAttribute"

    @Volatile
    private var _configurationProvider: AppFlowConfigurationProvider? = null

    @Volatile
    private var _manager: AppFlowManagerImpl? = null

    private var executorReference: WeakReference<ExecutorService>? = null
    private var cacheHandlerReference: WeakReference<AppFlowCacheHandler?>? = null
    private var handlerReference: WeakReference<AppFlowHandler?>? = null
    private var startFlowUseCaseReference:
            WeakReference<UseCase<TimeCaptureBoundModel<String?>, Unit>?>? = null
    private var endFlowUseCaseReference:
            WeakReference<UseCase<TimeCaptureBoundModel<String?>, Unit>?>? = null
    private var addAttributeUseCaseReference:
            WeakReference<UseCase<TimeCaptureBoundModel<AppFlowAttribute>, Unit>?>? = null
    private var disabledUseCaseReference:
            WeakReference<UseCase<Unit, Unit>?>? = null
    private var appFLowAppLaunchUseCaseReference:
            WeakReference<UseCase<Unit, Unit>?>? = null

    @Volatile
    private var newSessionUseCaseReference:
            WeakReference<UseCase<Pair<Session, Session?>, Unit>?>? = null
    private var backgroundUseCaseReference: WeakReference<UseCase<Unit, Unit>?>? = null
    private var foregroundUseCaseReference: WeakReference<UseCase<Pair<ForegroundAppStateEvent, BackgroundAppStateEvent?>, Unit>?>? =
        null
    private var refreshBackgroundFlowUseCaseReference: WeakReference<UseCase<Long, Boolean?>>? =
        null
    private var appFlowApmSessionReadinessHandlerReference: WeakReference<APMSessionReadinessHandler?>? =
        null


    private fun createAppFloConfigurationProvider(): AppFlowConfigurationProvider? =
        ServiceLocator.getAPMPreferencePropertyFactory()?.let { preferencePropertyFactory ->
            AppFlowConfigurationProviderImpl(
                ServiceLocator.getApmConfigurationProvider(),
                ServiceLocator.getLimitConstraintApplier(),
                preferencePropertyFactory
            )
        }

    val configurationProvider: AppFlowConfigurationProvider?
        get() = getOrCreateSingleton(
            ::_configurationProvider,
            { _configurationProvider = it },
            ::createAppFloConfigurationProvider
        )

    private inline val appFlowInsertionModelContentValuesMapper:
            Mapper<AppFlowInsertionCacheModel, ContentValues>
        get() = AppFlowContentValuesMapper()

    private inline val appFlowAttributeContentValuesMapper:
            Mapper<AppFlowAttributeInsertionCacheModel, ContentValues>
        get() = AppFlowAttributeContentValuesMapper()

    private inline val appFlowCursorParser: Parser<Cursor?, List<AppFlowCacheModel>?>
        get() = AppFlowCursorParser()

    private fun createAppFlowCacheHandler(): AppFlowCacheHandler? =
        ServiceLocator.getDatabaseManager()?.let { databaseManager ->
            AppFlowCacheHandlerImpl(
                databaseManager,
                ServiceLocator.getApmLogger(),
                appFlowInsertionModelContentValuesMapper,
                appFlowAttributeContentValuesMapper,
                appFlowCursorParser
            )
        }

    val cacheHandler: AppFlowCacheHandler?
        get() = getOrCreateSingleInstance(
            ::cacheHandlerReference,
            { cacheHandlerReference = it },
            ::createAppFlowCacheHandler
        )

    private fun createAppFlowHandler(): AppFlowHandler? =
        configurationProvider?.let { config ->
            cacheHandler?.let { cache ->
                ServiceLocator.getSessionMetaDataCacheHandler()?.let { metaDataHandler ->
                    AppFlowHandlerImpl(
                        cache,
                        metaDataHandler,
                        ServiceLocator.getSessionHandler(),
                        config,
                        ServiceLocator.getApmLogger(),
                        ServiceLocator.getAppLaunchIdProvider()
                    )
                }
            }
        }

    val handler: AppFlowHandler?
        get() = getOrCreateSingleInstance(
            ::handlerReference,
            { handlerReference = it },
            ::createAppFlowHandler
        )

    val configurationHandler: ConfigurationHandler?
        get() = configurationProvider?.let(::AppFlowConfigurationHandler)

    private fun createAppFlowExecutor(): ExecutorService =
        OrderedExecutor(
            APP_FLOW_EXECUTOR_ID,
            PoolProvider.getInstance().orderedExecutor
        )

    internal val appFlowExecutor: ExecutorService
        get() = getOrCreateSingleInstance(
            ::executorReference,
            { executorReference = it },
            ::createAppFlowExecutor
        )

    private fun getFlowConfigurationValidator(apiName: String): Validator<Unit>? =
        configurationProvider?.let { appFlowConfigurations ->
            ConfigurationsValidator(
                apiName,
                ServiceLocator.getApmConfigurationProvider(),
                appFlowConfigurations,
                ServiceLocator.getApmLogger()
            )
        }

    private val flowNameValidator: Validator<String?>
        get() = AppFlowNameValidator(
            ServiceLocator.getApmLogger()
        )

    private val flowNameSanitizer: Sanitizer<String?>
        get() = AppFlowNameSanitizer(
            ServiceLocator.getApmLogger()
        )

    private val flowAttributeValidator: Validator<AppFlowAttribute>
        get() = AppFlowAttributeValidator(flowNameValidator, ServiceLocator.getApmLogger())

    private val flowAttributeSanitizer: Sanitizer<AppFlowAttribute>
        get() = AppFlowAttributeSanitizer(flowNameSanitizer)

    private val flowAttributeConfigurationValidator: Validator<Unit>?
        get() =
            configurationProvider?.let { appFlowConfigurations ->
                AppFlowAttributeConfigurationValidator(
                    setFlowAttributeApiName,
                    ServiceLocator.getApmConfigurationProvider(),
                    appFlowConfigurations,
                    ServiceLocator.getApmLogger(),
                )
            }

    private val appStateProvider: Provider<AppStateEvent?>
        get() = AppStateProvider()

    private fun createStartFlowUseCase(): UseCase<TimeCaptureBoundModel<String?>, Unit>? =
        handler?.let { appFlowHandler ->
            getFlowConfigurationValidator(startFlowApiName)?.let { configurationsValidator ->
                StartAppFlowUseCase(
                    handler = appFlowHandler,
                    logger = ServiceLocator.getApmLogger(),
                    configurationsValidator = configurationsValidator,
                    flowNameValidator = flowNameValidator,
                    flowNameSanitizer = flowNameSanitizer,
                    appStateProvider = appStateProvider,
                    refreshBackgroundFlowUseCase = getRefreshBackgroundUseCase(appFlowHandler)
                )
            }
        }

    internal val startFlowUseCase: UseCase<TimeCaptureBoundModel<String?>, Unit>?
        get() = getOrCreateSingleInstance(
            ::startFlowUseCaseReference,
            { startFlowUseCaseReference = it },
            ::createStartFlowUseCase
        )

    private fun createEndFlowUseCase(): UseCase<TimeCaptureBoundModel<String?>, Unit>? =
        handler?.let { appFlowHandler ->
            getFlowConfigurationValidator(endFlowApiName)?.let { configurationsValidator ->
                EndAppFlowUseCase(
                    handler = appFlowHandler,
                    logger = ServiceLocator.getApmLogger(),
                    configurationsValidator = configurationsValidator,
                    flowNameValidator = flowNameValidator,
                    flowNameSanitizer = flowNameSanitizer,
                    refreshBackgroundFlowUseCase = getRefreshBackgroundUseCase(appFlowHandler)
                )
            }
        }

    internal val endFlowUseCase: UseCase<TimeCaptureBoundModel<String?>, Unit>?
        get() = getOrCreateSingleInstance(
            ::endFlowUseCaseReference,
            { endFlowUseCaseReference = it },
            ::createEndFlowUseCase
        )

    private fun createAddAttributeUseCase(): UseCase<TimeCaptureBoundModel<AppFlowAttribute>, Unit>? =
        handler?.let { appFlowHandler ->
            flowAttributeConfigurationValidator?.let { configurationValidator ->
                configurationProvider?.let { configurations ->
                    SetFlowAttributeUseCase(
                        handler = appFlowHandler,
                        logger = ServiceLocator.getApmLogger(),
                        configurationsValidator = configurationValidator,
                        attributeValidator = flowAttributeValidator,
                        attributeSanitizer = flowAttributeSanitizer,
                        appFlowConfigurations = configurations,
                        refreshBackgroundFlowUseCase = getRefreshBackgroundUseCase(appFlowHandler)
                    )
                }
            }
        }

    internal val addAttributeUseCase: UseCase<TimeCaptureBoundModel<AppFlowAttribute>, Unit>?
        get() = getOrCreateSingleInstance(
            ::addAttributeUseCaseReference,
            { addAttributeUseCaseReference = it },
            ::createAddAttributeUseCase
        )

    private fun createDisabledUseCase(): UseCase<Unit, Unit>? =
        handler?.let(::DisabledUseCase)

    internal val disableUseCase: UseCase<Unit, Unit>?
        get() = getOrCreateSingleInstance(
            ::disabledUseCaseReference,
            { disabledUseCaseReference = it },
            ::createDisabledUseCase
        )

    private fun createNewSessionUseCase(): UseCase<Pair<Session, Session?>, Unit>? =
        handler?.let { appFlowHandler ->
            configurationProvider?.let { configurations ->
                AppFlowNewSessionUseCase(
                    appFlowHandler,
                    configurations,
                    appFlowApmSessionReadinessHandler,
                    ServiceLocator.getAPMV3SessionSyncNotifier()
                )
            }
        }

    internal val newSessionUseCase: UseCase<Pair<Session, Session?>, Unit>?
        get() = getOrCreateSingleInstance(
            ::newSessionUseCaseReference,
            { newSessionUseCaseReference = it },
            ::createNewSessionUseCase
        )

    private val sessionObserver: APMSessionObserver
        get() = AppFlowSessionObserver(
            AppFlowExecutorProvider(),
            AppFlowNewSessionUseCaseProvider()
        )

    private fun createAppFlowManager(): AppFlowManagerImpl? =
        configurationProvider?.let { configurations ->
            AppFlowManagerImpl(
                startAppFlowUseCaseProvider = StartFlowUseCaseProvider(),
                endAppFlowUseCaseProvider = EndFlowUseCaseProvider(),
                setFlowAttributeUseCaseProvider = SetFlowAttributeUseCaseProvider(),
                appFlowDisabledUseCaseProvider = DisableUseCaseProvider(),
                appFlowAppLaunchUseCaseProvider = AppFlowAppLaunchUseCaseProvider(),
                appFlowConfigurationProvider = configurations,
                sessionObserver = sessionObserver,
                appFlowAppStateEventListenerProvider = AppFlowAppStateEventListenerProvider(),
                appStateEventDispatcher = ServiceLocator.getAppStateEventDispatcher(),
                executorProvider = AppFlowExecutorProvider(),
                appFlowApmSessionReadinessHandlerProvider = AppFlowApmSessionReadinessHandlerProvider(),
                apmSessionLazyDataProvider = ServiceLocator.getAPMSessionLazyDataProvider()
            )
        }

    private val managerImpl: AppFlowManagerImpl?
        get() = getOrCreateSingleton(
            ::_manager,
            { _manager = it },
            ::createAppFlowManager
        )

    val manager: AppFlowManager?
        get() = managerImpl

    val sessionReadinessManager: APMSessionReadinessManager?
        get() = managerImpl

    internal val appFlowBackgroundUseCase: UseCase<Unit, Unit>?
        get() = getOrCreateSingleInstance(
            ::backgroundUseCaseReference,
            { backgroundUseCaseReference = it },
            ::createBackGroundUseCase
        )

    private fun createBackGroundUseCase(): UseCase<Unit, Unit>? = handler?.let { appFlowHandler ->
        configurationProvider?.let { appFlowConfigurationProvider ->
            AppFlowBackgroundUseCase(appFlowHandler, appFlowConfigurationProvider)
        }
    }

    internal val appFlowForegroundUseCase: UseCase<Pair<ForegroundAppStateEvent, BackgroundAppStateEvent?>, Unit>?
        get() = getOrCreateSingleInstance(
            ::foregroundUseCaseReference,
            { foregroundUseCaseReference = it },
            ::createForegroundUseCase
        )

    private fun createForegroundUseCase(): UseCase<Pair<ForegroundAppStateEvent, BackgroundAppStateEvent?>, Unit>? =
        handler?.let { appFlowHandler ->
            configurationProvider?.let { appFlowConfigs ->
                AppFlowForegroundUseCase(appFlowHandler, appFlowConfigs)
            }
        }

    internal val appFlowAppStateEventListener: AppStateEventListener
        get() = AppFlowAppStateEventListener(
            foregroundUseCaseProvider = AppFlowForegroundUseCaseProvider(),
            backgroundUseCaseProvider = AppFlowBackgroundUseCaseProvider(),
            executor = appFlowExecutor
        )

    private fun getRefreshBackgroundUseCase(handler: AppFlowHandler): UseCase<Long, Boolean?> =
        getOrCreateSingleInstance(
            ::refreshBackgroundFlowUseCaseReference,
            { refreshBackgroundFlowUseCaseReference = it },
            { createRefreshBackgroundFlowUseCase(handler) }
        )

    private fun createRefreshBackgroundFlowUseCase(handler: AppFlowHandler): UseCase<Long, Boolean?> =
        RefreshBackgroundFlowUseCase(appStateProvider, handler)

    private val appFlowJsonMapper: Mapper<List<AppFlowCacheModel>?, JSONArray?>
        get() = AppFlowJsonMapper()

    val sessionModelFiller: SessionModelFiller?
        get() = configurationProvider?.let { appFlowConfigs ->
            handler?.let { appFlowHandler ->
                AppFlowSessionModelFiller(appFlowHandler, appFlowConfigs)
            }
        }


    internal val appFlowApmSessionReadinessHandler
        get() = getOrCreateSingleInstance(
            ::appFlowApmSessionReadinessHandlerReference,
            { appFlowApmSessionReadinessHandlerReference = it },
            ::createAppFlowApmSessionReadinessHandler
        )

    private fun createAppFlowApmSessionReadinessHandler(): APMSessionReadinessHandler? =
        handler?.let { AppFlowSessionReadinessHandler(appFlowExecutor, it) }

    val appFlowAndExecutionTracesSessionJsonFiller: SessionFeatureJsonFiller
        get() = SessionAppFlowAndExecutionTraceJsonFiller(
            appFlowJsonMapper,
            ServiceLocator.getExecutionTracesMapper()
        )

    val appFLowAppLaunchUseCase: UseCase<Unit, Unit>?
        get() = getOrCreateSingleInstance(
            ::appFLowAppLaunchUseCaseReference,
            { appFLowAppLaunchUseCaseReference = it },
            ::createAppFlowAppLaunchUseCase
        )

    private fun createAppFlowAppLaunchUseCase(): UseCase<Unit, Unit>? =
        handler?.let(::AppFLowAppLaunchUseCase)
}