package com.instabug.apm.uitrace.handler

import android.app.Activity
import android.os.Build
import android.os.Bundle
import androidx.annotation.RequiresApi
import com.instabug.apm.configuration.APMConfigurationProvider
import com.instabug.apm.handler.session.APMSessionObserver
import com.instabug.apm.handler.session.SessionObserverHandler
import com.instabug.apm.logger.internal.Logger
import com.instabug.apm.model.EventTimeMetricCapture
import com.instabug.apm.uitrace.activitycallbacks.APMUiTraceActivityCallbacks
import com.instabug.apm.uitrace.model.UiTraceInitParams
import com.instabug.apm.uitrace.repo.UiTracesRepo
import com.instabug.apm.uitrace.util.getUiTraceEndParams
import com.instabug.apm.uitrace.util.logAutomaticUiTraceSuccessfullyEnded
import com.instabug.apm.uitrace.util.logAutomaticUiTraceSuccessfullyStarted
import com.instabug.apm.uitrace.util.screenName
import com.instabug.apm.util.device.APMDeviceStateProvider
import com.instabug.apm.util.runOrReportAPMError
import com.instabug.apm.util.view.InstabugViews
import com.instabug.library.model.common.Session
import java.util.concurrent.Executor

@RequiresApi(Build.VERSION_CODES.JELLY_BEAN)
class NativeAutomaticUiTraceHandler(
    private val executor: Executor,
    private val repo: UiTracesRepo,
    private val configurationProvider: APMConfigurationProvider,
    private val logger: Logger,
    private val deviceStateProvider: APMDeviceStateProvider
) : APMUiTraceActivityCallbacks, APMSessionObserver {

    private val Activity.isValidToHandleUiTrace: Boolean
        get() = !InstabugViews.isInstabugActivity(this) && configurationProvider.isUiTraceEnabled
    private val Activity.runtimeTraceId
        get() = "$screenName-${hashCode()}"

    override fun onActivityPreCreated(
        activity: Activity,
        savedInstanceState: Bundle?,
        timeMetric: EventTimeMetricCapture,
        uiTraceId: Long
    ) = handleUiTraceOperationIfPossible(activity, "onActivityPreCreated") {
        repo.create(activity.runtimeTraceId, uiTraceId)
    }

    override fun onActivityCreated(
        activity: Activity,
        savedInstanceState: Bundle?,
        timeMetric: EventTimeMetricCapture,
        uiTraceId: Long
    ) = handleUiTraceOperationIfPossible(activity, "onActivityCreated") {
        repo.getOrCreateInitialUiTrace(activity.runtimeTraceId, uiTraceId)
    }

    override fun onActivityResumed(
        activity: Activity,
        timeMetric: EventTimeMetricCapture,
        uiTraceId: Long
    ) = handleUiTraceOperationIfPossible(activity, "onActivityResumed") {
        repo.start(activity.runtimeTraceId, getUiTraceInitParams(uiTraceId, timeMetric, activity))
        logger.logAutomaticUiTraceSuccessfullyStarted(activity.screenName)
    }

    private fun getUiTraceInitParams(
        uiTraceId: Long,
        timeMetric: EventTimeMetricCapture,
        activity: Activity
    ) = UiTraceInitParams(
        uiTraceId,
        timeMetric.getTimeStampMicro(),
        timeMetric.getMicroTime(),
        deviceStateProvider.getBatteryLevel(activity),
        deviceStateProvider.isPowerSaveModeEnabled(activity),
        deviceStateProvider.getScreenOrientation(activity),
        activity.screenName,
        activity.title?.toString().orEmpty(),
        false
    )

    override fun onActivityPaused(
        activity: Activity,
        timeMetric: EventTimeMetricCapture
    ) = handleUiTraceOperationIfPossible(activity, "onActivityPaused") {
        repo.end(
            activity.runtimeTraceId,
            activity.getUiTraceEndParams(deviceStateProvider, timeMetric.getMicroTime())
        )?.also { logger.logAutomaticUiTraceSuccessfullyEnded(activity.screenName, it) }
    }

    override fun onNewSessionStarted(
        runningSession: Session,
        lastSession: Session?
    ) = executor.execute {
        runOrReportAPMError("error while handling native ui traces on new session start", logger) {
            if (configurationProvider.isUiTraceEnabled)
                repo.setSessionIdAndSaveIfPossible(runningSession.id)
        }
    }

    override fun observeAPMSessions() =
        runOrReportAPMError(
            "error while registering native ui trace handler as SessionObserver",
            logger
        ) { SessionObserverHandler.register(this) }

    override fun stopObservingAPMSessions() =
        runOrReportAPMError(
            "error while unregistering native ui trace handler as SessionObserver",
            logger
        ) { SessionObserverHandler.unregister(this) }

    private inline fun handleUiTraceOperationIfPossible(
        activity: Activity,
        callback: String,
        crossinline operation: () -> Unit
    ) = executor.execute {
        runOrReportAPMError(
            "error while handling ui trace for ${activity.screenName} at $callback",
            logger
        ) { if (activity.isValidToHandleUiTrace) operation() }
    }
}