package com.instabug.apm.networkinterception.external_network_trace

import com.instabug.apm.configuration.APMConfigurationProvider
import com.instabug.apm.di.Provider
import com.instabug.library.factory.ParameterizedFactory

data class W3CExternalNetworkTraceIdInfo(
    val isCaptured: Boolean,
    val pid: Long? = null,
    val timestampSeconds: Long? = null,
    val fullyGeneratedId: String? = null,
    val syncableGeneratedTraceId: String? = null,
    val syncableCapturedTraceId: String? = null,
    val headers: List<Pair<String, String>>? = null
)

class W3CExternalNetworkTraceIdFactory(
    private val configProvider: APMConfigurationProvider,
    private val formatProvider: Provider<W3CFormatDetails>
) : ParameterizedFactory<W3CExternalNetworkTraceIdInfo?, String?> {
    override fun create(type: String?): W3CExternalNetworkTraceIdInfo? {
        if (!configProvider.isW3CNetworkExternalTraceIdEnabled) return null
        return type?.let {
            W3CExternalNetworkTraceIdInfo(
                isCaptured = true,
                syncableCapturedTraceId = it.takeIf { configProvider.isAttachingCapturedW3CExternalTraceIdEnabled }
            )
        } ?: generateTraceId()
    }

    private fun generateTraceId(): W3CExternalNetworkTraceIdInfo {
        val generatedW3CFormat = formatProvider()
        return W3CExternalNetworkTraceIdInfo(
            isCaptured = false,
            pid = generatedW3CFormat.pid,
            timestampSeconds = generatedW3CFormat.timestampSeconds,
            fullyGeneratedId = generatedW3CFormat.traceFormat,
            syncableGeneratedTraceId = generatedW3CFormat.traceFormat
                .takeIf { configProvider.isAttachingGeneratedW3CExternalTraceIdEnabled },
            headers = listOf(
                Pair(TraceParentHeaderKey, generatedW3CFormat.traceFormat)
            )
        )
    }

    companion object {
        const val TraceParentHeaderKey = "traceparent"
    }
}