package com.instabug.apm.lifecycle

import android.app.Activity
import com.instabug.apm.cache.model.AppLaunchCacheModel
import com.instabug.apm.constants.AppLaunchType
import com.instabug.apm.constants.Constants
import com.instabug.apm.constants.ErrorMessages
import com.instabug.apm.di.ServiceLocator
import com.instabug.apm.model.AppLaunchStage
import com.instabug.apm.model.AppLaunchStageDetails
import com.instabug.apm.model.EventTimeMetricCapture
import com.instabug.library.core.InstabugCore
import com.instabug.library.model.common.Session
import com.instabug.library.util.threading.PoolProvider

interface AppLaunchLifeCycleCallbacks {

    fun onActivityCreated(activity: Activity, timeMetricCapture: EventTimeMetricCapture)

    fun onActivityStarted(activity: Activity, timeMetricCapture: EventTimeMetricCapture)

    fun onActivityResumed(activity: Activity, timeMetricCapture: EventTimeMetricCapture)

    fun onActivityStopped()

    fun endAppLaunch()

    fun onNewSessionStarted(session: Session)
}

class AppLaunchLifeCycleCallbacksImpl (
    appStartedInBackground: () -> Boolean,
    private val isRegisteredBeforeFirstActivityLaunch: Boolean,
    private val appLaunchModelFactory: AppLaunchModelFactory
) : AppLaunchLifeCycleCallbacks {

    companion object {
        const val ORDERED_TASK_KEY = "CAPTURE_APP_LAUNCH"
    }

    private val appLaunchDataRepository = ServiceLocator.getAppLaunchDataRepository()
    private val apmConfigurationProvider by lazy { ServiceLocator.getApmConfigurationProvider() }
    private val apmLogger by lazy { ServiceLocator.getApmLogger() }

    init {
        appLaunchDataRepository.isFirstColdLaunch = !appStartedInBackground.invoke()
    }

    override fun onActivityCreated(activity: Activity, timeMetricCapture: EventTimeMetricCapture) {
        val startedActivitiesCount = getStartedActivitiesCount()
        PoolProvider.postOrderedIOTask(ORDERED_TASK_KEY) {
            if (startedActivitiesCount == 0) {
                appLaunchDataRepository.appLaunchStages[AppLaunchStage.APP_CREATION]?.apply {
                    stageEndTimeMicro = timeMetricCapture.getMicroTime()
                }
                appLaunchDataRepository.appLaunchStages[AppLaunchStage.ACTIVITY_CREATION] =
                    AppLaunchStageDetails(
                        stageStartTimeStampMicro = timeMetricCapture.getTimeStampMicro(),
                        stageStartTimeMicro = timeMetricCapture.getMicroTime(),
                        stageScreenName = activity.javaClass.name
                    )
                appLaunchDataRepository.lastCapturedAppLaunchType = null
            }
            // activity wasn't created before
            appLaunchDataRepository.isHotAppLaunch = false
        }
    }

    override fun onActivityStarted(activity: Activity, timeMetricCapture: EventTimeMetricCapture) {
        val startedActivitiesCount = getStartedActivitiesCount()
        PoolProvider.postOrderedIOTask(ORDERED_TASK_KEY) {
            val isFirstStartedActivity = startedActivitiesCount == 1
            appLaunchDataRepository.isAppWasInBG = isFirstStartedActivity
            appLaunchDataRepository.isLaunchFinished =
                appLaunchDataRepository.isLaunchFinished && !isFirstStartedActivity
            if (isFirstStartedActivity) {
                appLaunchDataRepository.appLaunchStages[AppLaunchStage.ACTIVITY_CREATION]?.apply {
                    stageEndTimeMicro = timeMetricCapture.getMicroTime()
                }
                appLaunchDataRepository.appLaunchStages[AppLaunchStage.ACTIVITY_START] =
                    AppLaunchStageDetails(
                        stageStartTimeStampMicro = timeMetricCapture.getTimeStampMicro(),
                        stageStartTimeMicro = timeMetricCapture.getMicroTime(),
                        stageScreenName = activity.javaClass.name
                    )
                appLaunchDataRepository.lastCapturedAppLaunchType = null
            }
        }
    }

    override fun onActivityResumed(activity: Activity, timeMetricCapture: EventTimeMetricCapture) {
        PoolProvider.postOrderedIOTask(ORDERED_TASK_KEY) {
            with(appLaunchDataRepository) {
                val screenName = activity.javaClass.name
                if (isAppWasInBG) {
                    appLaunchStages[AppLaunchStage.ACTIVITY_START]?.apply {
                        stageEndTimeMicro = timeMetricCapture.getMicroTime()
                    }
                    activityResumeStartTimeStampMicro = timeMetricCapture.getTimeStampMicro()
                    if (isFirstColdLaunch) {
                        if (isRegisteredBeforeFirstActivityLaunch) {
                            captureAppLaunch(screenName, AppLaunchType.COLD)
                        }
                    } else if (isHotAppLaunch && !isLaunchFinished) {
                        captureAppLaunch(screenName, AppLaunchType.HOT)
                    } else if (!isHotAppLaunch && !isLaunchFinished) {
                        captureAppLaunch(screenName, AppLaunchType.WARM)
                    }
                } else if (isHotAppLaunch && !isLaunchFinished) {
                    // capturing hot app launch for DOUBLE START case where a partial activity
                    // starts with another bg activity.
                    // Ex.: (App comes to FG where Instabug Invocation Prompt on top of the app main activity)
                    appLaunchStages[AppLaunchStage.ACTIVITY_START]?.stageEndTimeMicro =
                        timeMetricCapture.getMicroTime()
                    activityResumeStartTimeStampMicro =
                        timeMetricCapture.getTimeStampMicro()
                    captureAppLaunch(screenName, AppLaunchType.HOT)
                }
                isEndAppLaunchCalledEarly = false
                isFirstColdLaunch = false
                isHotAppLaunch = true
                isLaunchFinished = true
            }
        }
    }

    override fun onActivityStopped() {
        val startedActivitiesCount = getStartedActivitiesCount()
        PoolProvider.postOrderedIOTask(ORDERED_TASK_KEY) {
            if (startedActivitiesCount == 0) {
                appLaunchDataRepository.isFirstColdLaunch = false
            }
        }
    }

    override fun endAppLaunch() {
        PoolProvider.postOrderedIOTask(ORDERED_TASK_KEY) {
            val appLaunchType = appLaunchDataRepository.lastCapturedAppLaunchType
            if (appLaunchType == null) {
                apmLogger.logSDKError(ErrorMessages.END_APP_LAUNCH_CALLED_TOO_EARLY)
                appLaunchDataRepository.isEndAppLaunchCalledEarly = true
            }
            else if (!apmConfigurationProvider.isAppLaunchSdkEnabled(appLaunchType)) {
                apmLogger.logSDKError(
                    ErrorMessages.END_APP_LAUNCH_NOT_CALLED_APP_LAUNCH_TYPE_SDK_DISABLED
                        .replace("%s", appLaunchType)
                )
            } else if (!apmConfigurationProvider.isAppLaunchFeatureEnabled(appLaunchType)) {
                apmLogger.logSDKError(
                    ErrorMessages.END_APP_LAUNCH_NOT_CALLED_APP_LAUNCH_TYPE_FEATURE_DISABLED
                        .replace("%s", appLaunchType)
                )
            } else if (!apmConfigurationProvider.isEndAppLaunchFeatureEnabled(appLaunchType)) {
                apmLogger.logSDKError(ErrorMessages.END_APP_LAUNCH_NOT_CALLED_END_API_DISABLED)
            } else {
                validateAndUpdateEndAppLaunch(
                    appLaunchDataRepository.activityResumeStartTimeStampMicro,
                    appLaunchDataRepository.endAppLaunchStageDuration
                )
            }
        }
    }

    override fun onNewSessionStarted(session: Session) {
        PoolProvider.postOrderedIOTask(ORDERED_TASK_KEY) {
            appLaunchDataRepository.appLaunchCacheModel?.let {
                persistAppLaunchModel(session.id, it)
            }
        }
    }

    /**
     * Validates and updated App launch model with END_APP_LAUNCH stage
     * @return true if the model was changed and false otherwise
     */
    private fun validateAndUpdateEndAppLaunchModel(
        appLaunchModel: AppLaunchCacheModel,
        endAppLaunchStartTimeStampMicro: Long,
        endAppLaunchDuration: Long
    ): Boolean = appLaunchModel.type?.let { _ ->
            if (appLaunchModel.isEndAppLaunchRecorded) {
                apmLogger.logSDKWarning(ErrorMessages.END_APP_LAUNCH_NOT_CALLED_FOR_MULTIPLE_TIMES)
                false
            } else {
                if (endAppLaunchDuration < 0) {
                    apmLogger.logSDKError(ErrorMessages.END_APP_LAUNCH_CALLED_TOO_EARLY)
                    updateAppLaunchModel(
                        appLaunchModel,
                        0,
                        0
                    )
                    appLaunchDataRepository.isEndAppLaunchCalledEarly = true
                    true
                } else {
                    updateAppLaunchModel(
                        appLaunchModel,
                        endAppLaunchStartTimeStampMicro,
                        endAppLaunchDuration
                    )
                    true
                }
            }
        } ?: false

    private fun validateAndUpdateEndAppLaunch(
        endAppLaunchStartTimeStampMicro: Long,
        endAppLaunchDuration: Long
    ) {
        with(appLaunchDataRepository) {
            appLaunchCacheModel?.also {
                validateAndUpdateEndAppLaunchModel(
                    it,
                    endAppLaunchStartTimeStampMicro,
                    endAppLaunchDuration
                )
            } ?: getCurrentSessionId()?.takeIf { it.isNotEmpty() }
                ?.let { sessionId ->
                    ServiceLocator.getAppLaunchesHandler()?.getAppLaunchesForSession(sessionId)
                }?.takeIf { cachedAppLaunches ->
                    cachedAppLaunches.size == 1
                }?.firstOrNull()?.also {
                    val modelUpdated = validateAndUpdateEndAppLaunchModel(
                        it,
                        endAppLaunchStartTimeStampMicro,
                        endAppLaunchDuration
                    )
                    if (modelUpdated) {
                        ServiceLocator.getAppLaunchesHandler().updateAppLaunch(it)
                    }
                } ?: let {
                apmLogger.logSDKError(ErrorMessages.END_APP_LAUNCH_CALLED_TOO_EARLY)
                isEndAppLaunchCalledEarly = true
            }
        }
    }

    private fun updateAppLaunchModel(
        appLaunch: AppLaunchCacheModel,
        endAppLaunchStartTimeStampMicro: Long,
        endAppLaunchDuration: Long
    ) {
        // 1. Update the screen name
        appLaunch.screenName =
            appLaunchDataRepository.getScreenName(appLaunchDataRepository.lastCapturedAppLaunchType)
        // 2. Update the duration
        appLaunch.duration = appLaunch.duration + endAppLaunchDuration
        val stagesMap = appLaunch.stages
        if (stagesMap != null) {
            // 3. Add the new stage
            stagesMap[Constants.AppLaunch.END_APP_LAUNCH_DURATION] =
                endAppLaunchDuration.toString()
            if (endAppLaunchStartTimeStampMicro != 0L) {
                stagesMap[Constants.AppLaunch.END_APP_LAUNCH_TIME_STAMP] =
                    endAppLaunchStartTimeStampMicro.toString()
            }
            appLaunch.stages = stagesMap
        }
    }

    private fun captureAppLaunch(screenName: String, @AppLaunchType type: String) {
        appLaunchDataRepository.lastCapturedAppLaunchType = type
        if (apmConfigurationProvider.isAppLaunchEnabled(type)) {
            appLaunchModelFactory.createAppLaunchModelIfPossible(
                screenName,
                type,
                appLaunchDataRepository
            )?.let { appLaunchModel ->
                appLaunchDataRepository.appLaunchCacheModel = appLaunchModel
                getCurrentSessionId()?.let {
                        persistAppLaunchModel(it, appLaunchModel)
                    }
            }
        }
    }

    private fun persistAppLaunchModel(sessionId: String, appLaunch: AppLaunchCacheModel) {
        val handler = ServiceLocator.getAppLaunchesHandler()
        handler.insertAppLaunch(sessionId, appLaunch)
        clearAppLaunchCacheModel()
    }

    private fun clearAppLaunchCacheModel() {
        appLaunchDataRepository.appLaunchCacheModel = null
    }

    private fun getStartedActivitiesCount() = InstabugCore.getStartedActivitiesCount()

    private fun getCurrentSessionId() = ServiceLocator.getSessionHandler().currentSession?.id
}