package com.instabug.apm.handler.executiontraces;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.WorkerThread;

import com.instabug.apm.cache.handler.executiontraces.DanglingExecutionTracesCacheHandler;
import com.instabug.apm.cache.handler.executiontraces.ExecutionTracesCacheHandler;
import com.instabug.apm.cache.handler.session.SessionMetaDataCacheHandler;
import com.instabug.apm.cache.model.ExecutionTraceCacheModel;
import com.instabug.apm.configuration.APMConfigurationProvider;
import com.instabug.apm.di.ServiceLocator;
import com.instabug.apm.handler.session.SessionHandler;
import com.instabug.apm.logger.internal.Logger;
import com.instabug.library.model.common.Session;

import java.util.List;

@WorkerThread
public class ExecutionTracesHandlerImpl implements ExecutionTracesHandler {
    private ExecutionTracesCacheHandler executionTracesCacheHandler = ServiceLocator.getExecutionTracesCacheHandler();
    private DanglingExecutionTracesCacheHandler danglingExecutionTracesCacheHandler = ServiceLocator.getDanglingExecutionTracesCacheHandler();
    private Logger apmLogger = ServiceLocator.getApmLogger();
    private SessionHandler sessionHandler = ServiceLocator.getSessionHandler();
    @Nullable
    private SessionMetaDataCacheHandler sessionMetaDataCacheHandler = ServiceLocator.getSessionMetaDataCacheHandler();
    private APMConfigurationProvider apmConfigurationProvider = ServiceLocator.getApmConfigurationProvider();

    private int trimTracesToLimit(String sessionID, long limit) {
        return executionTracesCacheHandler.trimTracesToLimit(sessionID, limit);
    }

    @Override
    public void setAttribute(long traceId, @NonNull String traceName, @NonNull String key, @Nullable String value) {
        if (executionTracesCacheHandler.exists(traceId)) {
            executionTracesCacheHandler.setAttribute(traceId, traceName, key, value);
        }
        if (danglingExecutionTracesCacheHandler.exists(traceId)) {
            danglingExecutionTracesCacheHandler.setAttribute(traceId, traceName, key, value);
        }
    }

    @Override
    public void removeAll() {
        executionTracesCacheHandler.removeAll();
        danglingExecutionTracesCacheHandler.removeAll();
        if (sessionMetaDataCacheHandler != null) {
            sessionMetaDataCacheHandler.resetTracesCounts();
        }
    }

    @Override
    public void removeUnEndedTraces() {
        executionTracesCacheHandler.removeUnEndedTraces();
        danglingExecutionTracesCacheHandler.removeUnEndedTraces();
    }


    @Nullable
    @Override
    public List<ExecutionTraceCacheModel> getExecutionTracesForSession(String sessionID) {
        return executionTracesCacheHandler.getExecutionTracesForSession(sessionID);
    }

    @Override
    public void trimTracesToLimit(long limit) {
        executionTracesCacheHandler.trimTracesToLimit(limit);
    }

    @Override
    public boolean insertTrace(ExecutionTraceCacheModel cacheModel) {
        Session currentSession = sessionHandler.getCurrentSession();
        if (currentSession != null) {
            boolean isSuccessful = executionTracesCacheHandler.insertTrace(currentSession.getId(), cacheModel);
            if (isSuccessful) {
                if (sessionMetaDataCacheHandler != null) {
                    sessionMetaDataCacheHandler.addToTracesTotalCount(currentSession.getId(), 1);
                    int deletedTracesCount = trimTracesToLimit(currentSession.getId(),
                            apmConfigurationProvider.getExecutionTraceLimitPerRequest());
                    if (deletedTracesCount > 0) {
                        sessionMetaDataCacheHandler.addToTracesDroppedCount(currentSession.getId(), deletedTracesCount);
                    }
                }
                trimTracesToLimit(apmConfigurationProvider.getExecutionTraceStoreLimit());
            } else {
                apmLogger.logSDKProtected("Session meta data was not updated. Failed to insert custom trace " + cacheModel.getName());
            }
            return isSuccessful;
        } else {
            return danglingExecutionTracesCacheHandler.insertTrace(cacheModel);
        }
    }
}
