package com.instabug.apm.fragment

import androidx.annotation.VisibleForTesting
import com.instabug.apm.configuration.APMConfigurationProvider
import com.instabug.apm.di.Provider
import com.instabug.apm.handler.fragment.FragmentSpansHandler
import com.instabug.apm.logger.internal.Logger
import com.instabug.library.diagnostics.IBGDiagnostics
import java.util.concurrent.Executor

interface FragmentSpansHelper {
    fun startFragmentsLifecycleCapturing()
    fun stopFragmentsLifecycleCapturing()
    fun onFeatureDisabled()
    fun onNewSessionStarted(sessionId: String)
}

class FragmentSpansHelperImpl(
    private val apmConfigurationProvider: APMConfigurationProvider,
    private val fragmentsLifecycleListener: FragmentLifecycleEventListener,
    private val fragmentSpansHandlerProvider: Provider<FragmentSpansHandler?>,
    private val executor: Executor,
    private val logger: Logger
) : FragmentSpansHelper {

    override fun startFragmentsLifecycleCapturing() = executor.execute {
        runOrLogError("starting fragments capturing feature") {
            if (canStartFragmentsCapturing()) {
                FragmentEventDispatcher.addListener(fragmentsLifecycleListener)
            }
        }
    }

    override fun stopFragmentsLifecycleCapturing() = executor.execute {
        runOrLogError("stopping fragments capturing feature") {
            stopSynchronous()
        }
    }

    private fun stopSynchronous() {
        fragmentsLifecycleListener.cleanup()
        FragmentEventDispatcher.removeListener(fragmentsLifecycleListener)
    }

    override fun onFeatureDisabled() = executor.execute {
        runOrLogError("handling fragments capturing feature feature disabled") {
            stopSynchronous()
            fragmentSpansHandlerProvider()?.clearCache()
        }
    }

    override fun onNewSessionStarted(sessionId: String) = executor.execute {
        runOrLogError("handling fragment capturing on new session started") {
            if (canStartFragmentsCapturing()) fragmentSpansHandlerProvider()?.onNewSession(sessionId)
        }
    }

    @VisibleForTesting
    fun canStartFragmentsCapturing() =
        apmConfigurationProvider.isAPMEnabled && apmConfigurationProvider.isFragmentSpansEnabled

    private inline fun runOrLogError(message: String, action: () -> Unit) = runCatching {
        action()
    }.onFailure {
        val errorMessage = "Error occurred while $message: ${it.message}"
        IBGDiagnostics.reportNonFatal(it, errorMessage)
        logger.logSDKProtected(errorMessage)
    }
}