package com.instabug.apm.configuration;

import static com.instabug.apm.constants.APMConfigurationKeys.ExecutionTraces;
import static com.instabug.apm.constants.APMConfigurationKeys.Experiments.KEY_EXPERIMENTS_ENABLED;
import static com.instabug.apm.constants.APMConfigurationKeys.Experiments.KEY_EXPERIMENTS_LIMIT_PER_REQUEST;
import static com.instabug.apm.constants.APMConfigurationKeys.KEY_CRASH_DETECTION_ENABLED;
import static com.instabug.apm.constants.APMConfigurationKeys.KEY_DEBUG_MODE_ENABLED;
import static com.instabug.apm.constants.APMConfigurationKeys.KEY_DROPPED_SESSION_COUNT_STORE_LIMIT;
import static com.instabug.apm.constants.APMConfigurationKeys.KEY_IS_APM_FEATURE_AVAILABLE;
import static com.instabug.apm.constants.APMConfigurationKeys.KEY_IS_APM_SDK_ENABLED;
import static com.instabug.apm.constants.APMConfigurationKeys.KEY_LAST_SYNC_TIME;
import static com.instabug.apm.constants.APMConfigurationKeys.KEY_SHOULD_SEND_LEGACY_SESSIONS;
import static com.instabug.apm.constants.APMConfigurationKeys.KEY_SYNC_INTERVAL;
import static com.instabug.apm.constants.APMConfigurationKeys.Launches.KEY_COLD_LAUNCHES_PER_REQUEST_LIMIT;
import static com.instabug.apm.constants.APMConfigurationKeys.Launches.KEY_COLD_LAUNCHES_SDK_ENABLED;
import static com.instabug.apm.constants.APMConfigurationKeys.Launches.KEY_COLD_LAUNCHES_STORE_LIMIT;
import static com.instabug.apm.constants.APMConfigurationKeys.Launches.KEY_END_COLD_APP_LAUNCH_ENABLED;
import static com.instabug.apm.constants.APMConfigurationKeys.Launches.KEY_END_HOT_APP_LAUNCH_ENABLED;
import static com.instabug.apm.constants.APMConfigurationKeys.Launches.KEY_END_WARM_APP_LAUNCH_ENABLED;
import static com.instabug.apm.constants.APMConfigurationKeys.Launches.KEY_HOT_LAUNCHES_ENABLED;
import static com.instabug.apm.constants.APMConfigurationKeys.Launches.KEY_HOT_LAUNCHES_PER_REQUEST_LIMIT;
import static com.instabug.apm.constants.APMConfigurationKeys.Launches.KEY_HOT_LAUNCHES_SDK_ENABLED;
import static com.instabug.apm.constants.APMConfigurationKeys.Launches.KEY_HOT_LAUNCHES_STORE_LIMIT;
import static com.instabug.apm.constants.APMConfigurationKeys.Launches.KEY_LAUNCHES_ENABLED;
import static com.instabug.apm.constants.APMConfigurationKeys.Launches.KEY_WARM_APP_LAUNCH_ENABLED;
import static com.instabug.apm.constants.APMConfigurationKeys.Launches.KEY_WARM_APP_LAUNCH_PER_REQUEST_LIMIT;
import static com.instabug.apm.constants.APMConfigurationKeys.Launches.KEY_WARM_APP_LAUNCH_SDK_ENABLED;
import static com.instabug.apm.constants.APMConfigurationKeys.Launches.KEY_WARM_APP_LAUNCH_STORE_LIMIT;
import static com.instabug.apm.constants.APMConfigurationKeys.Network.KEY_NETWORK_ENABLED;
import static com.instabug.apm.constants.APMConfigurationKeys.Network.KEY_NETWORK_GRAPH_QL_ENABLED;
import static com.instabug.apm.constants.APMConfigurationKeys.Network.KEY_NETWORK_GRPC_ENABLED;
import static com.instabug.apm.constants.APMConfigurationKeys.Network.KEY_NETWORK_LOGS_CACHE_LIMIT;
import static com.instabug.apm.constants.APMConfigurationKeys.Network.KEY_NETWORK_LOGS_LAST_SYNC_TIME;
import static com.instabug.apm.constants.APMConfigurationKeys.Network.KEY_NETWORK_LOGS_REQUEST_LIMIT;
import static com.instabug.apm.constants.APMConfigurationKeys.RateLimitation.APM_SESSIONS_LIMITED_UNTIL;
import static com.instabug.apm.constants.APMConfigurationKeys.RateLimitation.LAST_APM_SESSIONS_REQUEST_STARTED_AT;
import static com.instabug.apm.constants.APMConfigurationKeys.UiHangs.KEY_UI_HANG_ENABLED;
import static com.instabug.apm.constants.APMConfigurationKeys.UiHangs.KEY_UI_HANG_SDK_ENABLED;
import static com.instabug.apm.constants.APMConfigurationKeys.UiLoadingMetrics.KEY_END_SCREEN_LOADING_ENABLED;
import static com.instabug.apm.constants.APMConfigurationKeys.UiLoadingMetrics.KEY_UI_LOADING_ENABLED;
import static com.instabug.apm.constants.APMConfigurationKeys.UiLoadingMetrics.KEY_UI_LOADING_SDK_ENABLED;
import static com.instabug.apm.constants.APMConfigurationKeys.UiTraces.KEY_UI_TRACE_LARGE_DROP_THRESHOLD;
import static com.instabug.apm.constants.APMConfigurationKeys.UiTraces.KEY_UI_TRACE_LIMIT_PER_REQUEST;
import static com.instabug.apm.constants.APMConfigurationKeys.UiTraces.KEY_UI_TRACE_SDK_ENABLED;
import static com.instabug.apm.constants.APMConfigurationKeys.UiTraces.KEY_UI_TRACE_SMALL_DROP_THRESHOLD;
import static com.instabug.apm.constants.APMConfigurationKeys.UiTraces.KEY_UI_TRACE_STORE_LIMIT;

import android.annotation.SuppressLint;
import android.content.SharedPreferences;

import androidx.annotation.Nullable;

import com.instabug.apm.constants.APMConfigurationKeys;
import com.instabug.apm.constants.AppLaunchType;
import com.instabug.apm.constants.DefaultValues;
import com.instabug.apm.di.ServiceLocator;
import com.instabug.apm.model.LogLevel;
import com.instabug.library.Feature;
import com.instabug.library.IBGFeature;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.networkinterception.config.IBGNetworkInterceptionConfigurationProvider;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.LimitConstraintApplier;

public class APMConfigurationProviderImpl implements APMConfigurationProvider, APMStateProvider {

    @Nullable
    private SharedPreferences sharedPreferences = ServiceLocator.getSharedPreferences();
    @Nullable
    private SharedPreferences.Editor editor;
    private APMPerSessionConfigurationProvider apmPerSessionConfigurationProvider = ServiceLocator.getApmPerSessionConfigurationProvider();
    private LimitConstraintApplier limitConstraintApplier;

    @Nullable
    private Boolean isUiLoadingMetricFeatureEnabled = null;
    @Nullable
    private Boolean isUiHangsFeatureEnabled = null;
    @Nullable
    private Boolean isAPMFeatureAvailable = null;
    @Nullable
    private volatile Boolean isNetworkFeatureEnabled = null;

    private final IBGNetworkInterceptionConfigurationProvider coreNetworkInterceptionConfigurationProvider;


    @SuppressLint("CommitPrefEdits")
    public APMConfigurationProviderImpl(
            LimitConstraintApplier limitConstraintApplier,
            IBGNetworkInterceptionConfigurationProvider coreNetworkInterceptionConfigurationProvider
    ) {
        if (sharedPreferences != null)
            editor = sharedPreferences.edit();
        this.limitConstraintApplier = limitConstraintApplier;
        this.coreNetworkInterceptionConfigurationProvider =
                coreNetworkInterceptionConfigurationProvider;
    }

    @Override
    public void setAPMFeatureAvailability(boolean enabled) {
        if (editor != null) {
            isAPMFeatureAvailable = enabled;
            editor.putBoolean(KEY_IS_APM_FEATURE_AVAILABLE, enabled).apply();
        }
    }

    @Override
    public boolean isAPMFeatureAvailable() {
        if (isAPMFeatureAvailable == null && sharedPreferences != null) {
            isAPMFeatureAvailable = sharedPreferences.getBoolean(KEY_IS_APM_FEATURE_AVAILABLE, DefaultValues.IS_APM_FEATURE_AVAILABLE);
        }
        return isAPMFeatureAvailable != null ? isAPMFeatureAvailable : DefaultValues.IS_APM_FEATURE_AVAILABLE;
    }

    @Override
    public void setAPMSdkEnabled(boolean enabled) {
        if (apmPerSessionConfigurationProvider != null) {
            apmPerSessionConfigurationProvider.putConfiguration(KEY_IS_APM_SDK_ENABLED, enabled);
        }
    }

    @Override
    public boolean isAPMSdkEnabled() {
        if (apmPerSessionConfigurationProvider != null) {
            return (boolean) apmPerSessionConfigurationProvider
                    .getConfiguration(KEY_IS_APM_SDK_ENABLED, DefaultValues.IS_APM_SDK_ENABLED);
        }
        return DefaultValues.IS_APM_SDK_ENABLED;
    }

    @Override
    public boolean isAPMEnabled() {
        return isInstabugEnabled() && isAPMSdkEnabled() && isAPMFeatureAvailable();
    }

    public boolean isInstabugEnabled() {
        return InstabugCore.isFeatureAvailable(IBGFeature.INSTABUG)
                && InstabugCore.getFeatureState(IBGFeature.INSTABUG) == Feature.State.ENABLED;
    }

    @Override
    public void setSyncInterval(long syncInterval) {
        if (editor != null)
            editor.putLong(KEY_SYNC_INTERVAL, syncInterval).apply();
    }

    @Override
    public long getSyncInterval() {
        if (sharedPreferences != null)
            return sharedPreferences.getLong(KEY_SYNC_INTERVAL, DefaultValues.SYNC_INTERVAL);
        return DefaultValues.SYNC_INTERVAL;
    }

    @Override
    public boolean isExecutionTraceFeatureEnabled() {
        if (sharedPreferences != null) {
            return sharedPreferences.getBoolean(ExecutionTraces.KEY_ENABLED, DefaultValues.Traces.ENABLED);
        }
        return DefaultValues.Traces.ENABLED;
    }

    @Override
    public void setExecutionTraceFeatureEnabled(boolean enabled) {
        if (editor != null) {
            editor.putBoolean(ExecutionTraces.KEY_ENABLED, enabled).apply();
        }
    }

    @Override
    public long getExecutionTraceStoreLimit() {
        long executionTraceStoreLimit = DefaultValues.Traces.STORE_LIMIT;
        if (sharedPreferences != null)
            executionTraceStoreLimit = sharedPreferences.getLong(ExecutionTraces.KEY_STORE_LIMIT, DefaultValues.Traces.STORE_LIMIT);
        return limitConstraintApplier.applyConstraints(executionTraceStoreLimit);
    }

    @Override
    public void setExecutionTraceStoreLimit(long limit) {
        if (editor != null)
            editor.putLong(ExecutionTraces.KEY_STORE_LIMIT, limit).apply();
    }

    @Override
    public long getExecutionTraceLimitPerRequest() {
        long executionTraceLimitPerRequest = DefaultValues.Traces.LIMIT_PER_REQUEST;
        if (sharedPreferences != null)
            executionTraceLimitPerRequest = sharedPreferences.getLong(ExecutionTraces.KEY_PER_REQUEST_LIMIT, DefaultValues.Traces.LIMIT_PER_REQUEST);
        return limitConstraintApplier.applyConstraints(executionTraceLimitPerRequest);
    }

    @Override
    public void setExecutionTraceLimitPerRequest(long limit) {
        if (editor != null)
            editor.putLong(ExecutionTraces.KEY_PER_REQUEST_LIMIT, limit).apply();
    }

    @Override
    public void setExecutionTraceStoreAttributesLimit(int customAttributesCount) {
        if (editor != null)
            editor.putInt(ExecutionTraces.KEY_ATTRIBUTES_COUNT, customAttributesCount).apply();
    }

    @Override
    public boolean isColdAppLaunchesFeatureEnabled() {
        if (sharedPreferences != null) {
            return sharedPreferences.getBoolean(KEY_LAUNCHES_ENABLED, DefaultValues.Launches.ENABLED);
        }
        return DefaultValues.Launches.ENABLED;
    }

    @Override
    public boolean isHotAppLaunchesFeatureEnabled() {
        if (sharedPreferences != null) {
            return sharedPreferences.getBoolean(KEY_HOT_LAUNCHES_ENABLED, DefaultValues.Launches.ENABLED);
        }
        return DefaultValues.Launches.ENABLED;
    }

    @Override
    public void setColdAppLaunchesFeatureEnabled(boolean enabled) {
        if (editor != null) {
            editor.putBoolean(KEY_LAUNCHES_ENABLED, enabled).apply();
        }
    }

    @Override
    public void setHotAppLaunchesFeatureEnabled(boolean enabled) {
        if (editor != null) {
            editor.putBoolean(KEY_HOT_LAUNCHES_ENABLED, enabled).apply();
        }
    }

    private long getColdAppLaunchesStoreLimit() {
        if (sharedPreferences != null) {
            return sharedPreferences.getLong(KEY_COLD_LAUNCHES_STORE_LIMIT, DefaultValues.Launches.STORE_LIMIT_COLD);
        }
        return DefaultValues.Launches.STORE_LIMIT_COLD;
    }

    @Override
    public void setColdAppLaunchesStoreLimit(long limit) {
        if (editor != null) {
            editor.putLong(KEY_COLD_LAUNCHES_STORE_LIMIT, limit).apply();
        }
    }

    private long getHotAppLaunchesStoreLimit() {
        if (sharedPreferences != null) {
            return sharedPreferences.getLong(KEY_HOT_LAUNCHES_STORE_LIMIT, DefaultValues.Launches.STORE_LIMIT_HOT);
        }
        return DefaultValues.Launches.STORE_LIMIT_HOT;
    }

    @Override
    public void setHotAppLaunchesStoreLimit(long limit) {
        if (editor != null) {
            editor.putLong(KEY_HOT_LAUNCHES_STORE_LIMIT, limit).apply();
        }
    }

    @Override
    public int getExecutionTraceStoreAttributesLimit() {
        if (sharedPreferences != null)
            return sharedPreferences.getInt(ExecutionTraces.KEY_ATTRIBUTES_COUNT, DefaultValues.Traces.ATTRIBUTES_STORE_LIMIT);
        return DefaultValues.Traces.ATTRIBUTES_STORE_LIMIT;
    }

    private long getColdAppLaunchesLimitPerRequest() {
        if (sharedPreferences != null) {
            return sharedPreferences.getLong(KEY_COLD_LAUNCHES_PER_REQUEST_LIMIT, DefaultValues.Launches.LIMIT_PER_REQUEST_COLD);
        }
        return DefaultValues.Launches.LIMIT_PER_REQUEST_COLD;
    }

    @Override
    public void setColdAppLaunchesLimitPerRequest(long limit) {
        if (editor != null) {
            editor.putLong(KEY_COLD_LAUNCHES_PER_REQUEST_LIMIT, limit).apply();
        }
    }

    private long getHotAppLaunchesLimitPerRequest() {
        if (sharedPreferences != null) {
            return sharedPreferences.getLong(KEY_HOT_LAUNCHES_PER_REQUEST_LIMIT, DefaultValues.Launches.LIMIT_PER_REQUEST_HOT);
        }
        return DefaultValues.Launches.LIMIT_PER_REQUEST_HOT;
    }

    @Override
    public void setHotAppLaunchesLimitPerRequest(long limit) {
        if (editor != null) {
            editor.putLong(KEY_HOT_LAUNCHES_PER_REQUEST_LIMIT, limit).apply();
        }
    }

    @Override
    public boolean isColdAppLaunchSDKEnabled() {
        return ((boolean) apmPerSessionConfigurationProvider
                .getConfiguration(KEY_COLD_LAUNCHES_SDK_ENABLED, DefaultValues.Launches.SDK_ENABLED));
    }

    @Override
    public boolean isHotAppLaunchSDKEnabled() {
        return ((boolean) apmPerSessionConfigurationProvider
                .getConfiguration(KEY_HOT_LAUNCHES_SDK_ENABLED, DefaultValues.Launches.SDK_ENABLED));
    }

    @Override
    public void setColdAppLaunchSDKEnabled(boolean enabled) {
        if (apmPerSessionConfigurationProvider != null) {
            apmPerSessionConfigurationProvider.putConfiguration(KEY_COLD_LAUNCHES_SDK_ENABLED, enabled);
        }
    }

    @Override
    public void setHotAppLaunchSDKEnabled(boolean enabled) {
        if (apmPerSessionConfigurationProvider != null) {
            apmPerSessionConfigurationProvider.putConfiguration(KEY_HOT_LAUNCHES_SDK_ENABLED, enabled);
        }
    }

    @Override
    public boolean isColdAppLaunchesEnabled() {
        return isColdAppLaunchFlagsEnabled() && isAPMEnabled();
    }

    private boolean isColdAppLaunchFlagsEnabled() {
        return isColdAppLaunchesFeatureEnabled() && isColdAppLaunchSDKEnabled();
    }

    @Override
    public boolean isHotAppLaunchesEnabled() {
        return isHotAppLaunchFlagsEnabled() && isAPMEnabled();
    }

    private boolean isHotAppLaunchFlagsEnabled() {
        return isHotAppLaunchesFeatureEnabled() && isHotAppLaunchSDKEnabled();
    }

    @Override
    public void setNetworkEnabled(boolean enabled) {
        if (editor != null) {
            isNetworkFeatureEnabled = enabled;
            editor.putBoolean(KEY_NETWORK_ENABLED, enabled).apply();
        }
    }

    @Override
    public boolean isNetworkFeatureEnabled() {
        Boolean localIsNetworkFeatureEnabled = isNetworkFeatureEnabled;
        if (localIsNetworkFeatureEnabled == null && sharedPreferences != null) {
            localIsNetworkFeatureEnabled = sharedPreferences.getBoolean(KEY_NETWORK_ENABLED, DefaultValues.Network.ENABLED);
            isNetworkFeatureEnabled = localIsNetworkFeatureEnabled;
        }
        return (localIsNetworkFeatureEnabled != null) ? localIsNetworkFeatureEnabled : DefaultValues.Network.ENABLED;
    }

    @Override
    public boolean isNetworkEnabled() {
        return isAPMEnabled() && isNetworkFeatureEnabled();
    }

    @Override
    public void setNetworkLogsLimitPerRequest(long limit) {
        if (editor != null) {
            editor.putLong(KEY_NETWORK_LOGS_REQUEST_LIMIT, limit).apply();
        }
    }

    @Override
    public long getNetworkLogsRequestLimit() {
        long networkLogRequestLimit = DefaultValues.Network.LIMIT_PER_REQUEST;
        if (sharedPreferences != null) {
            networkLogRequestLimit = sharedPreferences.getLong(KEY_NETWORK_LOGS_REQUEST_LIMIT,
                    DefaultValues.Network.LIMIT_PER_REQUEST);
        }
        return limitConstraintApplier.applyConstraints(networkLogRequestLimit);
    }

    @Override
    public void setNetworkLogsStoreLimit(long limit) {
        if (editor != null) {
            editor.putLong(KEY_NETWORK_LOGS_CACHE_LIMIT, limit).apply();
        }
    }

    @Override
    public long getNetworkLogsCacheLimit() {
        long networkLogCacheLimit = DefaultValues.Network.STORE_LIMIT;
        if (sharedPreferences != null) {
            networkLogCacheLimit = sharedPreferences.getLong(KEY_NETWORK_LOGS_CACHE_LIMIT,
                    DefaultValues.Network.STORE_LIMIT);
        }
        return limitConstraintApplier.applyConstraints(networkLogCacheLimit);
    }

    @Override
    public void setLastSyncTime(long lastSyncTime) {
        if (editor != null)
            editor.putLong(KEY_LAST_SYNC_TIME, lastSyncTime).apply();
    }

    @Override
    public long getLastSyncTime() {
        if (sharedPreferences != null) {
            return sharedPreferences.getLong(KEY_LAST_SYNC_TIME, DefaultValues.APM_LAST_SYNC_TIME);
        }
        return DefaultValues.APM_LAST_SYNC_TIME;
    }

    @Override
    public long getNetworkLogsLastSyncTime() {
        if (sharedPreferences != null) {
            return sharedPreferences.getLong(KEY_NETWORK_LOGS_LAST_SYNC_TIME,
                    DefaultValues.APM_LAST_SYNC_TIME);
        }
        return DefaultValues.APM_LAST_SYNC_TIME;
    }

    @Override
    public void setNetworkLogsLastSyncTime(long lastSyncTime) {
        if (editor != null)
            editor.putLong(KEY_NETWORK_LOGS_LAST_SYNC_TIME, lastSyncTime).apply();
    }

    @Override
    public boolean isNetworkGraphQlFeatureEnabled() {
        if (sharedPreferences != null) {
            return sharedPreferences.getBoolean(
                    KEY_NETWORK_GRAPH_QL_ENABLED,
                    DefaultValues.Network.GRAPH_QL_ENABLED
            );
        }
        return DefaultValues.Network.GRAPH_QL_ENABLED;
    }

    @Override
    public void setNetworkGraphQlFeatureEnabled(boolean enabled) {
        if (editor != null) {
            editor.putBoolean(KEY_NETWORK_GRAPH_QL_ENABLED, enabled).apply();
        }
    }

    @Override
    public boolean isNetworkGrpcInterceptionFeatureEnabled() {
        if (sharedPreferences != null) {
            return sharedPreferences.getBoolean(
                    KEY_NETWORK_GRPC_ENABLED,
                    DefaultValues.Network.GRPC_ENABLED
            );
        }
        return DefaultValues.Network.GRPC_ENABLED;
    }

    @Override
    public void setNetworkGrpcInterceptionFeatureEnabled(boolean enabled) {
        if (editor != null) {
            editor.putBoolean(KEY_NETWORK_GRPC_ENABLED, enabled).apply();
        }
    }

    @Override
    public boolean isNetworkGrpcInterceptionEnabled() {
        return isNetworkGrpcInterceptionFeatureEnabled() && isNetworkEnabled();
    }

    @Override
    public float getUiTraceLargeDropThreshold() {
        if (sharedPreferences != null) {
            return sharedPreferences.getFloat(KEY_UI_TRACE_LARGE_DROP_THRESHOLD,
                    DefaultValues.UiTraces.LARGE_DROP_THRESHOLD);
        }
        return DefaultValues.UiTraces.LARGE_DROP_THRESHOLD;
    }

    @Override
    public void setUiTraceLargeDropThreshold(float threshold) {
        if (editor != null) {
            editor.putFloat(KEY_UI_TRACE_LARGE_DROP_THRESHOLD, threshold).apply();
        }
    }

    @Override
    public float getUiTraceSmallDropThreshold() {
        if (sharedPreferences != null) {
            return sharedPreferences.getFloat(KEY_UI_TRACE_SMALL_DROP_THRESHOLD,
                    DefaultValues.UiTraces.SMALL_DROP_THRESHOLD);
        }
        return DefaultValues.UiTraces.SMALL_DROP_THRESHOLD;
    }

    @Override
    public void setUiTraceSmallDropThreshold(float threshold) {
        if (editor != null) {
            editor.putFloat(KEY_UI_TRACE_SMALL_DROP_THRESHOLD, threshold).apply();
        }
    }

    @Override
    public long getUiTraceLimitPerRequest() {
        long uiTraceLimitPerRequest = DefaultValues.UiTraces.LIMIT_PER_REQUEST;
        if (sharedPreferences != null) {
            uiTraceLimitPerRequest = sharedPreferences.getLong(KEY_UI_TRACE_LIMIT_PER_REQUEST,
                    DefaultValues.UiTraces.LIMIT_PER_REQUEST);
        }
        return limitConstraintApplier.applyConstraints(uiTraceLimitPerRequest);
    }

    @Override
    public void setUiTraceLimitPerRequest(long size) {
        if (editor != null) {
            editor.putLong(KEY_UI_TRACE_LIMIT_PER_REQUEST, size).apply();
        }
    }

    @Override
    public void setUiTraceSdkEnabled(boolean enabled) {
        if (apmPerSessionConfigurationProvider != null) {
            apmPerSessionConfigurationProvider.putConfiguration(KEY_UI_TRACE_SDK_ENABLED, enabled);
        }
    }

    @Override
    public long getUiTraceStoreLimit() {
        long uiTraceStoreLimit = DefaultValues.UiTraces.STORE_LIMIT;
        if (sharedPreferences != null) {
            uiTraceStoreLimit = sharedPreferences.getLong(KEY_UI_TRACE_STORE_LIMIT,
                    DefaultValues.UiTraces.STORE_LIMIT);
        }
        return limitConstraintApplier.applyConstraints(uiTraceStoreLimit);
    }

    @Override
    public boolean isUiTraceSdkEnabled() {
        return (boolean) apmPerSessionConfigurationProvider
                .getConfiguration(KEY_UI_TRACE_SDK_ENABLED, DefaultValues.UiTraces.SDK_ENABLED);
    }

    @Override
    public boolean isUiTraceSdkEnabledIncludingASubFeature() {
        return isUiTraceSdkEnabled() && (isUiHangsSdkEnabled() || isUiLoadingMetricsSdkEnabled());
    }

    @Override
    public void setUiTraceStoreLimit(long limit) {
        if (editor != null) {
            editor.putLong(KEY_UI_TRACE_STORE_LIMIT, limit).apply();
        }
    }

    @Override
    public void setUiHangsFeatureEnabled(boolean enabled) {
        if (editor != null) {
            isUiHangsFeatureEnabled = enabled;
            editor.putBoolean(KEY_UI_HANG_ENABLED, enabled).apply();
        }
    }

    @Override
    public boolean isUiHangsFeatureEnabled() {
        if (isUiHangsFeatureEnabled == null && sharedPreferences != null) {
            isUiHangsFeatureEnabled = sharedPreferences.getBoolean(KEY_UI_HANG_ENABLED,
                    DefaultValues.UiHangs.ENABLED);
        }
        return isUiHangsFeatureEnabled != null ? isUiHangsFeatureEnabled : DefaultValues.UiHangs.ENABLED;
    }

    @Override
    public void setUiHangsSdkEnabled(boolean enabled) {
        if (apmPerSessionConfigurationProvider != null) {
            apmPerSessionConfigurationProvider.putConfiguration(KEY_UI_HANG_SDK_ENABLED, enabled);
        }
    }

    @Override
    public boolean isUiHangsSdkEnabled() {
        if (apmPerSessionConfigurationProvider != null) {
            return (boolean) apmPerSessionConfigurationProvider
                    .getConfiguration(KEY_UI_HANG_SDK_ENABLED, DefaultValues.UiHangs.SDK_ENABLED);
        }
        return DefaultValues.UiHangs.SDK_ENABLED;
    }

    @Override
    public boolean isUiTraceEnabled() {
        return isUiTraceFeatureEnabled() && isUiTraceSdkEnabledIncludingASubFeature() && isAPMEnabled();
    }

    @Override
    public boolean isAutoUiHangsEnabled() {
        return isUiHangsFeatureEnabled() && isUiHangsSdkEnabled();
    }

    @Override
    public boolean isUiLoadingMetricsFeatureEnabled() {
        if (isUiLoadingMetricFeatureEnabled == null && sharedPreferences != null) {
            isUiLoadingMetricFeatureEnabled = sharedPreferences.getBoolean(
                    KEY_UI_LOADING_ENABLED,
                    DefaultValues.UiLoadingMetrics.ENABLED
            );
        }
        return isUiLoadingMetricFeatureEnabled != null ? isUiLoadingMetricFeatureEnabled : DefaultValues.UiLoadingMetrics.ENABLED;
    }

    @Override
    public void setUiLoadingMetricsFeatureEnabled(boolean enabled) {
        if (editor != null) {
            isUiLoadingMetricFeatureEnabled = enabled;
            editor.putBoolean(KEY_UI_LOADING_ENABLED, enabled).apply();
        }
    }

    @Override
    public boolean isUiTraceFeatureEnabled() {
        return isUiHangsFeatureEnabled() || isUiLoadingMetricsFeatureEnabled();
    }

    @Override
    public boolean isUiLoadingMetricsSdkEnabled() {
        return (boolean) apmPerSessionConfigurationProvider
                .getConfiguration(KEY_UI_LOADING_SDK_ENABLED, DefaultValues.UiLoadingMetrics.SDK_ENABLED);
    }

    @Override
    public void setUiLoadingMetricsSdkEnabled(boolean enabled) {
        if (apmPerSessionConfigurationProvider != null) {
            apmPerSessionConfigurationProvider.putConfiguration(KEY_UI_LOADING_SDK_ENABLED, enabled);
        }
    }

    @Override
    public boolean isAutoUiLoadingMetricsEnabled() {
        return isUiLoadingMetricsFeatureEnabled() && isUiLoadingMetricsSdkEnabled();
    }

    @Override
    public boolean isAutoUiLoadingMetricsFullyEnabled() {
        return isAPMEnabled() && isUiTraceSdkEnabled() && isAutoUiLoadingMetricsEnabled();
    }

    @Override
    public void setEndScreenLoadingFeatureEnabled(boolean enabled) {
        if (editor != null) {
            editor.putBoolean(KEY_END_SCREEN_LOADING_ENABLED, enabled)
                    .apply();
        }
    }

    @Override
    public boolean isEndScreenLoadingFeatureEnabled() {
        if (sharedPreferences != null) {
            return sharedPreferences.getBoolean(
                    KEY_END_SCREEN_LOADING_ENABLED,
                    DefaultValues.UiLoadingMetrics.END_SCREEN_LOADING_ENABLED
            );
        }
        return DefaultValues.UiLoadingMetrics.END_SCREEN_LOADING_ENABLED;
    }

    @Override
    public void setCrashDetectionEnabled(boolean isCrashDetectionEnabled) {
        if (editor != null) {
            editor.putBoolean(KEY_CRASH_DETECTION_ENABLED, isCrashDetectionEnabled).apply();
        }
    }

    @Override
    public boolean isCrashDetectionEnabled() {
        if (sharedPreferences != null) {
            return sharedPreferences.getBoolean(KEY_CRASH_DETECTION_ENABLED, DefaultValues.CRASH_DETECTION_ENABLED)
                    && isAPMEnabled();
        }
        return DefaultValues.CRASH_DETECTION_ENABLED;
    }

    @LogLevel
    @Override
    public int getLogLevel() {
        return getCorrespondingAPMLogLevel();
    }

    @LogLevel
    private int getCorrespondingAPMLogLevel() {
        switch (SettingsManager.getInstance().getLogLevel()) {
            case com.instabug.library.LogLevel.NONE:
                return LogLevel.NONE;
            case com.instabug.library.LogLevel.DEBUG:
                return LogLevel.DEBUG;
            case com.instabug.library.LogLevel.VERBOSE:
                return LogLevel.VERBOSE;
            case com.instabug.library.LogLevel.ERROR:
            default:
                return LogLevel.WARNING;
        }
    }

    @Override
    public void setNetworkLogsStoreAttributesLimit(int attributesStoreLimit) {
        if (editor != null)
            editor.putInt(APMConfigurationKeys.Network.KEY_ATTRIBUTES_COUNT, attributesStoreLimit).apply();
    }

    @Override
    public int getNetworkLogsStoreAttributesLimit() {
        if (sharedPreferences != null)
            return sharedPreferences.getInt(APMConfigurationKeys.Network.KEY_ATTRIBUTES_COUNT, DefaultValues.Network.ATTRIBUTES_STORE_LIMIT);
        return DefaultValues.Network.ATTRIBUTES_STORE_LIMIT;
    }

    @Override
    public void setDebugModeEnabled(boolean debugModeEnabled) {
        if (editor != null) {
            editor.putBoolean(KEY_DEBUG_MODE_ENABLED, debugModeEnabled).apply();
        }
    }

    @Override
    public boolean isDebugModeEnabled() {
        if (sharedPreferences != null) {
            return sharedPreferences.getBoolean(KEY_DEBUG_MODE_ENABLED, DefaultValues.DEBUG_MODE_ENABLED)
                    && isAPMEnabled();
        }
        return DefaultValues.DEBUG_MODE_ENABLED;
    }

    @Override
    public void setEndColdAppLaunchFeatureEnabled(boolean isEndColdLaunchEnabled) {
        if (editor != null) {
            editor.putBoolean(KEY_END_COLD_APP_LAUNCH_ENABLED, isEndColdLaunchEnabled).apply();
        }
    }

    @Override
    public void setEndHotAppLaunchFeatureEnabled(boolean isEndHotLaunchEnabled) {
        if (editor != null) {
            editor.putBoolean(KEY_END_HOT_APP_LAUNCH_ENABLED, isEndHotLaunchEnabled).apply();
        }
    }

    @Override
    public boolean isEndColdAppLaunchFeatureEnabled() {
        if (sharedPreferences != null) {
            return sharedPreferences.getBoolean(KEY_END_COLD_APP_LAUNCH_ENABLED, DefaultValues.DEBUG_MODE_ENABLED)
                    && isAPMEnabled();
        }
        return DefaultValues.Launches.ENABLED;
    }

    @Override
    public boolean isEndHotAppLaunchFeatureEnabled() {
        if (sharedPreferences != null) {
            return sharedPreferences.getBoolean(KEY_END_HOT_APP_LAUNCH_ENABLED, DefaultValues.DEBUG_MODE_ENABLED)
                    && isAPMEnabled();
        }
        return DefaultValues.Launches.ENABLED;
    }

    @Override
    public void setWarmAppLaunchFeatureEnabled(boolean enabled) {
        if (editor != null) {
            editor.putBoolean(KEY_WARM_APP_LAUNCH_ENABLED, enabled)
                    .apply();
        }
    }

    @Override
    public boolean isWarmAppLaunchFeatureEnabled() {
        boolean defaultValue = DefaultValues.Launches.WARM_ENABLED;
        if (sharedPreferences != null) {
            return sharedPreferences.getBoolean(KEY_WARM_APP_LAUNCH_ENABLED, defaultValue);
        }
        return defaultValue;
    }

    @Override
    public void setWarmAppLaunchRequestLimit(long limit) {
        if (editor != null) {
            editor.putLong(KEY_WARM_APP_LAUNCH_PER_REQUEST_LIMIT, limit)
                    .apply();
        }
    }

    private long getWarmAppLaunchRequestLimit() {
        long defaultValue = DefaultValues.Launches.LIMIT_PER_REQUEST_WARM;
        if (sharedPreferences != null) {
            return sharedPreferences.getLong(KEY_WARM_APP_LAUNCH_PER_REQUEST_LIMIT, defaultValue);
        }
        return defaultValue;
    }

    @Override
    public void setWarmAppLaunchStoreLimit(long limit) {
        if (editor != null) {
            editor.putLong(KEY_WARM_APP_LAUNCH_STORE_LIMIT, limit)
                    .apply();
        }
    }

    private long getWarmAppLaunchStoreLimit() {
        long defaultValue = DefaultValues.Launches.STORE_LIMIT_WARM;
        if (sharedPreferences != null) {
            return sharedPreferences.getLong(KEY_WARM_APP_LAUNCH_STORE_LIMIT, defaultValue);
        }
        return defaultValue;
    }

    @Override
    public void setEndWarmAppLaunchFeatureEnabled(boolean enabled) {
        if (editor != null) {
            editor.putBoolean(KEY_END_WARM_APP_LAUNCH_ENABLED, enabled)
                    .apply();
        }
    }

    @Override
    public boolean isEndWarmAppLaunchFeatureEnabled() {
        boolean defaultValue = DefaultValues.Launches.END_WARM_ENABLED;
        if (sharedPreferences != null) {
            return sharedPreferences.getBoolean(KEY_END_WARM_APP_LAUNCH_ENABLED, defaultValue);
        }
        return defaultValue;
    }

    @Override
    public void setWarmAppLaunchSdkEnabled(boolean enabled) {
        if (apmPerSessionConfigurationProvider != null) {
            apmPerSessionConfigurationProvider.putConfiguration(KEY_WARM_APP_LAUNCH_SDK_ENABLED, enabled);
        }
    }

    @Override
    public boolean isWarmAppLaunchSdkEnabled() {
        boolean defaultValue = DefaultValues.Launches.WARM_SDK_ENABLED;
        if (apmPerSessionConfigurationProvider != null) {
            return (boolean) apmPerSessionConfigurationProvider.getConfiguration(
                    KEY_WARM_APP_LAUNCH_SDK_ENABLED,
                    defaultValue
            );
        }
        return defaultValue;
    }

    @Override
    public boolean isWarmAppLaunchEnabled() {
        return isAPMEnabled() && isWarmAppLaunchFlagsEnabled();
    }

    private boolean isWarmAppLaunchFlagsEnabled() {
        return isWarmAppLaunchFeatureEnabled() && isWarmAppLaunchSdkEnabled();
    }

    @Override
    public void resetWarmAppLaunchFeatureFlag() {
        if (editor != null) {
            editor.remove(KEY_WARM_APP_LAUNCH_ENABLED).apply();
        }
    }

    @Override
    public void resetWarmAppLaunchRequestLimit() {
        if (editor != null) {
            editor.remove(KEY_WARM_APP_LAUNCH_PER_REQUEST_LIMIT).apply();
        }
    }

    @Override
    public void resetWarmAppLaunchStoreLimit() {
        if (editor != null) {
            editor.remove(KEY_WARM_APP_LAUNCH_STORE_LIMIT).apply();
        }
    }

    @Override
    public void resetEndWarmAppLaunchFeatureFlag() {
        if (editor != null) {
            editor.remove(KEY_END_WARM_APP_LAUNCH_ENABLED).apply();
        }
    }

    @Override
    public void resetWarmAppLaunchConfigurations() {
        resetWarmAppLaunchFeatureFlag();
        resetWarmAppLaunchRequestLimit();
        resetWarmAppLaunchStoreLimit();
        resetEndWarmAppLaunchFeatureFlag();
    }

    @Override
    public boolean isAppLaunchesEnabled() {
        return isAPMEnabled() &&
                (isColdAppLaunchFlagsEnabled()
                        || isWarmAppLaunchFlagsEnabled()
                        || isHotAppLaunchFlagsEnabled()
                );
    }

    @Override
    public long getAppLaunchStoreLimit(@AppLaunchType String appLaunchType) {
        switch (appLaunchType) {
            case AppLaunchType.COLD:
                return getColdAppLaunchesStoreLimit();
            case AppLaunchType.WARM:
                return getWarmAppLaunchStoreLimit();
            case AppLaunchType.HOT:
                return getHotAppLaunchesStoreLimit();
            default:
                return 0;
        }
    }

    @Override
    public long getAppLaunchRequestLimit(@AppLaunchType String appLaunchType) {
        switch (appLaunchType) {
            case AppLaunchType.COLD:
                return getColdAppLaunchesLimitPerRequest();
            case AppLaunchType.WARM:
                return getWarmAppLaunchRequestLimit();
            case AppLaunchType.HOT:
                return getHotAppLaunchesLimitPerRequest();

            default:
                return 0;
        }
    }

    @Override
    public boolean isAppLaunchFeatureEnabled(@AppLaunchType String appLaunchType) {
        switch (appLaunchType) {
            case AppLaunchType.COLD:
                return isColdAppLaunchesFeatureEnabled();
            case AppLaunchType.WARM:
                return isWarmAppLaunchFeatureEnabled();
            case AppLaunchType.HOT:
                return isHotAppLaunchesFeatureEnabled();

            default:
                return false;
        }
    }

    @Override
    public boolean isAppLaunchSdkEnabled(@AppLaunchType String appLaunchType) {
        switch (appLaunchType) {
            case AppLaunchType.COLD:
                return isColdAppLaunchSDKEnabled();
            case AppLaunchType.WARM:
                return isWarmAppLaunchSdkEnabled();
            case AppLaunchType.HOT:
                return isHotAppLaunchSDKEnabled();

            default:
                return false;
        }
    }

    @Override
    public boolean isEndAppLaunchFeatureEnabled(@AppLaunchType String appLaunchType) {
        switch (appLaunchType) {
            case AppLaunchType.COLD:
                return isEndColdAppLaunchFeatureEnabled();
            case AppLaunchType.WARM:
                return isEndWarmAppLaunchFeatureEnabled();
            case AppLaunchType.HOT:
                return isEndHotAppLaunchFeatureEnabled();

            default:
                return false;
        }
    }

    @Override
    public boolean isAppLaunchEnabled(String appLaunchType) {
        switch (appLaunchType) {
            case AppLaunchType.COLD:
                return isColdAppLaunchesEnabled();
            case AppLaunchType.WARM:
                return isWarmAppLaunchEnabled();
            case AppLaunchType.HOT:
                return isHotAppLaunchesEnabled();

            default:
                return false;
        }
    }

    @Override
    public void setExperimentsFeatureEnabled(boolean enabled) {
        if (editor != null) {
            editor.putBoolean(KEY_EXPERIMENTS_ENABLED, enabled).apply();
        }
    }

    @Override
    public void resetExperimentsFeatureEnabledFlag() {
        setExperimentsFeatureEnabled(DefaultValues.Experiments.ENABLED);
    }

    @Override
    public boolean isExperimentsFeatureEnabled() {
        boolean defaultValue = DefaultValues.Experiments.ENABLED;
        if (sharedPreferences != null) {
            return sharedPreferences.getBoolean(KEY_EXPERIMENTS_ENABLED, defaultValue);
        }
        return defaultValue;
    }

    @Override
    public void setExperimentsLimitPerRequest(int limit) {
        if (editor != null) {
            editor.putInt(KEY_EXPERIMENTS_LIMIT_PER_REQUEST, limit).apply();
        }
    }

    @Override
    public void resetExperimentsLimitPerRequest() {
        setExperimentsLimitPerRequest(DefaultValues.Experiments.LIMIT_PER_REQUEST);
    }

    @Override
    public int getExperimentsLimitPerRequest() {
        int defaultValue = DefaultValues.Experiments.LIMIT_PER_REQUEST;
        if (sharedPreferences != null) {
            return sharedPreferences.getInt(KEY_EXPERIMENTS_LIMIT_PER_REQUEST, defaultValue);
        }
        return defaultValue;
    }

    @Override
    public void setSessionStoreLimitEnabled(boolean enabled) {
        if (editor != null) {
            editor.putBoolean(APMConfigurationKeys.KEY_SESSION_STORE_LIMIT_ENABLED, enabled)
                    .apply();
        }
    }

    @Override
    public boolean isSessionStoreLimitEnabled() {
        boolean defaultValue = DefaultValues.SESSION_STORE_LIMIT_ENABLED;
        if (sharedPreferences != null) {
            return sharedPreferences.getBoolean(
                    APMConfigurationKeys.KEY_SESSION_STORE_LIMIT_ENABLED, defaultValue
            );
        }
        return defaultValue;
    }

    @Override
    public void resetSessionStoreLimitEnabledFlag() {
        if (editor != null) {
            editor.remove(APMConfigurationKeys.KEY_SESSION_STORE_LIMIT_ENABLED)
                    .apply();
        }
    }

    @Override
    public void setSessionStoreLimit(int limit) {
        if (editor != null) {
            editor.putInt(APMConfigurationKeys.KEY_SESSION_STORE_LIMIT, limit)
                    .apply();
        }
    }

    @Override
    public int getSessionStoreLimit() {
        int limit = DefaultValues.SESSION_STORE_LIMIT;
        if (sharedPreferences != null) {
            return sharedPreferences.getInt(
                    APMConfigurationKeys.KEY_SESSION_STORE_LIMIT, limit
            );
        }
        return limit;
    }

    @Override
    public void resetSessionStoreLimit() {
        if (editor != null) {
            editor.remove(APMConfigurationKeys.KEY_SESSION_STORE_LIMIT)
                    .apply();
        }
    }

    @Override
    public void setShouldDependOnV3Session(boolean shouldDependOnV3Session) {
        if (editor != null) {
            editor
                    .putBoolean(APMConfigurationKeys.KEY_SHOULD_DEPEND_ON_V3_SESSION, shouldDependOnV3Session)
                    .apply();
        }
    }

    @Override
    public boolean shouldDependOnV3Session() {

        return sharedPreferences != null
                ? sharedPreferences
                .getBoolean(APMConfigurationKeys.KEY_SHOULD_DEPEND_ON_V3_SESSION, DefaultValues.SHOULD_DEPEND_ON_V3_SESSION)
                : DefaultValues.SHOULD_DEPEND_ON_V3_SESSION;
    }

    @Override
    public void clearState() {
        resetStoreLimitDroppedSessionCount();
    }

    @Override
    public void incrementStoreLimitDroppedSessionsCount(int count) {
        int totalDroppedSessionsCount = getStoreLimitDroppedSessionsCount() + count;
        if (editor != null) {
            editor.putInt(KEY_DROPPED_SESSION_COUNT_STORE_LIMIT, totalDroppedSessionsCount)
                    .apply();
        }
    }

    @Override
    public int getStoreLimitDroppedSessionsCount() {
        if (sharedPreferences != null) {
            return sharedPreferences.getInt(KEY_DROPPED_SESSION_COUNT_STORE_LIMIT, 0);
        }
        return 0;
    }

    @Override
    public void resetStoreLimitDroppedSessionCount() {
        if (editor != null) {
            editor.remove(KEY_DROPPED_SESSION_COUNT_STORE_LIMIT)
                    .apply();
        }
    }

    private long getLastApmSessionsRequestStartedAt() {
        return (sharedPreferences == null)
                ? 0L
                : sharedPreferences.getLong(LAST_APM_SESSIONS_REQUEST_STARTED_AT, 0L);
    }

    @Override
    public void setLastApmSessionsRequestStartedAt(long time) {
        if (editor == null) return;
        editor.putLong(LAST_APM_SESSIONS_REQUEST_STARTED_AT, time).apply();
    }

    private long getApmSessionsLimitedUntil() {
        return (sharedPreferences == null)
                ? 0L
                : sharedPreferences.getLong(APM_SESSIONS_LIMITED_UNTIL, 0L);
    }

    @Override
    public void setApmSessionsLimitedUntil(int period) {
        if (sharedPreferences == null) return;
        long limitedUntil = getLastApmSessionsRequestStartedAt() + period * 1000L;
        if (editor != null)
            editor.putLong(APM_SESSIONS_LIMITED_UNTIL, limitedUntil).apply();
    }

    @Override
    public boolean isApmSessionsRateLimited() {
        long start = getLastApmSessionsRequestStartedAt();
        long limitedUntil = getApmSessionsLimitedUntil();
        long currentTime = System.currentTimeMillis();
        return start != 0 && limitedUntil != 0
                && currentTime > start && currentTime < limitedUntil;
    }

    @Override
    public void setSendLegacyAPMSessions(boolean status) {
        if (editor != null) {
            editor.putBoolean(KEY_SHOULD_SEND_LEGACY_SESSIONS, status).apply();
        }
    }

    @Override
    public boolean shouldSendLegacyAPMSessions() {
        if (sharedPreferences != null) {
            return sharedPreferences.getBoolean(KEY_SHOULD_SEND_LEGACY_SESSIONS, false);
        }
        return false;
    }

    @Override
    public int getFragmentSpansStoreLimit() {
        return (sharedPreferences == null)
                ? DefaultValues.FragmentSpans.STORE_LIMIT
                : sharedPreferences.getInt(APMConfigurationKeys.FragmentSpans.KEY_FRAGMENT_SPANS_STORE_LIMIT,
                DefaultValues.FragmentSpans.STORE_LIMIT);
    }

    @Override
    public int getFragmentSpansLimitPerRequest() {
        return (sharedPreferences == null)
                ? DefaultValues.FragmentSpans.LIMIT_PER_REQUEST
                : sharedPreferences.getInt(APMConfigurationKeys.FragmentSpans.KEY_FRAGMENT_SPANS_LIMIT_PER_REQUEST,
                DefaultValues.FragmentSpans.LIMIT_PER_REQUEST);
    }

    @Override
    public void setFragmentSpansFeatureEnabledFlag(boolean enabled) {
        if (sharedPreferences == null) return;
        if (editor != null)
            editor.putBoolean(APMConfigurationKeys.FragmentSpans.KEY_FRAGMENT_SPANS_ENABLED, enabled).apply();
    }

    @Override
    public void setFragmentSpansStoreLimit(int limit) {
        if (sharedPreferences == null) return;
        if (editor != null)
            editor.putInt(APMConfigurationKeys.FragmentSpans.KEY_FRAGMENT_SPANS_STORE_LIMIT, limit).apply();
    }

    @Override
    public void setFragmentSpansLimitPerRequest(int limit) {
        if (sharedPreferences == null) return;
        if (editor != null)
            editor.putInt(APMConfigurationKeys.FragmentSpans.KEY_FRAGMENT_SPANS_LIMIT_PER_REQUEST, limit).apply();
    }

    @Override
    public boolean isFragmentSpansFeatureEnabled() {
        return (sharedPreferences == null)
                ? DefaultValues.FragmentSpans.ENABLED
                : sharedPreferences.getBoolean(APMConfigurationKeys.FragmentSpans.KEY_FRAGMENT_SPANS_ENABLED,
                DefaultValues.FragmentSpans.ENABLED);
    }

    @Override
    public void resetFragmentSpansFeatureEnabledFlag() {
        setFragmentSpansFeatureEnabledFlag(DefaultValues.FragmentSpans.ENABLED);
    }

    @Override
    public void resetFragmentSpansStoreLimit() {
        setFragmentSpansStoreLimit(DefaultValues.FragmentSpans.STORE_LIMIT);
    }

    @Override
    public void resetFragmentSpansLimitPerRequest() {
        setFragmentSpansLimitPerRequest(DefaultValues.FragmentSpans.LIMIT_PER_REQUEST);
    }

    @Override
    public void setFragmentSpansSDKEnabled(boolean enabled) {
        if (apmPerSessionConfigurationProvider != null) {
            apmPerSessionConfigurationProvider.putConfiguration(APMConfigurationKeys.FragmentSpans.KEY_FRAGMENT_SPANS_SDK_ENABLED, enabled);
        }
    }

    @Override
    public boolean isFragmentSpansSDKEnabled() {
        return (boolean) apmPerSessionConfigurationProvider
                .getConfiguration(APMConfigurationKeys.FragmentSpans.KEY_FRAGMENT_SPANS_SDK_ENABLED, DefaultValues.FragmentSpans.SDK_ENABLED);
    }

    @Override
    public boolean isFragmentSpansEnabled() {
        return isFragmentSpansFeatureEnabled() && isFragmentSpansSDKEnabled();
    }

    @Override
    public boolean isW3CNetworkExternalTraceIdEnabled() {
        return isNetworkEnabled() &&
                coreNetworkInterceptionConfigurationProvider.isW3CNetworkExternalTraceIdEnabled();
    }

    @Override
    public boolean isAttachingGeneratedW3CExternalTraceIdFeatureAvailable() {
        return coreNetworkInterceptionConfigurationProvider.
                isAttachingGeneratedW3CExternalTraceIdFeatureAvailable();
    }

    @Override
    public boolean isAttachingGeneratedW3CExternalTraceIdEnabled() {
        return isNetworkEnabled() && coreNetworkInterceptionConfigurationProvider
                .isAttachingGeneratedW3CExternalTraceIdEnabled();
    }

    @Override
    public boolean isAttachingCapturedW3CExternalTraceIdFeatureAvailable() {
        return coreNetworkInterceptionConfigurationProvider.
                isAttachingCapturedW3CExternalTraceIdFeatureAvailable();
    }

    @Override
    public boolean isAttachingCapturedW3CExternalTraceIdEnabled() {
        return isNetworkEnabled() && coreNetworkInterceptionConfigurationProvider
                .isAttachingCapturedW3CExternalTraceIdEnabled();
    }

    private boolean isFeatureAvailable(String key, Boolean defaultValue) {
        return (sharedPreferences == null) ? defaultValue : sharedPreferences.getBoolean(key, defaultValue);
    }
}
