package com.instabug.apm.cache.handler.uitrace;

import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMUiLoadingMetricEntry.COLUMN_DURATION;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMUiLoadingMetricEntry.COLUMN_ID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMUiLoadingMetricEntry.COLUMN_START_TIME;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMUiLoadingMetricEntry.COLUMN_TYPE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMUiLoadingMetricEntry.COLUMN_UI_TRACE_ID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMUiLoadingMetricEntry.DELETE_ALL;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMUiLoadingMetricEntry.TABLE_NAME;

import android.annotation.SuppressLint;
import android.content.ContentValues;
import android.database.Cursor;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.apm.cache.model.UiLoadingModel;
import com.instabug.apm.di.ServiceLocator;
import com.instabug.apm.logger.internal.Logger;
import com.instabug.library.internal.storage.cache.db.DatabaseManager;
import com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMUiLoadingStageEntry;
import com.instabug.library.internal.storage.cache.db.SQLiteDatabaseWrapper;

import java.util.HashMap;
import java.util.Map;

public class UiLoadingMetricCacheHandlerImpl implements UiLoadingMetricCacheHandler {

    private final Logger apmLogger = ServiceLocator.getApmLogger();

    @Override
    public long insert(UiLoadingModel uiLoadingModel, long uiTraceId) {
        if (uiLoadingModel == null) {
            return -1;
        }
        long uiLoadingMetricId = -1;
        DatabaseManager databaseManager = getDatabaseManager();
        if (databaseManager != null) {
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            ContentValues contentValues = getUiLoadingMetricModelContentValues(uiLoadingModel, uiTraceId);
            try {
                uiLoadingMetricId =
                        sqLiteDatabaseWrapper.insert(TABLE_NAME, null, contentValues);
                Map<String, Long> stages = uiLoadingModel.getStages();
                if (stages != null && !stages.isEmpty() && uiLoadingMetricId != -1) {
                    insertStages(sqLiteDatabaseWrapper, stages, uiLoadingMetricId);
                }
            } finally {
                sqLiteDatabaseWrapper.close();
            }
        }
        return uiLoadingMetricId;
    }

    private ContentValues getUiLoadingMetricModelContentValues(UiLoadingModel uiLoadingModel, long uiTraceId) {
        ContentValues contentValues = new ContentValues();
        contentValues.put(COLUMN_UI_TRACE_ID, uiTraceId);
        contentValues.put(COLUMN_DURATION, uiLoadingModel.getDurationInMicro());
        contentValues.put(COLUMN_START_TIME, uiLoadingModel.getStartTimeStampMicro());
        contentValues.put(COLUMN_TYPE, uiLoadingModel.getType());
        return contentValues;
    }

    private void insertStages(
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper,
            Map<String, Long> stages,
            long uiLoadingId
    ) {
        if (sqLiteDatabaseWrapper == null || stages == null) {
            return;
        }
        for (Map.Entry<String, Long> entry : stages.entrySet()) {
            ContentValues contentValues = getStagesContentValues(entry.getKey(), entry.getValue(), uiLoadingId);
            sqLiteDatabaseWrapper.insert(APMUiLoadingStageEntry.TABLE_NAME, null, contentValues);
        }
    }

    private ContentValues getStagesContentValues(String stageName, Long stageDuration, long uiLoadingId) {
        ContentValues contentValues = new ContentValues();
        contentValues.put(APMUiLoadingStageEntry.COLUMN_UI_LOADING_METRIC_ID, uiLoadingId);
        contentValues.put(APMUiLoadingStageEntry.COLUMN_STAGE_NAME, stageName);
        contentValues.put(APMUiLoadingStageEntry.COLUMN_STAGE_DURATION, stageDuration);
        return contentValues;
    }

    @Nullable
    @Override
    public UiLoadingModel getUiLoadingMetricForUiTrace(long uiTraceId) {
        UiLoadingModel uiLoadingModel = getUiLoadingModelFromDatabase(uiTraceId);
        if (uiLoadingModel != null) {
            Map<String, Long> stagesForUiModel = getStagesForUiModel(uiLoadingModel.getId());
            uiLoadingModel.setStages(stagesForUiModel);
        }
        return uiLoadingModel;
    }

    @Nullable
    private UiLoadingModel getUiLoadingModelFromDatabase(long uiTraceId) {
        UiLoadingModel uiLoadingModel = null;
        DatabaseManager databaseManager = getDatabaseManager();
        if (databaseManager != null) {
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            String[] selectionArgs = {String.valueOf(uiTraceId)};
            Cursor cursor = null;
            try {
                cursor = sqLiteDatabaseWrapper.query (
                        TABLE_NAME,
                        null,
                        COLUMN_UI_TRACE_ID + " = ?",
                        selectionArgs,
                        null,
                        null,
                        null
                );
                if (cursor != null) {
                    uiLoadingModel = readUiLoadingModelFromCursor(cursor);
                }
            } catch (Exception ex) {
                apmLogger.logSDKError("DB execution a sql failed: " + ex.getMessage(), ex);
            } finally {
                if (cursor != null) {
                    cursor.close();
                }
                sqLiteDatabaseWrapper.close();
            }

        }
        return uiLoadingModel;
    }

    @SuppressLint("Range")
    @Nullable
    private UiLoadingModel readUiLoadingModelFromCursor(@NonNull Cursor cursor) {
        if (cursor == null) {
            return null;
        }
        UiLoadingModel uiLoadingModel = null;
        if (cursor.moveToFirst()) {
            uiLoadingModel = new UiLoadingModel();
            uiLoadingModel.setId(
                    cursor.getLong(cursor.getColumnIndex(COLUMN_ID))
            );
            uiLoadingModel.setDurationInMicro(
                    cursor.getLong(cursor.getColumnIndex(COLUMN_DURATION))
            );
            uiLoadingModel.setStartTimeStampMicro(
                    cursor.getLong(cursor.getColumnIndex(COLUMN_START_TIME))
            );
            uiLoadingModel.setType(
                    cursor.getString(cursor.getColumnIndex(COLUMN_TYPE))
            );
        }
        return uiLoadingModel;
    }

    @Nullable
    private Map<String, Long> getStagesForUiModel(
            long uiLoadingMetricId
    ) {
        Map<String, Long> stages = null;
        DatabaseManager databaseManager = getDatabaseManager();
        if (databaseManager != null) {
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            String[] selectionArgs = {String.valueOf(uiLoadingMetricId)};
            Cursor cursor = null;
            try {
                cursor = sqLiteDatabaseWrapper.query(
                        APMUiLoadingStageEntry.TABLE_NAME,
                        null,
                        APMUiLoadingStageEntry.COLUMN_UI_LOADING_METRIC_ID + " = ?",
                        selectionArgs,
                        null,
                        null,
                        null
                );
                if (cursor != null) {
                    stages = readStagesFromCursor(cursor);
                }
            } catch (Exception exc) {
                apmLogger.logSDKError("DB execution a sql failed: " + exc.getMessage(), exc);
            } finally {
                if (cursor != null) {
                    cursor.close();
                }
                sqLiteDatabaseWrapper.close();
            }
        }
        return stages;
    }

    @Nullable
    private Map<String, Long> readStagesFromCursor(@NonNull Cursor cursor) {
        if (cursor == null) {
            return null;
        }
        Map<String, Long> stages = new HashMap<>();
        while (cursor.moveToNext()) {
            @SuppressLint("Range") String stageName = cursor.getString(
                    cursor.getColumnIndex(APMUiLoadingStageEntry.COLUMN_STAGE_NAME)
            );
            @SuppressLint("Range") Long stageDuration = cursor.getLong(
                    cursor.getColumnIndex(APMUiLoadingStageEntry.COLUMN_STAGE_DURATION)
            );
            stages.put(stageName, stageDuration);
        }
        if (stages.isEmpty()) {
            return null;
        } else {
            return stages;
        }
    }

    @Override
    public void removeAll() {
        DatabaseManager databaseManager = getDatabaseManager();
        if (databaseManager != null) {
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            try {
                sqLiteDatabaseWrapper.execSQL(DELETE_ALL);
            } catch (Exception exception) {
                apmLogger.logSDKError(
                        "DB execution a sql failed: " + exception.getMessage(),
                        exception);
            } finally {
                sqLiteDatabaseWrapper.close();
            }
        }
    }

    @Nullable
    private DatabaseManager getDatabaseManager(){
        return ServiceLocator.getDatabaseManager();
    }
}
