package com.instabug.apm.cache.handler.session;

import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMSessionEntry.COLUMN_APP_VERSION;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMSessionEntry.COLUMN_CORE_ID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMSessionEntry.COLUMN_CORE_VERSION;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMSessionEntry.COLUMN_DURATION;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMSessionEntry.COLUMN_ID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMSessionEntry.COLUMN_OS;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMSessionEntry.COLUMN_STARTED_AT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMSessionEntry.COLUMN_SYNC_STATUS;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMSessionEntry.COLUMN_TERMINATION_CODE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMSessionEntry.COLUMN_UUID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMSessionEntry.TABLE_NAME;

import android.annotation.SuppressLint;
import android.content.ContentValues;
import android.database.Cursor;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.apm.cache.model.SessionCacheModel;
import com.instabug.apm.constants.Constants;
import com.instabug.apm.di.ServiceLocator;
import com.instabug.apm.logger.internal.Logger;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.storage.cache.db.DatabaseManager;
import com.instabug.library.internal.storage.cache.db.SQLiteDatabaseWrapper;
import com.instabug.library.internal.utils.stability.execution.ReturnableExecutable;
import com.instabug.library.internal.utils.stability.handler.exception.ExceptionHandler;
import com.instabug.library.model.common.Session;
import com.instabug.library.model.common.SessionVersion;
import com.instabug.library.util.InstabugSDKLogger;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

public class SessionCacheHandlerImpl implements SessionCacheHandler {

    private static final int NO_LIMIT = -1;

    @NonNull
    private final ExceptionHandler exceptionHandler;
    @NonNull
    private final Logger logger;

    public SessionCacheHandlerImpl(@NonNull ExceptionHandler exceptionHandler,
                                   @NonNull Logger logger) {
        this.exceptionHandler = exceptionHandler;
        this.logger = logger;
    }

    @Override
    @Nullable
    public SessionCacheModel insert(@NonNull final Session coreSession) {
        return exceptionHandler.executeAndGet(() -> {
            DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
            if (databaseManager != null) {
                ContentValues values = map(coreSession);
                SQLiteDatabaseWrapper database = databaseManager.openDatabase();
                try {
                    long rowId = database.insertWithOnConflict(TABLE_NAME, null, values);
                    return new SessionCacheModel(String.valueOf(rowId), coreSession);
                } finally {
                    database.close();
                }
            }
            return null;
        });
    }

    @Override
    @SuppressLint("ERADICATE_NULLABLE_DEREFERENCE")
    public int update(@NonNull final SessionCacheModel session) {
        return exceptionHandler.executeAndGet(new ReturnableExecutable<Integer>() {
            @Override
            public Integer execute() {
                DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
                if (databaseManager != null) {
                    ContentValues values = map(session);
                    SQLiteDatabaseWrapper database = databaseManager.openDatabase();
                    try {
                        String whereClause = COLUMN_ID + " = ?";
                        String[] whereArgs = {session.getId()};
                        return database.update(TABLE_NAME, values, whereClause, whereArgs);
                    } catch (Exception ex) {
                        logger.logSDKError("DB execution a sql failed: " + ex.getMessage(), ex);
                        IBGDiagnostics.reportNonFatal(ex, "Error while updating session: " + ex.getMessage());
                    } finally {
                        database.close();
                    }
                }
                return 0;
            }
        }, 0);
    }


    @Override
    @SuppressLint("ERADICATE_NULLABLE_DEREFERENCE")
    public int updateEndReason(@NonNull final String coreSessionId, final long sessionDuration, @Constants.SessionEndStatusCode final int reason) {
        return exceptionHandler.executeAndGet(new ReturnableExecutable<Integer>() {
            @Override
            public Integer execute() {
                DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
                if (databaseManager != null) {
                    ContentValues values = new ContentValues();
                    values.put(COLUMN_TERMINATION_CODE, reason);
                    values.put(COLUMN_DURATION, sessionDuration);
                    SQLiteDatabaseWrapper database = databaseManager.openDatabase();
                    try {
                        String whereClause = COLUMN_CORE_ID + " = ?";
                        String[] whereArgs = {coreSessionId};
                        return database.update(TABLE_NAME, values, whereClause, whereArgs);
                    } finally {
                        database.close();
                    }
                }
                return 0;
            }
        }, 0);

    }

    @Override
    @SuppressLint("ERADICATE_NULLABLE_DEREFERENCE")
    public int trimSessions(int maxRowCount) {
        return exceptionHandler.executeAndGet(() -> {
            DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
            if (databaseManager != null) {
                String selectByLimitDescendingQuery =
                        "SELECT " + COLUMN_ID
                                + " FROM " + TABLE_NAME
                                + " ORDER BY " + COLUMN_ID + " DESC"
                                + " LIMIT " + maxRowCount;
                String whereClause =
                        COLUMN_ID + " NOT IN (" + selectByLimitDescendingQuery + ")";
                SQLiteDatabaseWrapper databaseWrapper = databaseManager.openDatabase();
                return databaseWrapper.delete(TABLE_NAME, whereClause, null);
            }
            return 0;
        }, 0);
    }

    @Nullable
    @Override
    public SessionCacheModel query(@NonNull final String id) {
        return exceptionHandler.executeAndGet(new ReturnableExecutable<SessionCacheModel>() {
            @Nullable
            @Override
            public SessionCacheModel execute() {
                DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
                if (databaseManager != null) {
                    String whereClause = COLUMN_ID + " = ? ";
                    String[] whereArgs = new String[]{id};
                    SQLiteDatabaseWrapper database = databaseManager.openDatabase();
                    Cursor cursor = null;
                    try {
                        cursor = database.query(TABLE_NAME, null, whereClause, whereArgs, null, null, null);
                        if (cursor != null && cursor.moveToFirst()) {
                            return map(cursor);
                        } else {
                            logger.logSDKProtected("Failed to query session with ID: " + id);
                        }
                    } catch (Exception ex) {
                        logger.logSDKError("DB execution a sql failed: " + ex.getMessage(), ex);
                        IBGDiagnostics.reportNonFatal(ex, "Error while getting session by id: " + ex.getMessage());
                    } finally {
                        if (cursor != null) {
                            cursor.close();
                        }
                        database.close();
                    }
                }
                return null;
            }
        });
    }

    @NonNull
    @Override
    public Collection<SessionCacheModel> queryAll() {
        return queryAll(NO_LIMIT);
    }

    @NonNull
    @Override
    public Collection<SessionCacheModel> queryAll(final int limit) {
        return exceptionHandler.executeAndGet(new ReturnableExecutable<Collection<SessionCacheModel>>() {
            @Override
            public Collection<SessionCacheModel> execute() {
                DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
                if (databaseManager != null) {
                    SQLiteDatabaseWrapper database = databaseManager.openDatabase();
                    Cursor cursor = null;
                    try {
                        String limitClause = limit > 0 ? String.valueOf(limit) : null;
                        cursor = database.query(TABLE_NAME, null, null, null, null, null, null, limitClause);
                        if (cursor != null && cursor.moveToFirst()) {
                            Collection<SessionCacheModel> cacheModels = new ArrayList<>();
                            do {
                                cacheModels.add(map(cursor));
                            }
                            while (cursor.moveToNext());
                            logger.logSDKProtected("Queried " + cacheModels.size() + " session cache models with limit = " + limit);
                            return cacheModels;
                        }
                    } catch (Exception ex) {
                        logger.logSDKError("Error while getting sessions: " + ex.getMessage(), ex);
                        IBGDiagnostics.reportNonFatal(ex, "Error while getting sessions: " + ex.getMessage());
                    } finally {
                        if (cursor != null) {
                            cursor.close();
                        }
                        database.close();
                    }
                }
                return Collections.emptyList();
            }
        }, Collections.emptyList());
    }


    @NonNull
    @Override
    public List<SessionCacheModel> queryByCoreIds(@NonNull List<String> coreIds) {
        return exceptionHandler.executeAndGet(() -> {
            DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
            String whereClause = COLUMN_CORE_ID + " IN " + prepareArgs(coreIds.size());
            if (databaseManager != null) {
                SQLiteDatabaseWrapper database = databaseManager.openDatabase();
                try (Cursor cursor = database.query(TABLE_NAME, null, whereClause, coreIds.toArray(new String[0]), null, null, null)) {
                    if (cursor != null && cursor.moveToFirst()) {
                        List<SessionCacheModel> cacheModels = new ArrayList<>();
                        do {
                            cacheModels.add(map(cursor));
                        }
                        while (cursor.moveToNext());
                        return cacheModels;
                    }
                } catch (Exception ex) {
                    logger.logSDKError("Error while querying sessions by core ids: " + ex.getMessage(), ex);
                    IBGDiagnostics.reportNonFatal(ex, "Error while querying sessions by core ids: " + ex.getMessage());
                }
            }
            return Collections.emptyList();
        }, Collections.emptyList());
    }

    private String prepareArgs(int size) {
        StringBuilder builder = new StringBuilder();
        builder.append("(");
        for (int i = 0; i < size; i++) {
            builder.append("?");
            if (i < size - 1) builder.append(",");
        }
        builder.append(")");
        return builder.toString();
    }


    @Override
    @SuppressLint("ERADICATE_NULLABLE_DEREFERENCE")
    public int delete(@NonNull final String id) {
        return exceptionHandler.executeAndGet(new ReturnableExecutable<Integer>() {
            @Override
            public Integer execute() {
                DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
                if (databaseManager != null) {
                    SQLiteDatabaseWrapper database = databaseManager.openDatabase();
                    try {
                        String whereClause = COLUMN_ID + " = ? ";
                        String[] whereArgs = new String[]{id};
                        int affectedRows = database.delete(TABLE_NAME, whereClause, whereArgs);
                        logger.logSDKProtected("Deleted " + affectedRows + " row/s associated with session ID: " + id);
                        return affectedRows;
                    } catch (Exception ex) {
                        logger.logSDKError("Error while deleting session: " + ex.getMessage(), ex);
                        IBGDiagnostics.reportNonFatal(ex, "Error while deleting session: " + ex.getMessage());
                    } finally {
                        database.close();
                    }
                }
                return 0;
            }
        }, 0);
    }

    @Override
    public void deleteByCoreIds(@NonNull List<String> coreIds) {
        exceptionHandler.execute(() -> {
            DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
            if (databaseManager != null) {
                SQLiteDatabaseWrapper database = databaseManager.openDatabase();
                try {
                    String whereClause = COLUMN_CORE_ID + " IN " + prepareArgs(coreIds.size());
                    String[] whereArgs = coreIds.toArray(new String[0]);
                    int affectedRows = database.delete(TABLE_NAME, whereClause, whereArgs);
                    logger.logSDKProtected("Deleted " + affectedRows + " row/s associated with session IDs: " + String.join(",", coreIds));

                } catch (Exception ex) {
                    logger.logSDKError("Error while deleting sessions: " + ex.getMessage(), ex);
                    IBGDiagnostics.reportNonFatal(ex, "Error while deleting sessions: " + ex.getMessage());
                }
            }

        });
    }

    @NonNull
    @Override
    public List<SessionCacheModel> getReadyToBeSentSessions() {
        DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
        List<SessionCacheModel> sessionsList = new ArrayList<>();
        if (databaseManager != null) {
            SQLiteDatabaseWrapper databaseWrapper = databaseManager.openDatabase();
            String selectionClause = COLUMN_SYNC_STATUS + " = ? and " + COLUMN_CORE_VERSION + " = ?";
            String[] selectionArgs = new String[]{String.valueOf(Constants.SessionSyncStatus.READY_TO_BE_SENT),
                    SessionVersion.V2};
            Cursor cursor = null;
            try {
                cursor = databaseWrapper.query(TABLE_NAME,
                        null, selectionClause, selectionArgs,
                        null, null, null);
                if (cursor != null) {
                    if (cursor.moveToFirst()) {
                        do {
                            SessionCacheModel cacheModel = map(cursor);
                            sessionsList.add(cacheModel);
                        } while (cursor.moveToNext());
                    }
                }
                databaseWrapper.close();
                return sessionsList;
            } catch (Exception ex) {
                logger.logSDKError("Error while getting ready to sync sessions: " + ex.getMessage(), ex);
                IBGDiagnostics.reportNonFatal(ex, "Error while getting ready to sync sessions: " + ex.getMessage());
            } finally {
                if (cursor != null) {
                    cursor.close();
                }
            }
        }
        return sessionsList;
    }

    @Override
    @Nullable
    public SessionCacheModel getNextSession(String currentSessionID) {
        SessionCacheModel cacheModel = null;
        DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
        if (databaseManager != null) {
            SQLiteDatabaseWrapper databaseWrapper = databaseManager.openDatabase();
            // Selecting the sessions list
            //exclude APM session linked to v3 session
            // 1. EXCLUDING last session by neglecting the max id (last session is the current running session, session duration is null)
            // 2. INCLUDING last session by selecting the max id if the duration is not null (last session is already ended once SDK is disabled)
            String selectionClause = COLUMN_ID + " > ?" +
                    " and " + COLUMN_CORE_VERSION + " = ? " +
                    " and (" +
                    COLUMN_ID + " not in (" + "select MAX(" + COLUMN_ID + ") from " + TABLE_NAME + ") or (" +
                    COLUMN_ID + " in(select MAX(" + COLUMN_ID + ") from " + TABLE_NAME + ") and (" +
                    COLUMN_DURATION + " not null)))";
            String[] selectionArgs = {currentSessionID, SessionVersion.V2};
            String orderByClause = COLUMN_ID + " ASC";
            Cursor cursor = null;
            try {
                cursor = databaseWrapper.query(TABLE_NAME,
                        null, selectionClause, selectionArgs,
                        null, null, orderByClause, "1");
                if (cursor != null) {
                    if (cursor.moveToFirst()) {
                        cacheModel = map(cursor);
                    }
                }
                databaseWrapper.close();
                return cacheModel;
            } catch (Exception ex) {
                logger.logSDKError("Error while getting next session: " + ex.getMessage(), ex);
                IBGDiagnostics.reportNonFatal(ex, "Error while getting next session: " + ex.getMessage());
            } finally {
                if (cursor != null) {
                    cursor.close();
                }
            }
        }
        return null;
    }

    @Override
    public void changeSessionSyncStatus(@NonNull List<String> sessionsIDs, int syncStatus) {
        DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
        if (databaseManager != null) {
            SQLiteDatabaseWrapper databaseWrapper = databaseManager.openDatabase();
            ContentValues contentValues = new ContentValues();
            contentValues.put(COLUMN_SYNC_STATUS, syncStatus);
            for (String sessionID : sessionsIDs) {
                databaseWrapper.update(TABLE_NAME, contentValues,
                        COLUMN_ID + " in (?)",
                        new String[]{sessionID});
            }
            databaseWrapper.close();
        }
    }

    @Override
    public void deleteSessionsBySyncStatus(int syncStatus) {
        DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
        if (databaseManager != null) {
            SQLiteDatabaseWrapper databaseWrapper = databaseManager.openDatabase();
            databaseWrapper.delete(TABLE_NAME,
                    COLUMN_SYNC_STATUS + " = ?",
                    new String[]{String.valueOf(syncStatus)});
            databaseWrapper.close();
        }
    }


    @Nullable
    @Override
    public SessionCacheModel getPreviousSession(final String sessionId) {
        return exceptionHandler.executeAndGet(new ReturnableExecutable<SessionCacheModel>() {
            @Override
            @Nullable
            public SessionCacheModel execute() {
                DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
                if (databaseManager != null) {
                    SQLiteDatabaseWrapper database = databaseManager.openDatabase();
                    Cursor cursor = null;
                    try {
                        String query = "SELECT * FROM " + TABLE_NAME
                                + " where " + COLUMN_ID + " < " + sessionId
                                + " ORDER BY " + COLUMN_STARTED_AT + " DESC"
                                + " LIMIT 1";
                        cursor = database.rawQuery(query, null);
                        if (cursor != null && cursor.moveToFirst()) {
                            return map(cursor);
                        } else {
                            return null;
                        }
                    } catch (Exception e) {
                        InstabugSDKLogger.e(Constants.LOG_TAG, "Error while getting previous session from DB: " + e.getMessage(), e);
                        IBGDiagnostics.reportNonFatal(e, "Error while getting previous session from DB: " + e.getMessage());
                        return null;
                    } finally {
                        if (cursor != null) {
                            cursor.close();
                        }
                        database.close();
                    }
                }
                return null;
            }
        });
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    private ContentValues map(Session coreSession) {
        ContentValues values = new ContentValues();
        values.put(COLUMN_CORE_ID, coreSession.getId());
        values.put(COLUMN_OS, coreSession.getOs());
        values.put(COLUMN_UUID, coreSession.getUuid());
        values.put(COLUMN_CORE_VERSION, coreSession.getVersion());
        values.put(COLUMN_APP_VERSION, coreSession.getAppVersion());
        values.put(COLUMN_STARTED_AT, coreSession.getStartTimestampMicros());
        return values;
    }

    private ContentValues map(@NonNull SessionCacheModel model) {
        ContentValues values = new ContentValues();
        values.put(COLUMN_ID, model.getId());
        values.put(COLUMN_CORE_ID, model.getCoreId());
        values.put(COLUMN_OS, model.getOs());
        values.put(COLUMN_UUID, model.getUuid());
        values.put(COLUMN_APP_VERSION, model.getAppVersion());
        values.put(COLUMN_STARTED_AT, model.getStartTimestampMicros());
        values.put(COLUMN_DURATION, model.getDuration());
        values.put(COLUMN_CORE_VERSION, model.getVersion());
        values.put(COLUMN_TERMINATION_CODE, model.getTerminationStatusCode());
        return values;
    }

    private SessionCacheModel map(@NonNull Cursor cursor) {
        int columnIdIndex = cursor.getColumnIndex(COLUMN_ID);
        int columnCoreIdIndex = cursor.getColumnIndex(COLUMN_CORE_ID);
        int columnCoreVersionIndex = cursor.getColumnIndex(COLUMN_CORE_VERSION);
        int columnOsIndex = cursor.getColumnIndex(COLUMN_OS);
        int columnAppVersionIndex = cursor.getColumnIndex(COLUMN_APP_VERSION);
        int columnUuidIndex = cursor.getColumnIndex(COLUMN_UUID);
        int columnDurationIndex = cursor.getColumnIndex(COLUMN_DURATION);
        int columnStartedAtIndex = cursor.getColumnIndex(COLUMN_STARTED_AT);
        int columnTerminationCodeIndex = cursor.getColumnIndex(COLUMN_TERMINATION_CODE);
        int columnSyncStatus = cursor.getColumnIndex(COLUMN_SYNC_STATUS);
        return new SessionCacheModel(
                cursor.getString(columnIdIndex),
                cursor.getString(columnCoreIdIndex),
                cursor.getString(columnOsIndex),
                cursor.getString(columnAppVersionIndex),
                cursor.getString(columnUuidIndex),
                cursor.getLong(columnDurationIndex),
                cursor.getLong(columnStartedAtIndex),
                0,
                cursor.getString(columnCoreVersionIndex),
                cursor.getInt(columnTerminationCodeIndex),
                cursor.getInt(columnSyncStatus));
    }

}
