package com.instabug.apm.cache.handler.fragments

import android.content.ContentValues
import android.database.Cursor
import com.instabug.apm.cache.model.FragmentSpansEventCacheModel
import com.instabug.apm.di.ServiceLocator
import com.instabug.apm.fragment.model.FragmentSpansEvent
import com.instabug.apm.logger.internal.Logger
import com.instabug.library.diagnostics.IBGDiagnostics
import com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMFragmentEventsEntry

interface FragmentSpansEventsCacheHandler {
    fun getEventsForFragment(fragmentId: Long): List<FragmentSpansEventCacheModel>
    fun saveEvents(events: List<FragmentSpansEvent>, fragmentId: Long)
}

class FragmentSpansEventsCacheHandlerImpl : FragmentSpansEventsCacheHandler {

    private val databaseWrapper
        get() = ServiceLocator.getDatabaseManager()?.openDatabase()

    private val apmLogger: Logger
        get() = ServiceLocator.getApmLogger()

    override fun getEventsForFragment(fragmentId: Long): List<FragmentSpansEventCacheModel> {
        val fragmentEvents = mutableListOf<FragmentSpansEventCacheModel>()
        var cursor: Cursor? = null
        kotlin.runCatching {
            cursor = databaseWrapper?.let {
                val selectionClause = APMFragmentEventsEntry.COLUMN_FRAGMENT_ID + " = ?"
                val selectionArgs = arrayOf("$fragmentId")
                it.query(
                    APMFragmentEventsEntry.TABLE_NAME,
                    null,
                    selectionClause,
                    selectionArgs,
                    null,
                    null,
                    null
                )
            }
            while (cursor?.moveToNext() == true) {
                cursor?.toFragmentEvent()?.let(fragmentEvents::add)
            }
        }.also {
            cursor?.close()
        }.onFailure {
            apmLogger.logSDKError("Error while getting fragment events for fragment with id $fragmentId from db due to ${it.message}")
            IBGDiagnostics.reportNonFatal(
                it,
                "Error while getting fragment events for fragment with id $fragmentId from db due to ${it.message}"
            )
        }

        return fragmentEvents
    }

    override fun saveEvents(events: List<FragmentSpansEvent>, fragmentId: Long) {
        events.forEach { event ->
            saveEvent(event, fragmentId)
        }
    }

    private fun saveEvent(fragmentSpansEvent: FragmentSpansEvent, fragmentId: Long) {
        kotlin.runCatching {
            databaseWrapper?.let {
                val values = ContentValues().apply {
                    put(APMFragmentEventsEntry.COLUMN_NAME, fragmentSpansEvent.name)
                    put(APMFragmentEventsEntry.COLUMN_FRAGMENT_ID, fragmentId)
                    put(APMFragmentEventsEntry.COLUMN_START_TIME, fragmentSpansEvent.startTime)
                    put(APMFragmentEventsEntry.COLUMN_DURATION, fragmentSpansEvent.duration)
                }
                it.insert(APMFragmentEventsEntry.TABLE_NAME, null, values)
            }
        }.onFailure {
            apmLogger.logSDKError("Error while inserting fragment event ${fragmentSpansEvent.name} into db due to ${it.message}")
            IBGDiagnostics.reportNonFatal(
                it,
                "Error while inserting fragment event ${fragmentSpansEvent.name} into db due to ${it.message}"
            )
        }
    }

    private fun Cursor.toFragmentEvent() = FragmentSpansEventCacheModel(
        id = getLong(getColumnIndexOrThrow(APMFragmentEventsEntry.COLUMN_ID)),
        fragmentId = getLong(getColumnIndexOrThrow(APMFragmentEventsEntry.COLUMN_FRAGMENT_ID)),
        name = getString(getColumnIndexOrThrow(APMFragmentEventsEntry.COLUMN_NAME)),
        startTime = getLong(getColumnIndexOrThrow(APMFragmentEventsEntry.COLUMN_START_TIME)),
        duration = getLong(getColumnIndexOrThrow(APMFragmentEventsEntry.COLUMN_DURATION))
    )
}