package com.instabug.apm.cache.handler.fragments

import android.content.ContentValues
import android.database.Cursor
import com.instabug.apm.cache.model.FragmentSpansCacheModel
import com.instabug.apm.fragment.model.FragmentSpans
import com.instabug.apm.logger.internal.Logger
import com.instabug.library.SpanIDProvider
import com.instabug.library.diagnostics.IBGDiagnostics
import com.instabug.library.internal.storage.cache.db.DatabaseManager
import com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMFragmentEntry
import com.instabug.library.internal.storage.cache.db.SQLiteDatabaseWrapper

interface FragmentSpansCacheHandler {
    fun saveFragment(fragmentSpans: FragmentSpans): Long?
    fun trimToLimit(limit: Int)
    fun trimToLimit(sessionId: String, limit: Int): Int?
    fun deleteFragmentsForSession(sessionId: String)
    fun getFragmentsForSession(sessionID: String): List<FragmentSpansCacheModel>
    fun clearAll()
    fun migrateDanglingOccurrencesIfPossible(sessionId: String): Int
    fun dropInvalidDanglingOccurrences()
}

class FragmentSpansCacheHandlerImpl(
    private val apmLogger: Logger,
    private val databaseManager: DatabaseManager,
    private val appLaunchIdProvider: SpanIDProvider
) : FragmentSpansCacheHandler {

    private val databaseWrapper: SQLiteDatabaseWrapper
        get() = databaseManager.openDatabase()
    private val currentAppLaunchId: String
        get() = appLaunchIdProvider.spanId

    override fun saveFragment(fragmentSpans: FragmentSpans): Long? =
        runOrLogError("Error while inserting fragment ${fragmentSpans.name} into db due to") {
            databaseWrapper.let {
                val values = ContentValues().apply {
                    put(APMFragmentEntry.COLUMN_NAME, fragmentSpans.name)
                    put(APMFragmentEntry.COLUMN_SESSION_ID, fragmentSpans.sessionId)
                    put(APMFragmentEntry.COLUMN_APP_LAUNCH_ID, currentAppLaunchId)
                }
                it.insert(APMFragmentEntry.TABLE_NAME, null, values)
            }
        }.getOrNull()

    override fun trimToLimit(limit: Int) {
        runOrLogError("Error while trimming apm fragments due to") {
            databaseWrapper.delete(
                APMFragmentEntry.TABLE_NAME,
                STORE_LIMIT_WHERE_CLAUSE,
                arrayOf(limit.toString())
            )
        }
    }

    override fun trimToLimit(sessionId: String, limit: Int): Int? =
        runOrLogError("Error while trimming apm fragments due to") {
            databaseWrapper.delete(
                APMFragmentEntry.TABLE_NAME,
                REQUEST_LIMIT_WHERE_CLAUSE,
                arrayOf(sessionId, sessionId, limit.toString())
            )
        }.getOrNull()

    override fun deleteFragmentsForSession(sessionId: String) {
        runOrLogError("Error while deleting apm fragments for session $sessionId db due to") {
            databaseWrapper.delete(
                APMFragmentEntry.TABLE_NAME,
                WHERE_SESSION_ID,
                arrayOf(sessionId)
            )
        }
    }

    override fun getFragmentsForSession(sessionID: String): List<FragmentSpansCacheModel> {
        val fragments = mutableListOf<FragmentSpansCacheModel>()
        var cursor: Cursor? = null
        kotlin.runCatching {
            cursor = databaseWrapper.query(
                APMFragmentEntry.TABLE_NAME,
                null,
                WHERE_SESSION_ID,
                arrayOf(sessionID),
                null,
                null,
                null
            )
            while (cursor?.moveToNext() == true) {
                cursor?.toFragmentCacheModel()?.let(fragments::add)
            }
        }.also {
            cursor?.close()
        }.onFailure {
            reportAndLogError("Error while getting apm fragments from db db due to", it)
        }
        return fragments.toList()
    }

    override fun clearAll() {
        runOrLogError("Error while deleting apm fragments due to") {
            databaseWrapper.delete(APMFragmentEntry.TABLE_NAME, null, null)
        }
    }

    override fun migrateDanglingOccurrencesIfPossible(sessionId: String): Int =
        runOrLogError("Error while updating dangling apm fragments due to") {
            ContentValues()
                .apply { put(APMFragmentEntry.COLUMN_SESSION_ID, sessionId) }
                .let { values ->
                    databaseWrapper.update(
                        APMFragmentEntry.TABLE_NAME,
                        values,
                        MIGRATE_DANGLING_OCCURRENCES_WHERE_CLAUSE,
                        arrayOf(currentAppLaunchId)
                    )
                }
        }.getOrNull() ?: 0

    override fun dropInvalidDanglingOccurrences() {
        runOrLogError("Error while deleting old dangling apm fragments due to") {
            databaseWrapper.delete(
                APMFragmentEntry.TABLE_NAME,
                DROP_INVALID_DANGLING_OCCURRENCES_WHERE_CLAUSE,
                arrayOf(currentAppLaunchId)
            )
        }
    }

    private fun Cursor.toFragmentCacheModel(): FragmentSpansCacheModel = FragmentSpansCacheModel(
        id = getLong(getColumnIndexOrThrow(APMFragmentEntry.COLUMN_ID)),
        name = getString(getColumnIndexOrThrow(APMFragmentEntry.COLUMN_NAME)),
        sessionId = getLong(getColumnIndexOrThrow(APMFragmentEntry.COLUMN_SESSION_ID))
    )

    private inline fun <T> runOrLogError(errorMessage: String, operation: () -> T): Result<T> =
        kotlin.runCatching { operation() }
            .onFailure { reportAndLogError(errorMessage, it) }

    private fun reportAndLogError(errorMessage: String, throwable: Throwable) {
        val message = "$errorMessage ${throwable.message}"
        apmLogger.logSDKError(message)
        IBGDiagnostics.reportNonFatal(throwable, message)
    }

    companion object {
        private const val WHERE_SESSION_ID =
            APMFragmentEntry.COLUMN_SESSION_ID + " = ?"
        private const val STORE_LIMIT_WHERE_CLAUSE =
            (APMFragmentEntry.COLUMN_ID + " not in (" + " select " + APMFragmentEntry.COLUMN_ID +
                    " from " + APMFragmentEntry.TABLE_NAME +
                    " order by " + APMFragmentEntry.COLUMN_ID + " desc" +
                    " limit ? )"
                    )
        private const val REQUEST_LIMIT_WHERE_CLAUSE = (
                APMFragmentEntry.COLUMN_SESSION_ID + " = ? AND " +
                        APMFragmentEntry.COLUMN_ID + " NOT IN (" +
                        "SELECT " + APMFragmentEntry.COLUMN_ID +
                        " FROM " + APMFragmentEntry.TABLE_NAME +
                        " where " + WHERE_SESSION_ID +
                        " ORDER BY " + APMFragmentEntry.COLUMN_ID + " DESC" +
                        " LIMIT ?" +
                        ")"
                )
        private const val MIGRATE_DANGLING_OCCURRENCES_WHERE_CLAUSE =
            "${APMFragmentEntry.COLUMN_APP_LAUNCH_ID} = ? AND ${APMFragmentEntry.COLUMN_SESSION_ID} IS NULL"
        private const val DROP_INVALID_DANGLING_OCCURRENCES_WHERE_CLAUSE =
            "${APMFragmentEntry.COLUMN_APP_LAUNCH_ID} != ? AND ${APMFragmentEntry.COLUMN_SESSION_ID} IS NULL "
    }
}
