package com.instabug.apm.cache.handler.executiontraces;

import android.annotation.SuppressLint;
import android.content.ContentValues;
import android.database.Cursor;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.collection.ArrayMap;

import com.instabug.apm.cache.model.ExecutionTraceCacheModel;
import com.instabug.apm.constants.ErrorMessages;
import com.instabug.apm.logger.internal.Logger;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.storage.cache.db.DatabaseManager;
import com.instabug.library.internal.storage.cache.db.InstabugDbContract.DanglingExecutionTracesAttributesEntry;
import com.instabug.library.internal.storage.cache.db.InstabugDbContract.DanglingExecutionTracesEntry;
import com.instabug.library.internal.storage.cache.db.SQLiteDatabaseWrapper;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

public class DanglingExecutionTracesCacheHandlerImpl implements DanglingExecutionTracesCacheHandler {

    @Nullable
    private final DatabaseManager databaseManager;
    @NonNull
    private final Logger logger;

    public DanglingExecutionTracesCacheHandlerImpl(@Nullable DatabaseManager databaseManager,
                                                   @NonNull Logger logger) {
        this.databaseManager = databaseManager;
        this.logger = logger;
    }


    @Override
    public boolean insertTrace(@NonNull ExecutionTraceCacheModel trace) {
        if (databaseManager != null) {
            try {
                SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
                ContentValues values = new ContentValues();
                values.put(DanglingExecutionTracesEntry.COLUMN_ID, trace.getId());
                if (trace.getName() != null)
                    values.put(DanglingExecutionTracesEntry.COLUMN_NAME, trace.getName());
                values.put(DanglingExecutionTracesEntry.COLUMN_START_TIME, trace.getStartTime());
                values.put(DanglingExecutionTracesEntry.COLUMN_STARTED_ON_BACKGROUND, trace.startedInBackground() ? 1 : 0);
                values.put(DanglingExecutionTracesEntry.COLUMN_ENDED_ON_BACKGROUND, trace.endedInBackground() ? 1 : 0);
                values.put(DanglingExecutionTracesEntry.COLUMN_DURATION, trace.getDuration());
                long rowId = sqLiteDatabaseWrapper.insert(DanglingExecutionTracesEntry.TABLE_NAME, null, values);
                sqLiteDatabaseWrapper.close();
                return rowId != -1;
            } catch (Throwable t) {
                logger.logSDKError("Inserting execution trace failed: " + t.getMessage(), t);
                IBGDiagnostics.reportNonFatal(t, "Inserting execution trace failed: " + t.getMessage());
            }
        }
        return false;
    }

    @Override
    @Nullable
    public List<ExecutionTraceCacheModel> getEndedTraces(long limit) {
        String tracesQuery = "Select * from " + DanglingExecutionTracesEntry.TABLE_NAME
                + " where " + DanglingExecutionTracesEntry.COLUMN_DURATION + " != -1"
                + " limit " + limit;
        return getTracesByQueryString(tracesQuery);
    }

    @SuppressLint("Range")
    @Nullable
    private List<ExecutionTraceCacheModel> getTracesByQueryString(String tracesQuery) {
        if (databaseManager != null) {
            ArrayList<ExecutionTraceCacheModel> executionTraceCacheModels = new ArrayList<>();
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            Cursor cursor = null;
            try {
                cursor = sqLiteDatabaseWrapper.rawQuery(tracesQuery, null);

                if (cursor != null) {
                    while (cursor.moveToNext()) {
                        ExecutionTraceCacheModel executionTraceCacheModel = new ExecutionTraceCacheModel();
                        executionTraceCacheModel.setId(cursor.getLong(cursor.getColumnIndex(DanglingExecutionTracesEntry.COLUMN_ID)));
                        executionTraceCacheModel.setName(cursor.getString(cursor.getColumnIndex(DanglingExecutionTracesEntry.COLUMN_NAME)));
                        executionTraceCacheModel.setStartTime(cursor.getLong(cursor.getColumnIndex(DanglingExecutionTracesEntry.COLUMN_START_TIME)));
                        executionTraceCacheModel.setDuration(cursor.getLong(cursor.getColumnIndex(DanglingExecutionTracesEntry.COLUMN_DURATION)));
                        executionTraceCacheModel
                                .setStartedInBackground(cursor.getInt(cursor.getColumnIndex(DanglingExecutionTracesEntry.COLUMN_STARTED_ON_BACKGROUND)) == 1);
                        executionTraceCacheModel.setEndedInBackground(cursor.getInt(cursor.getColumnIndex(DanglingExecutionTracesEntry.COLUMN_ENDED_ON_BACKGROUND)) == 1);
                        String attributesQuery = "select * from " + DanglingExecutionTracesAttributesEntry.TABLE_NAME
                                + " where " + DanglingExecutionTracesAttributesEntry.COLUMN_EXECUTION_TRACE_ID + " = " + executionTraceCacheModel.getId();
                        Cursor attributesCursor = sqLiteDatabaseWrapper.rawQuery(attributesQuery, null);
                        if (attributesCursor != null) {
                            Map<String, String> attrsMap = new ArrayMap<>();
                            while (attributesCursor.moveToNext()) {
                                attrsMap.put(attributesCursor.getString(attributesCursor.getColumnIndex(DanglingExecutionTracesAttributesEntry.COLUMN_KEY)),
                                        attributesCursor.getString(attributesCursor.getColumnIndex(DanglingExecutionTracesAttributesEntry.COLUMN_VALUE)));
                            }
                            attributesCursor.close();
                            executionTraceCacheModel.setAttrs(attrsMap);
                        }
                        executionTraceCacheModels.add(executionTraceCacheModel);
                    }
                }
                sqLiteDatabaseWrapper.close();
                return executionTraceCacheModels;
            } catch (Exception ex) {
                logger.logSDKError("DB execution a sql failed: " + ex.getMessage(), ex);
                IBGDiagnostics.reportNonFatal(ex, "DB execution a sql failed: " + ex.getMessage());
            } finally {
                if (cursor != null) {
                    cursor.close();
                }
            }
        }
        return null;
    }

    @Override
    public void removeEndedTraces(long limit) {
        if (databaseManager != null) {
            String deleteTracesQuery = "delete from " + DanglingExecutionTracesEntry.TABLE_NAME + " where " + DanglingExecutionTracesEntry.COLUMN_ID + " in (select "
                    + DanglingExecutionTracesEntry.COLUMN_ID + " from " + DanglingExecutionTracesEntry.TABLE_NAME
                    + " where " + DanglingExecutionTracesEntry.COLUMN_DURATION + " != -1"
                    + " limit " + limit + ")";
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            sqLiteDatabaseWrapper.execSQL(deleteTracesQuery);
            sqLiteDatabaseWrapper.close();
        }
    }

    @Override
    public void setAttribute(long traceId, @NonNull String traceName, @NonNull String key, @Nullable String value) {
        addAttribute(traceId, key, value);
    }

    private void updateAttribute(long traceId, String key, String value) {
        if (databaseManager != null) {
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            ContentValues contentValues = new ContentValues();
            contentValues.put(DanglingExecutionTracesAttributesEntry.COLUMN_VALUE, value);
            String whereClause = DanglingExecutionTracesAttributesEntry.COLUMN_EXECUTION_TRACE_ID + " = ? AND "
                    + DanglingExecutionTracesAttributesEntry.COLUMN_KEY + "= ?";
            String[] whereValues = {traceId + "", key};
            sqLiteDatabaseWrapper.update(DanglingExecutionTracesAttributesEntry.TABLE_NAME, contentValues, whereClause, whereValues);
            sqLiteDatabaseWrapper.close();
        }
    }

    @SuppressLint("Range")
    @Nullable
    public Map<String, String> getTraceAttributes(long traceId) {
        if (databaseManager != null) {
            Map<String, String> attributes = new ArrayMap<>();
            String whereClause = DanglingExecutionTracesAttributesEntry.COLUMN_EXECUTION_TRACE_ID + " = ?";
            String[] whereValues = {traceId + ""};
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            Cursor cursor = null;
            try {
                cursor = sqLiteDatabaseWrapper.query(DanglingExecutionTracesAttributesEntry.TABLE_NAME, null
                        , whereClause, whereValues, null, null, null);

                if (cursor != null) {
                    while (cursor.moveToNext()) {
                        attributes.put(cursor.getString(cursor.getColumnIndex(DanglingExecutionTracesAttributesEntry.COLUMN_KEY)),
                                cursor.getString(cursor.getColumnIndex(DanglingExecutionTracesAttributesEntry.COLUMN_VALUE)));
                    }
                    cursor.close();
                }
                sqLiteDatabaseWrapper.close();
                return attributes;
            } catch (Exception ex) {
                logger.logSDKError("DB execution a sql failed: " + ex.getMessage(), ex);
                IBGDiagnostics.reportNonFatal(ex, "DB execution a sql failed: " + ex.getMessage());
            } finally {
                if (cursor != null) {
                    cursor.close();
                }
            }
        }
        return null;
    }

    @Override
    public void removeAll() {
        if (databaseManager != null) {
            String removeTracesQuery = "delete from " + DanglingExecutionTracesEntry.TABLE_NAME;
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            sqLiteDatabaseWrapper.execSQL(removeTracesQuery);
            sqLiteDatabaseWrapper.close();
        }
    }

    @Override
    public void removeUnEndedTraces() {
        if (databaseManager != null) {
            String selectTracesQuery = "select * from " + DanglingExecutionTracesEntry.TABLE_NAME + " where " + DanglingExecutionTracesEntry.COLUMN_DURATION + " = " + DanglingExecutionTracesEntry.DURATION_DEFAULT_VALUE;
            List<ExecutionTraceCacheModel> unEndedTraces = getTracesByQueryString(selectTracesQuery);
            if (unEndedTraces != null) {
                for (ExecutionTraceCacheModel trace : unEndedTraces) {
                    if (trace.getName() != null) {
                        logger.logSDKError(ErrorMessages.UN_ENDED_TRACE_NOT_SAVED.replace("$s", trace.getName()));
                    }
                }
            }
            String deleteTracesQuery = "delete from " + DanglingExecutionTracesEntry.TABLE_NAME + " where " + DanglingExecutionTracesEntry.COLUMN_DURATION + " = " + DanglingExecutionTracesEntry.DURATION_DEFAULT_VALUE;
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            sqLiteDatabaseWrapper.execSQL(deleteTracesQuery);
            sqLiteDatabaseWrapper.close();
        }
    }

    @Override
    public boolean exists(long traceId) {
        if (databaseManager != null) {
            String whereClause = DanglingExecutionTracesEntry.COLUMN_ID + " = ? ";
            String[] whereValues = {String.valueOf(traceId)};
            SQLiteDatabaseWrapper database = databaseManager.openDatabase();
            Cursor cursor = null;
            try {
                cursor = database.query(DanglingExecutionTracesEntry.TABLE_NAME, null, whereClause, whereValues, null, null, null);
                if (cursor != null && cursor.getCount() > 0) {
                    return true;
                }
            } finally {
                if (cursor != null) {
                    cursor.close();
                }
                database.close();
            }
        }
        return false;
    }

    private void addAttribute(long traceId, @NonNull String key, @Nullable String value) {
        if (databaseManager != null) {
            ContentValues contentValues = new ContentValues();
            contentValues.put(DanglingExecutionTracesAttributesEntry.COLUMN_EXECUTION_TRACE_ID, traceId);
            contentValues.put(DanglingExecutionTracesAttributesEntry.COLUMN_KEY, key);
            if (value != null)
                contentValues.put(DanglingExecutionTracesAttributesEntry.COLUMN_VALUE, value);
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            sqLiteDatabaseWrapper.insert(DanglingExecutionTracesAttributesEntry.TABLE_NAME, null, contentValues);
            sqLiteDatabaseWrapper.close();
        }
    }

    private void removeAttribute(long traceId, String key) {
        if (databaseManager != null) {
            String query = "delete from " + DanglingExecutionTracesAttributesEntry.TABLE_NAME + " where "
                    + DanglingExecutionTracesAttributesEntry.COLUMN_EXECUTION_TRACE_ID + " = " + traceId
                    + " and " + DanglingExecutionTracesAttributesEntry.COLUMN_KEY + " = \"" + key + "\"";
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            sqLiteDatabaseWrapper.execSQL(query);
            sqLiteDatabaseWrapper.close();
        }
    }
}
