package com.instabug.apm.cache.handler.applaunch;

import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AppLaunchEntry.COLUMN_DURATION;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AppLaunchEntry.COLUMN_ID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AppLaunchEntry.COLUMN_NAME;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AppLaunchEntry.COLUMN_SCREEN_NAME;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AppLaunchEntry.COLUMN_SESSION_ID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AppLaunchEntry.COLUMN_START_TIME;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AppLaunchEntry.TABLE_NAME;

import android.annotation.SuppressLint;
import android.content.ContentValues;
import android.database.Cursor;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.collection.ArrayMap;

import com.instabug.apm.cache.model.AppLaunchCacheModel;
import com.instabug.apm.constants.AppLaunchType;
import com.instabug.apm.constants.Constants;
import com.instabug.apm.di.ServiceLocator;
import com.instabug.apm.logger.internal.Logger;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.storage.cache.db.DatabaseManager;
import com.instabug.library.internal.storage.cache.db.InstabugDbContract;
import com.instabug.library.internal.storage.cache.db.SQLiteDatabaseWrapper;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

public class AppLaunchCacheHandlerImpl implements AppLaunchCacheHandler {

    @Nullable
    private DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
    private Logger apmLogger = ServiceLocator.getApmLogger();
    private String[] updatableStagesKeys = new String[] {
            Constants.AppLaunch.END_APP_LAUNCH_DURATION,
            Constants.AppLaunch.END_APP_LAUNCH_TIME_STAMP
    };

    @Override
    public long insertAppLaunch(String sessionID, AppLaunchCacheModel appLaunchCacheModel) {
        if (databaseManager != null && appLaunchCacheModel != null) {
            apmLogger.logSDKProtected("inserting app launch");
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            ContentValues values = new ContentValues();
            values.put(COLUMN_SESSION_ID, sessionID);
            if (appLaunchCacheModel.getType() != null) {
                values.put(COLUMN_NAME, appLaunchCacheModel.getType());
            }
            if (appLaunchCacheModel.getScreenName() != null) {
                values.put(COLUMN_SCREEN_NAME, appLaunchCacheModel.getScreenName());
            }
            values.put(COLUMN_START_TIME, appLaunchCacheModel.getStartTime());
            values.put(COLUMN_DURATION, appLaunchCacheModel.getDuration());
            long traceId = sqLiteDatabaseWrapper.insert(TABLE_NAME, null, values);
            if (appLaunchCacheModel.getStages() != null) {
                for (Map.Entry<String, String> entry : appLaunchCacheModel.getStages().entrySet()) {
                    ContentValues contentValues = new ContentValues();
                    contentValues.put(InstabugDbContract.AppLaunchAttributesEntry.COLUMN_APP_LAUNCH_ID, traceId);
                    contentValues.put(InstabugDbContract.AppLaunchAttributesEntry.COLUMN_KEY, entry.getKey());
                    contentValues.put(InstabugDbContract.AppLaunchAttributesEntry.COLUMN_VALUE, entry.getValue());
                    sqLiteDatabaseWrapper.insert(InstabugDbContract.AppLaunchAttributesEntry.TABLE_NAME,
                            null,
                            contentValues);
                }
            }
            sqLiteDatabaseWrapper.close();
            apmLogger.logSDKProtected("inserting app launch done with id " + traceId);
            return traceId;
        }
        return -1;
    }

    @Nullable
    @Override
    public List<AppLaunchCacheModel> getAppLaunches(long limit) {
        String tracesQuery = "Select * from " + TABLE_NAME
                + " where " + InstabugDbContract.AppLaunchEntry.COLUMN_DURATION + " != -1"
                + " limit " + limit;
        return getTracesByQueryString(tracesQuery);
    }

    @Nullable
    private List<AppLaunchCacheModel> getTracesByQueryString(String tracesQuery) {
        if (databaseManager != null) {
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            Cursor cursor = null;
            try {
                cursor = sqLiteDatabaseWrapper.rawQuery(tracesQuery, null);
                ArrayList<AppLaunchCacheModel> cacheModels = getAppLaunchesFromCursor(sqLiteDatabaseWrapper, cursor);
                sqLiteDatabaseWrapper.close();
                return cacheModels;
            } catch (Exception ex) {
                apmLogger.logSDKError("DB execution a sql failed: " + ex.getMessage(), ex);
                IBGDiagnostics.reportNonFatal(ex, "Error while querying traces: " + ex.getMessage());
            } finally {
                if (cursor != null) {
                    cursor.close();
                }
            }
        }
        return null;
    }

    @SuppressLint("Range")
    private ArrayList<AppLaunchCacheModel> getAppLaunchesFromCursor(SQLiteDatabaseWrapper sqLiteDatabaseWrapper,
                                                                    @Nullable Cursor cursor) {
        ArrayList<AppLaunchCacheModel> cacheModels = new ArrayList<>();
        if (cursor != null) {
            while (cursor.moveToNext()) {
                AppLaunchCacheModel appLaunchCacheModel = new AppLaunchCacheModel();
                appLaunchCacheModel.setId(cursor.getLong(cursor.getColumnIndex(COLUMN_ID)));
                appLaunchCacheModel.setType(cursor.getString(cursor.getColumnIndex(COLUMN_NAME)));
                appLaunchCacheModel.setScreenName(cursor.getString(cursor.getColumnIndex(COLUMN_SCREEN_NAME)));
                appLaunchCacheModel.setStartTime(cursor.getLong(cursor.getColumnIndex(COLUMN_START_TIME)));
                appLaunchCacheModel.setDuration(cursor.getLong(cursor.getColumnIndex(COLUMN_DURATION)));
                appLaunchCacheModel.setStages(getAppLaunchStages(appLaunchCacheModel.getId(), sqLiteDatabaseWrapper));
                cacheModels.add(appLaunchCacheModel);
            }
        }
        return cacheModels;
    }

    @SuppressLint("Range")
    @Nullable
    public Map<String, String> getAppLaunchStages(long id, @Nullable SQLiteDatabaseWrapper sqLiteDatabaseWrapper) {
        if (databaseManager != null) {

            // Check if this method needs to manage the db connection itself
            boolean isHandledDBConnection = sqLiteDatabaseWrapper != null;
            if (!isHandledDBConnection) {
                sqLiteDatabaseWrapper = databaseManager.openDatabase();
            }
            Map<String, String> stages = new ArrayMap<>();
            String attributesQuery = "select * from " + InstabugDbContract.AppLaunchAttributesEntry.TABLE_NAME
                    + " where " + InstabugDbContract.AppLaunchAttributesEntry.COLUMN_APP_LAUNCH_ID + " = " + id;
            if (sqLiteDatabaseWrapper != null) {
                Cursor attributesCursor = sqLiteDatabaseWrapper.rawQuery(attributesQuery, null);
                if (attributesCursor != null) {
                    while (attributesCursor.moveToNext()) {
                        stages.put(attributesCursor.getString(attributesCursor.getColumnIndex(InstabugDbContract.AppLaunchAttributesEntry.COLUMN_KEY)),
                                attributesCursor.getString(attributesCursor.getColumnIndex(InstabugDbContract.AppLaunchAttributesEntry.COLUMN_VALUE)));
                    }
                    attributesCursor.close();
                }
            }

            if (!isHandledDBConnection && sqLiteDatabaseWrapper != null) {
                sqLiteDatabaseWrapper.close();
            }
            return stages;
        }
        return null;
    }

    @Override
    public void removeAppLaunches(long limit) {
        if (databaseManager != null) {
            String deleteTracesQuery = "delete from " + TABLE_NAME + " where " + COLUMN_ID + " in (select "
                    + COLUMN_ID + " from " + TABLE_NAME + " where "
                    + InstabugDbContract.AppLaunchEntry.COLUMN_DURATION + " != -1"
                    + " limit " + limit + ")";
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            sqLiteDatabaseWrapper.execSQL(deleteTracesQuery);
            sqLiteDatabaseWrapper.close();
        }
    }

    @Override
    public void removeAppLaunches(@AppLaunchType String type, long limit) {
        if (databaseManager != null) {
            //DELETE FROM demo WHERE ID IN( SELECT ID FROM demo WHERE id < 10 LIMIT 4)
            String selectLimit = "SELECT "
                    + COLUMN_ID + " FROM " + TABLE_NAME + " WHERE "
                    + COLUMN_DURATION + " != -1 AND " + COLUMN_NAME + " = \'" + type
                    + "\' LIMIT " + limit;
            String deleteTracesQuery = "DELETE FROM " + TABLE_NAME + " WHERE " + COLUMN_ID + " IN (" + selectLimit + ")";
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            sqLiteDatabaseWrapper.execSQL(deleteTracesQuery);
            sqLiteDatabaseWrapper.close();
        }
    }

    @Override
    public void removeAppLaunches(@NonNull @AppLaunchType String type) {
        if (databaseManager != null) {
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            sqLiteDatabaseWrapper.delete(TABLE_NAME,
                    COLUMN_NAME + " = ? ",
                    new String[]{type});
            sqLiteDatabaseWrapper.close();
        }
    }

    @Override
    public void removeAll() {
        if (databaseManager != null) {
            String removeTracesQuery = "delete from " + TABLE_NAME;
            String removeAttributesQuery = "delete from " + InstabugDbContract.AppLaunchAttributesEntry.TABLE_NAME;
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            sqLiteDatabaseWrapper.execSQL(removeAttributesQuery);
            sqLiteDatabaseWrapper.execSQL(removeTracesQuery);
            sqLiteDatabaseWrapper.close();
        }
    }

    @Nullable
    @Override
    public List<AppLaunchCacheModel> getAppLaunchesForSession(String sessionID) {
        if (databaseManager != null) {
            SQLiteDatabaseWrapper databaseWrapper = databaseManager.openDatabase();
            String selectionClause = COLUMN_SESSION_ID + " = ?";
            String[] selectionArgs = {sessionID};
            Cursor cursor = null;
            try {
                cursor = databaseWrapper.query(TABLE_NAME,
                        null, selectionClause, selectionArgs,
                        null, null, null);
                ArrayList<AppLaunchCacheModel> cacheModels = getAppLaunchesFromCursor(databaseWrapper, cursor);
                databaseWrapper.close();
                return cacheModels;
            } catch (Exception ex) {
                apmLogger.logSDKError("Error while getting app launches for session: " + ex.getMessage(), ex);
                IBGDiagnostics.reportNonFatal(ex, "Error while getting app launches for session: " + ex.getMessage());
            } finally {
                if (cursor != null) {
                    cursor.close();
                }
            }

        }
        return null;
    }

    @Override
    public void trimAppLaunchesToLimit(long limit, String appLaunchType) {
        if (databaseManager != null) {
            String selectByLimitDescendingQuery = "SELECT " + COLUMN_ID
                    + " FROM " + TABLE_NAME
                    + " WHERE " + COLUMN_NAME + " = ?"
                    + " ORDER BY " + COLUMN_ID + " DESC"
                    + " LIMIT ? OFFSET ?";

            String whereClause = COLUMN_ID + " IN (" + selectByLimitDescendingQuery + ")";
            String[] whereArgs = { appLaunchType, "-1", String.valueOf(limit) };

            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            try {
                sqLiteDatabaseWrapper.delete(TABLE_NAME, whereClause, whereArgs);
            } catch (Exception ex) {
                apmLogger.logSDKError("Error while trimming app launches: " + ex.getMessage(), ex);
                IBGDiagnostics.reportNonFatal(ex, "Error while trimming app launches: " + ex.getMessage());
            }
        }
    }

    @Override
    public int trimAppLaunchesToLimit(String sessionID, long limit, String appLaunchType) {
        if (databaseManager != null) {
            String selectByLimitDescendingQuery = "SELECT " + COLUMN_ID
                    + " FROM " + TABLE_NAME
                    + " where " + COLUMN_SESSION_ID + " = ? AND " + COLUMN_NAME + " = ?"
                    + " ORDER BY " + COLUMN_ID + " DESC"
                    + " LIMIT ?";

            String whereClause = COLUMN_SESSION_ID + " = ? AND " + COLUMN_NAME + " = ? AND "
                    + COLUMN_ID + " NOT IN (" + selectByLimitDescendingQuery + ")";
            String[] whereArgs = {sessionID, appLaunchType, sessionID, appLaunchType, String.valueOf(limit)};
            try {
                SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
                return sqLiteDatabaseWrapper.delete(TABLE_NAME, whereClause, whereArgs);
            } catch (Exception ex) {
                apmLogger.logSDKError("Error while trimming app launches: " + ex.getMessage(), ex);
                IBGDiagnostics.reportNonFatal(ex, "Error while trimming app launches: " + ex.getMessage());
            }
        }
        return -1;
    }

    @Override
    public void updateAppLaunch(AppLaunchCacheModel appLaunch) {
        if (databaseManager != null && appLaunch != null) {
            apmLogger.logSDKProtected("updating app launch");
            SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
            ContentValues values = new ContentValues();
            // 1. Update the screen name
            if (appLaunch.getScreenName() != null) {
                values.put(COLUMN_SCREEN_NAME, appLaunch.getScreenName());
            }
            // 2. Update the duration
            values.put(COLUMN_DURATION, appLaunch.getDuration());
            String whereClause = COLUMN_ID + " = ?";
            String[] whereArgs = {String.valueOf(appLaunch.getId())};
            // 3. Update the AppLaunch entry
            sqLiteDatabaseWrapper.update(TABLE_NAME, values, whereClause, whereArgs);
            // 4. Add the new stage
            Map<String, String> stages = appLaunch.getStages();
            if (stages != null) {
                for (String key : updatableStagesKeys) {
                    if (stages.containsKey(key)) {
                        ContentValues contentValues = new ContentValues();
                        contentValues.put(InstabugDbContract.AppLaunchAttributesEntry.COLUMN_APP_LAUNCH_ID, appLaunch.getId());
                        contentValues.put(InstabugDbContract.AppLaunchAttributesEntry.COLUMN_KEY, key);
                        contentValues.put(InstabugDbContract.AppLaunchAttributesEntry.COLUMN_VALUE, stages.get(key));
                        sqLiteDatabaseWrapper.insert(InstabugDbContract.AppLaunchAttributesEntry.TABLE_NAME,
                                null,
                                contentValues);
                    }
                }
            }
            sqLiteDatabaseWrapper.close();
            apmLogger.logSDKProtected("updating app launch done with id: " + appLaunch.getId());
        }
    }
}
