package com.instabug.apm.attributes.listeners;

import com.instabug.apm.attributes.predicates.UrlPredicate;
import com.instabug.apm.model.NetworkTrace;

import java.util.Map;

import androidx.annotation.Nullable;

/**
 A listener that fires after a network trace finishes.
 Adds attributes map to the network trace after matching the trace url by the passed predicate.
 */
public abstract class OnNetworkTraceListener {

    @Nullable private final UrlPredicate predicate;

    /**
     Accepts {@link UrlPredicate} to match before performing next action (Adding attributes to a trace)

     Passing null {@link UrlPredicate}, action will be done on each and every url (request)

     Passing a {@link UrlPredicate}, action will be done on all urls (requests) that match the predicate

     @param predicate a predicate to match before performing next action
     */
    public OnNetworkTraceListener(@Nullable UrlPredicate predicate) {
        this.predicate = predicate;
    }

    /**
     Gets the action predicate to be checked before performing the action

     @return urlPredicate
     */
    @Nullable
    public UrlPredicate getPredicate() {
        return predicate;
    }

    /**
     Adds the returned attributes map to the network trace.

     @param trace is the network trace that occurred and should include the returned attribute map.

     @return Map of attributes {key, value} to be added to network trace.
     */
    public abstract Map<String, String> addAttributesOnFinish(NetworkTrace trace);
}
