package com.instabug.apm

import androidx.annotation.WorkerThread
import com.instabug.apm.cache.model.SessionCacheModel
import com.instabug.apm.constants.AppLaunchType
import com.instabug.apm.di.Provider
import com.instabug.apm.di.ServiceLocator
import com.instabug.apm.handler.session.SessionHandler
import com.instabug.apm.networking.mapping.sessions.SessionModelFiller
import com.instabug.library.model.v3Session.IBGSessionData
import com.instabug.library.sessionV3.providers.FeatureSessionDataController
import com.instabug.library.sessionV3.providers.SessionID
import com.instabug.library.sessionreplay.model.SRData
import com.instabug.library.sessionreplay.model.SessionMetadata

class APMSessionDataController(
    private val sessionModelFillersProvider: Provider<Array<SessionModelFiller?>>?
) : FeatureSessionDataController {
    private val sessionMapper get() = ServiceLocator.getSessionMapper()
    private val sessionHandler get() = ServiceLocator.getSessionHandler()
    private val sessionMetaDataCacheHandler get() = ServiceLocator.getSessionMetaDataCacheHandler()
    private val appLaunchesHandler get() = ServiceLocator.getAppLaunchesHandler()
    private val networkLogHandler get() = ServiceLocator.getNetworkLogHandler()
    private val executionTracesHandler get() = ServiceLocator.getExecutionTracesHandler()
    private val uiTraceHandler get() = ServiceLocator.getUiTraceCacheHandler()
    private val fragmentSpansHandler get() = ServiceLocator.getFragmentSpansHandler()

    @WorkerThread
    override fun collectSessionsData(sessionsIds: List<SessionID>): Map<SessionID, IBGSessionData> =
        sessionHandler
            .takeUnless { sessionsIds.isEmpty() }
            ?.getSessionsByCoreIds(sessionsIds)
            ?.onEach { session -> session.fillSessionModel() }
            ?.let(sessionMapper::toSessionData)
            .orEmpty()

    private fun SessionCacheModel.fillSessionModel() {
        sessionMetaData = sessionMetaDataCacheHandler?.getSessionMetaData(id)
        appLaunches = appLaunchesHandler.getAppLaunchesForSession(id)
        networkLogs = networkLogHandler.getEndedNetworkLogsForSession(id)
        executionTraces = executionTracesHandler.getExecutionTracesForSession(id)
        uiTraces = uiTraceHandler.getUiTracesForSession(id)
        fragmentSpans = fragmentSpansHandler?.getFragmentsForSession(id)
        sessionModelFillersProvider?.invoke()
            ?.forEach { it?.fill(sessionId = id, sessionCacheModel = this) }
    }

    @WorkerThread
    override fun dropSessionData(sessionsIds: List<SessionID>) {
        sessionHandler.deleteSessionsByCoreIds(sessionsIds)
    }


    override fun collectSessionReplayData(sessionId: SessionID): SRData? {
        val apmSessionId =
            sessionHandler.getSessionIdByCoreId(sessionId)
        return apmSessionId?.let {
            val networkLogs = getSessionMetaDataNetworkLogs(it)
            val (launchType, launchDuration) = getSessionAppLaunch(it)
            SRData.ApmSRMetadata(
                launchType = launchType,
                launchDuration = launchDuration,
                networkLogs = networkLogs
            )
        }
    }

    private fun SessionHandler.getSessionIdByCoreId(sessionId: String) =
        getSessionsByCoreIds(listOf(sessionId))
            .takeIf { it.isNotEmpty() }?.first()?.id

    private fun getSessionMetaDataNetworkLogs(sessionId: SessionID): List<SessionMetadata.NetworkLog>? =
        networkLogHandler.getEndedNetworkLogsForSession(sessionId)
            ?.map {
                SessionMetadata.NetworkLog(it.url, it.totalDuration, it.responseCode)
            }?.toList()

    private fun getSessionAppLaunch(sessionId: String) : Pair<@SessionMetadata.LaunchType String?, Long?> {
        val appLaunch =
            appLaunchesHandler.getAppLaunchesForSession(sessionId)
                ?.takeIf { it.isNotEmpty() }?.first()
        @SessionMetadata.LaunchType var launchType: String? = null
        var launchDuration: Long? = null
        appLaunch?.let {
            launchType = when (it.type) {
                AppLaunchType.COLD -> SessionMetadata.LaunchType.COLD
                AppLaunchType.WARM -> SessionMetadata.LaunchType.WARM
                AppLaunchType.HOT -> SessionMetadata.LaunchType.HOT
                else -> null
            }
            launchDuration = it.duration
        }
        return Pair(launchType, launchDuration)
    }
}