package com.instabug.apm;

import static com.instabug.apm.constants.Constants.SESSION_PURGING_THREAD_EXECUTOR;

import android.annotation.SuppressLint;
import android.app.Application;
import android.content.Context;
import android.os.Build;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.annotation.WorkerThread;

import com.instabug.apm.appflow.di.AppFlowServiceLocator;
import com.instabug.apm.appflow.manager.AppFlowManager;
import com.instabug.apm.cache.handler.executiontraces.ExecutionTracesMigrationHandler;
import com.instabug.apm.cache.handler.networklog.NetworkLogMigrationHandler;
import com.instabug.apm.compose.compose_spans.ComposeSpansManager;
import com.instabug.apm.compose.compose_spans.ComposeSpansServiceLocator;
import com.instabug.apm.configuration.APMConfigurationHandler;
import com.instabug.apm.configuration.APMConfigurationProvider;
import com.instabug.apm.constants.Constants;
import com.instabug.apm.di.ServiceLocator;
import com.instabug.apm.eventbus.APMSdkStateEventBus;
import com.instabug.apm.handler.executiontraces.ExecutionTracesHandler;
import com.instabug.apm.handler.networklog.NetworkLogHandler;
import com.instabug.apm.handler.session.APMSessionObserver;
import com.instabug.apm.handler.session.APMUncaughtExceptionHandler;
import com.instabug.apm.handler.session.SessionHandler;
import com.instabug.apm.handler.session.SessionObserverHandler;
import com.instabug.apm.handler.uitrace.customuitraces.CustomUiTraceHandler;
import com.instabug.apm.lifecycle.ActivityCallbacks;
import com.instabug.apm.logger.internal.Logger;
import com.instabug.apm.model.EventTimeMetricCapture;
import com.instabug.apm.screenloading.di.ScreenLoadingServiceLocator;
import com.instabug.apm.screenloading.manager.ScreenLoadingManager;
import com.instabug.apm.sync.APMSyncManager;
import com.instabug.apm.v3_session_data_readiness.APMSessionReadinessManager;
import com.instabug.apm.uitrace.di.UiTracesServiceLocator;
import com.instabug.apm.uitrace.manager.UiTracesManager;
import com.instabug.apm.webview.webview_trace.manager.WebViewTraceManager;
import com.instabug.library.BuildFieldsProvider;
import com.instabug.library.InstabugState;
import com.instabug.library.Platform;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.eventbus.InstabugStateEventBus;
import com.instabug.library.core.eventbus.NDKSessionCrashedEvent;
import com.instabug.library.core.eventbus.coreeventbus.IBGCoreEventSubscriber;
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent;
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent.FeaturesFetched;
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent.V3Session;
import com.instabug.library.core.eventbus.eventpublisher.IBGCompositeDisposable;
import com.instabug.library.core.eventbus.eventpublisher.IBGDisposable;
import com.instabug.library.core.plugin.Plugin;
import com.instabug.library.model.common.Session;
import com.instabug.library.model.common.SessionVersion;
import com.instabug.library.sessionV3.providers.FeatureSessionDataController;
import com.instabug.library.sessionV3.providers.FeatureSessionDataControllerHost;
import com.instabug.library.sessionV3.providers.FeatureSessionLazyDataProvider;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.threading.PoolProvider;

import java.util.List;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;

import io.reactivexport.disposables.CompositeDisposable;
import io.reactivexport.functions.Consumer;

public class APMPlugin extends Plugin implements APMSessionObserver, FeatureSessionDataControllerHost, FeatureSessionLazyDataProvider {
    private boolean isFirstLaunch = false;
    private final SessionHandler sessionHandler;
    private final Logger apmLogger;
    @Nullable
    private CompositeDisposable sdkCoreEventsSubscriberDisposable;
    @Nullable
    IBGCompositeDisposable compositeDisposable;
    @Nullable
    @VisibleForTesting
    IBGDisposable apmSdkStateObserver;


    public static final Object lock = new Object();

    public APMPlugin() {
        sessionHandler = ServiceLocator.getSessionHandler();
        apmLogger = ServiceLocator.getApmLogger();
    }


    @Override
    public void init(Context context) {
        super.init(context);
        registerDataReadinessHandlers();
        handleAppFlowsAppLaunch();
    }

    private void handleAppFlowsAppLaunch() {
        AppFlowManager appFlowManager = AppFlowServiceLocator.INSTANCE.getManager();
        if (appFlowManager != null) appFlowManager.onNewAppLaunch();
    }

    private static void registerDataReadinessHandlers() {
        APMSessionReadinessManager[] managers = ServiceLocator.getAPMSessionReadinessManagersProvider().invoke();
        if (managers == null) return;
        for (APMSessionReadinessManager manager : managers) {
            if (manager != null) manager.registerReadinessHandler();
        }
    }

    @Override
    public void wake() {
        //Don't do anything if APM is not enabled
        APMConfigurationProvider configurationProvider = ServiceLocator.getApmConfigurationProvider();
        if (configurationProvider.isAPMEnabled()) {
            if (isFirstLaunch) {
                clearInvalidCache();
                isFirstLaunch = false;
            }
        }
        Session coreSession = InstabugCore.getRunningSession();
        if (shouldDependOnV3Session(configurationProvider, coreSession)) {
            apmLogger.d("v2 trying to start apm session while v3 enabled.. skipping");
            return;
        }
        startApmSessionAndMetrics(coreSession);
    }

    private void startApmSessionAndMetrics(@Nullable Session coreSession) {
        if (coreSession != null) {

            SessionObserverHandler.register(this);
            startSession(coreSession);
            registerSessionCrashHandler();
            registerFragmentLifecycleEventListener();
            startComposeSpansManager();
            startAppFlowManager();
            startUiTracesManager();
            startScreenLoadingManager();
            registerAPMSdkStateEventBus();
        } else {
            apmLogger.logSDKError("APM session not created. Core session is null");
        }
    }

    private boolean shouldDependOnV3Session(APMConfigurationProvider configurationProvider, @Nullable Session coreSession) {
        return coreSession != null &&
                coreSession.getVersion().equals(SessionVersion.V2) &&
                configurationProvider.shouldDependOnV3Session();
    }

    @VisibleForTesting
    void registerAPMSdkStateEventBus() {
        if (apmSdkStateObserver == null) {
            apmSdkStateObserver = APMSdkStateEventBus.INSTANCE.subscribe((enabled -> {
                if (enabled) {
                    registerFragmentLifecycleEventListener();
                } else {
                    unRegisterFragmentLifecycleEventListener();
                }
                handleComposeSpansStateChange();
                handleUiTracesStateChanged();
                handleScreenLoadingStateChanged();
                handleWebViewTracesStateChange();
                handleAppFlowStateChange();
            }));
        }
    }

    private void unRegisterApmSDKStateEventBus() {
        if (apmSdkStateObserver != null) {
            apmSdkStateObserver.dispose();
            apmSdkStateObserver = null;
        }
    }

    @Override
    public void stop() {
        if (sdkCoreEventsSubscriberDisposable != null &&
                !sdkCoreEventsSubscriberDisposable.isDisposed()) {
            sdkCoreEventsSubscriberDisposable.dispose();
        }
        if (compositeDisposable != null) compositeDisposable.dispose();
    }

    private void registerFragmentLifecycleEventListener() {
        ServiceLocator.getFragmentSpansHelper().startFragmentsLifecycleCapturing();
    }

    private void startComposeSpansManager() {
        ComposeSpansManager manager =
                ComposeSpansServiceLocator.INSTANCE.getComposeSpansManager();
        if (manager != null) {
            manager.start();
        }
    }

    private void startUiTracesManager() {
        UiTracesManager manager = UiTracesServiceLocator.getManager();
        if (manager != null) manager.start();
    }

    private void startScreenLoadingManager() {
        ScreenLoadingManager manager = ScreenLoadingServiceLocator.getManager();
        if (manager != null) manager.start();
    }

    private void handleComposeSpansStateChange() {
        ComposeSpansManager manager =
                ComposeSpansServiceLocator.INSTANCE.getComposeSpansManager();
        if (manager != null) {
            manager.onStateChanged();
        }
    }

    private void handleUiTracesStateChanged() {
        UiTracesManager manager = UiTracesServiceLocator.getManager();
        if (manager != null) manager.onStateChanged();
    }

    private void handleScreenLoadingStateChanged() {
        ScreenLoadingManager manager = ScreenLoadingServiceLocator.getManager();
        if (manager != null) manager.onStateChanged();
    }

    private void handleWebViewTracesStateChange() {
        WebViewTraceManager manager = ServiceLocator.getWebViewTraceManager();
        if (manager != null) {
            manager.onStateChanged();
        }
    }

    private void startAppFlowManager() {
        AppFlowManager manager = AppFlowServiceLocator.INSTANCE.getManager();
        if (manager != null) {
            manager.start();
        }
    }

    private void handleAppFlowStateChange() {
        AppFlowManager manager = AppFlowServiceLocator.INSTANCE.getManager();
        if (manager != null) {
            manager.onStateChanged();
        }
    }

    private void unRegisterFragmentLifecycleEventListener() {
        ServiceLocator.getFragmentSpansHelper().stopFragmentsLifecycleCapturing();
    }

    private void registerSessionCrashHandler() {
        if (ServiceLocator.getApmConfigurationProvider().isCrashDetectionEnabled()) {
            if (!(Thread.getDefaultUncaughtExceptionHandler() instanceof APMUncaughtExceptionHandler)) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "setting Uncaught Exception Handler APMUncaughtExceptionHandler");
                Thread.setDefaultUncaughtExceptionHandler(new APMUncaughtExceptionHandler());
            }
        }
    }

    @Override
    public void sleep() {
        unRegisterApmSDKStateEventBus();
        endSession();
    }

    private void subscribeToSDKState() {
        if (sdkCoreEventsSubscriberDisposable == null || sdkCoreEventsSubscriberDisposable.isDisposed()) {
            sdkCoreEventsSubscriberDisposable = new CompositeDisposable();
        }
        sdkCoreEventsSubscriberDisposable.add(InstabugStateEventBus.getInstance().getEventObservable().subscribe(
                new Consumer<InstabugState>() {
                    @Override

                    public void accept(InstabugState instabugState) throws Exception {
                        if (instabugState == InstabugState.DISABLED) {
                            apmLogger.logSDKDebug("Instabug is disabled, purging APM data…");
                            /** Set flag to fire APM sync after ending the current session
                             *
                             * @see {@link APMPlugin#purgeData()}
                             */
                            stopRunningMetrics();
                            endSession();
                            purgeData();
                        }
                    }
                }
        ));
    }

    /**
     * Resets the last sync time of APM data to be able to request an immediate sync.
     */
    private void purgeData() {
        APMConfigurationProvider apmConfigurationProvider = ServiceLocator.getApmConfigurationProvider();
        apmConfigurationProvider.setLastSyncTime(-1L);
        // Getting the sync status before the executor as it might be changed by the time executor executes.
        // i.e. isAPMEnabled returns false as the SDK has been already disabled.
        final APMSyncManager apmSyncManager = ServiceLocator.getApmSyncManager();
        final boolean shouldSync = apmSyncManager.shouldSync();
        Executor executor = ServiceLocator.getSingleThreadPoolExecutor(SESSION_PURGING_THREAD_EXECUTOR);
        executor.execute(new Runnable() {
            @Override
            public void run() {
                apmSyncManager.start(shouldSync);
            }
        });
    }

    @SuppressLint("NewApi")
    private void stopRunningMetrics() {
        final NetworkLogHandler networkLogHandler = ServiceLocator.getNetworkLogHandler();
        networkLogHandler.forceStop();
        PoolProvider.postMainThreadTask(() -> {
            if (BuildFieldsProvider.INSTANCE.provideBuildVersion() >= Build.VERSION_CODES.JELLY_BEAN) {
                finalizeScreenLoadingCapturing();
                endAutomaticUiTraces();
                final CustomUiTraceHandler customUiTraceHandler = ServiceLocator.getCustomUiTraceHandler();
                if (customUiTraceHandler != null) {
                    customUiTraceHandler.forceStop();
                }
            }
        });
    }

    private static void endAutomaticUiTraces() {
        final UiTracesManager uiTracesManager = UiTracesServiceLocator.getManager();
        if (uiTracesManager != null) uiTracesManager.endAll();
    }

    private static void finalizeScreenLoadingCapturing() {
        final ScreenLoadingManager screenLoadingManager = ScreenLoadingServiceLocator.getManager();
        if (screenLoadingManager != null) screenLoadingManager.endAll();
    }

    @Override
    public void start(Context context) {
        isFirstLaunch = true;
        registerConfigurationChange();
        subscribeToSDKState();
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.JELLY_BEAN) {
            apmLogger.logSDKError("Could not enable Auto UI Trace. Feature is supported on API level 16 and up only.");
        }
    }

    private void clearInvalidCache() {
        final ExecutionTracesHandler executionTracesHandler = ServiceLocator.getExecutionTracesHandler();
        final NetworkLogHandler networkLogHandler = ServiceLocator.getNetworkLogHandler();
        ServiceLocator.getSingleThreadPoolExecutor(Constants.EXECUTION_TRACES_THREAD_EXECUTOR).execute(new Runnable() {
            @Override
            public void run() {
                synchronized (lock) {
                    executionTracesHandler.removeUnEndedTraces();
                }
            }
        });

        ServiceLocator.getSingleThreadPoolExecutor(Constants.NETWORK_LOG_THREAD_EXECUTOR)
                .execute(new Runnable() {
                    @Override
                    public void run() {
                        if (ServiceLocator.getApmConfigurationProvider().isNetworkEnabled()) {
                            synchronized (lock) {
                                networkLogHandler.cleanUp();
                            }
                        }
                    }
                });
    }

    @Override
    public long getLastActivityTime() {
        return 0;
    }

    private void registerConfigurationChange() {
        if (sdkCoreEventsSubscriberDisposable == null || sdkCoreEventsSubscriberDisposable.isDisposed()) {
            sdkCoreEventsSubscriberDisposable = new CompositeDisposable();
        }

        sdkCoreEventsSubscriberDisposable.add(ServiceLocator.getOnSessionCrashedEventBus().subscribe(new Consumer<NDKSessionCrashedEvent>() {
            @Override
            public void accept(NDKSessionCrashedEvent sessionCrashedEvent) {
                sessionHandler.updateSessionEndReason(sessionCrashedEvent.getSessionId(),
                        TimeUnit.MILLISECONDS.toMicros(sessionCrashedEvent.getSessionDuration()),
                        Constants.SessionEndStatusCode.CRASHING_SESSION);
            }
        }));
        getOrCreateCompositeDisposable().add(subscribeToSdkCoreEvents());
    }

    @NonNull
    private IBGDisposable subscribeToSdkCoreEvents() {
        return IBGCoreEventSubscriber.subscribe((this::handleCoreEvent));
    }

    private void handleCoreEvent(IBGSdkCoreEvent event) {
        if (event instanceof V3Session)
            handleV3SessionEvent((V3Session) event);
        else if (event instanceof FeaturesFetched)
            handleFeaturesFetched(((FeaturesFetched) event).getResponse());
        else if (event instanceof IBGSdkCoreEvent.CrossPlatformCrashed)
            updateCurrentSession();
        else if (event instanceof IBGSdkCoreEvent.CrossPlatformScreenChanged)
            handleCPScreenChanged((IBGSdkCoreEvent.CrossPlatformScreenChanged) event);

    }

    private void handleCPScreenChanged(IBGSdkCoreEvent.CrossPlatformScreenChanged event) {
        EventTimeMetricCapture timeMetricCapture = new EventTimeMetricCapture();
        if (SettingsManager.getInstance().getCurrentPlatform() != Platform.FLUTTER) {
            UiTracesServiceLocator.getUiTracesExecutor().execute(() -> {
                String screenName = event.getScreenName();
                UiTracesManager manager = UiTracesServiceLocator.getManager();
                if (manager != null) manager.onScreenChangedSync(
                        screenName,
                        timeMetricCapture.getTimeStampMicro(),
                        timeMetricCapture.getTimeStampMillis()
                );
            });
        }
    }

    private void handleFeaturesFetched(@NonNull String featuresResponse) {
        APMConfigurationHandler apmConfigurationHandler = ServiceLocator.getApmConfigurationHandler();
        boolean didParseCorrectly = apmConfigurationHandler.handleConfiguration(featuresResponse);
        APMConfigurationProvider apmConfigurationProvider = ServiceLocator.getApmConfigurationProvider();
        apmConfigurationProvider.setShouldDependOnV3Session(InstabugCore.isV3SessionEnabled());
        if (didParseCorrectly && apmConfigurationProvider.isAPMEnabled()) {
            Session coreSession = InstabugCore.getRunningSession();
            if (coreSession != null) {
                SessionObserverHandler.register(APMPlugin.this);
                startSession(coreSession);
                registerSessionCrashHandler();
            }
            registerActivityLifeCycleCallbacks();
            registerSessionCrashHandler();
            registerFragmentLifecycleEventListener();
        }
        handleComposeSpansStateChange();
        handleUiTracesStateChanged();
        handleScreenLoadingStateChanged();
        handleWebViewTracesStateChange();
        handleAppFlowStateChange();
    }

    private void handleV3SessionEvent(IBGSdkCoreEvent.V3Session sdkCoreEvent) {
        if (sdkCoreEvent instanceof V3Session.V3SessionStarted) {
            startApmSessionAndMetrics(InstabugCore.getRunningSession());
        } else if (sdkCoreEvent instanceof V3Session.V3SessionFinished) {
            endSession();
        }
    }

    private void updateCurrentSession() {
        Executor sessionExecutor =
                ServiceLocator.getSyncThreadExecutor();
        sessionExecutor.execute(new Runnable() {
            @Override
            public void run() {
                if (InstabugCore.getRunningSession() != null) {
                    // Update the session in the DB
                    sessionHandler.endSession(Constants.SessionEndStatusCode.CRASHING_SESSION);
                }
            }
        });
    }

    private void registerActivityLifeCycleCallbacks() {
        APMConfigurationProvider apmConfigurationProvider = ServiceLocator.getApmConfigurationProvider();
        if (apmConfigurationProvider.isAPMEnabled()) {
            Context context = ServiceLocator.getContext();
            if (context != null && !ActivityCallbacks.isRegistered()) {
                ActivityCallbacks activityCallbacks =
                        ServiceLocator.getActivityCallbacks(context, false);
                if (activityCallbacks != null) {
                    ((Application) context.getApplicationContext())
                            .registerActivityLifecycleCallbacks(activityCallbacks);
                }
            }
        }
    }

    private void startSession(@NonNull final Session coreSession) {
        if (InstabugCore.isV3SessionEnabled() && !coreSession.getVersion().equals(SessionVersion.V3))
            return;

        sessionHandler.startSession(coreSession);
    }

    private void endSession() {
        sessionHandler.endSession(Constants.SessionEndStatusCode.NON_CRASHING_SESSION);
    }

    @WorkerThread
    @Override
    public void onNewSessionStarted(@NonNull final Session runningSession, @Nullable final Session lastSession) {
        if (lastSession != null) {
            NetworkLogMigrationHandler networkLogMigrationHandler = ServiceLocator.getNetworkLogMigrationHandler();
            networkLogMigrationHandler.migrate(runningSession, lastSession);
            ExecutionTracesMigrationHandler executionTracesMigrationHandler = ServiceLocator.getExecutionTracesMigrationHandler();
            executionTracesMigrationHandler.migrate(runningSession, lastSession);
        }
        handleFragmentsOnNewSession(runningSession.getId());
        APMSyncManager apmSyncManager = ServiceLocator.getApmSyncManager();
        apmSyncManager.start();
    }

    private void handleFragmentsOnNewSession(String sessionId) {
        ServiceLocator.getFragmentSpansHelper().onNewSessionStarted(sessionId);
    }

    @Override
    public boolean isFeatureEnabled() {
        return ServiceLocator.getApmConfigurationProvider().isAPMEnabled();
    }


    @NonNull
    @Override
    public FeatureSessionDataController getSessionDataController() {
        return ServiceLocator.getSessionDataController();
    }

    @NonNull
    private IBGCompositeDisposable getOrCreateCompositeDisposable() {
        return compositeDisposable != null ?
                compositeDisposable :
                (compositeDisposable = new IBGCompositeDisposable());
    }

    @NonNull
    @Override
    public Map<String, Boolean> isDataReady(@NonNull List<String> sessionIds) {
        return ServiceLocator.getAPMSessionLazyDataProvider().isDataReady(sessionIds);
    }
}
