package com.instabug.apm.webview.webview_trace.handler

import android.content.ContentValues
import android.database.Cursor
import com.instabug.apm.logger.internal.Logger
import com.instabug.apm.webview.webview_trace.model.WebViewCacheModel
import com.instabug.apm.webview.webview_trace.model.WebViewCacheModelToCursorParser
import com.instabug.apm.webview.webview_trace.model.WebViewsCacheContentValuesMapper
import com.instabug.library.diagnostics.IBGDiagnostics
import com.instabug.library.internal.storage.cache.db.DatabaseManager
import com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMUiTraceEntry
import com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMWebViewTraceEntry
import com.instabug.library.map.Mapper
import com.instabug.library.parse.Parser

interface APMWebViewTraceCacheHandler {
    fun insert(webViewModel: WebViewCacheModel, uiTraceId: Long): Long
    fun retrieve(uiTraceId: Long): List<WebViewCacheModel>?
    fun trim(sessionId: String, limit: Int): Int
    fun trimAll(limit: Int): Int
    fun getSessionIdForUiTrace(uiTraceId: Long): String?
    fun clearAll()
}

class APMWebViewTraceCacheHandlerImpl @JvmOverloads constructor(
    private val databaseManager: DatabaseManager,
    private val logger: Logger,
    private val modelContentValuesMapper: Mapper<Pair<WebViewCacheModel, Long>, ContentValues> = WebViewsCacheContentValuesMapper(),
    private val cursorParser: Parser<Cursor?, List<WebViewCacheModel>?> = WebViewCacheModelToCursorParser()
) : APMWebViewTraceCacheHandler {

    private val databaseWrapper
        get() = databaseManager.openDatabase()

    override fun insert(webViewModel: WebViewCacheModel, uiTraceId: Long): Long = databaseWrapper
        .runCatching {
            insert(
                APMWebViewTraceEntry.TABLE_NAME,
                null,
                modelContentValuesMapper.map(webViewModel to uiTraceId)
            )
        }.onFailure(::reportAndLog).getOrNull() ?: -1

    override fun retrieve(uiTraceId: Long): List<WebViewCacheModel>? = databaseWrapper.runCatching {
        val cursor = query(
            APMWebViewTraceEntry.TABLE_NAME,
            null,
            "${APMWebViewTraceEntry.COLUMN_UI_TRACE_ID} = ?",
            arrayOf("$uiTraceId"), null, null, null
        )
        try {
            cursorParser.parse(cursor)
        } finally {
            cursor?.close()
        }
    }.onFailure(::reportAndLog).getOrNull()

    override fun trim(sessionId: String, limit: Int): Int = databaseWrapper.runCatching {
        databaseWrapper.delete(
            APMWebViewTraceEntry.TABLE_NAME,
            getTrimmingWhereClause(),
            arrayOf(sessionId, sessionId, limit.toString())
        )
    }.onFailure(::reportAndLog).getOrNull() ?: 0

    private fun getTrimmingWhereClause(): String {
        val uiTraceTableName = APMUiTraceEntry.TABLE_NAME
        val webViewTableName = APMWebViewTraceEntry.TABLE_NAME
        val webViewTraceId = "$webViewTableName.${APMWebViewTraceEntry.COLUMN_ID}"
        val uiTraceID = "$uiTraceTableName.${APMUiTraceEntry.COLUMN_ID}"
        val uiTraceIdForeignKey = "$webViewTableName.${APMWebViewTraceEntry.COLUMN_UI_TRACE_ID}"
        val sessionIdFK = "$uiTraceTableName.${APMUiTraceEntry.COLUMN_SESSION_ID}"
        val selectUiTraceIdsPerSessionQuery =
            "SELECT ${APMUiTraceEntry.COLUMN_ID} " +
                    "FROM $uiTraceTableName " +
                    "WHERE ${APMUiTraceEntry.COLUMN_SESSION_ID} = ? "
        val selectByLimitDescendingQuery =
            "SELECT $webViewTraceId " +
                    "FROM $webViewTableName " +
                    "INNER JOIN $uiTraceTableName ON $uiTraceIdForeignKey = $uiTraceID " +
                    "WHERE $sessionIdFK = ? " +
                    "ORDER BY $webViewTraceId DESC " +
                    " LIMIT ?"
        return "${APMWebViewTraceEntry.COLUMN_UI_TRACE_ID} IN ($selectUiTraceIdsPerSessionQuery) " +
                "AND ${APMWebViewTraceEntry.COLUMN_ID} NOT IN ($selectByLimitDescendingQuery)"
    }

    override fun trimAll(limit: Int): Int = databaseWrapper.runCatching {
        val selectByLimitDescendingQuery = "SELECT ${APMWebViewTraceEntry.COLUMN_ID} " +
                "FROM ${APMWebViewTraceEntry.TABLE_NAME} " +
                "ORDER BY ${APMWebViewTraceEntry.COLUMN_ID} DESC " +
                " LIMIT ?"
        val whereClause =
            "${APMWebViewTraceEntry.COLUMN_ID} NOT IN ($selectByLimitDescendingQuery)"
        val whereArgs = arrayOf(limit.toString())
        databaseWrapper.delete(APMWebViewTraceEntry.TABLE_NAME, whereClause, whereArgs)
    }.onFailure(::reportAndLog).getOrNull() ?: 0

    override fun getSessionIdForUiTrace(uiTraceId: Long): String? = databaseWrapper.runCatching {
        query(
            APMUiTraceEntry.TABLE_NAME,
            arrayOf(APMUiTraceEntry.COLUMN_SESSION_ID),
            "${APMUiTraceEntry.COLUMN_ID} = ?",
            arrayOf("$uiTraceId"), null, null, null
        )?.getUiTraceSessionId()
    }.onFailure(::reportAndLog).getOrNull()

    private fun Cursor.getUiTraceSessionId() =
        try {
            takeIf { moveToFirst() }
                ?.let { getColumnIndex(APMUiTraceEntry.COLUMN_SESSION_ID) }
                ?.takeIf { index -> index >= 0 }
                ?.let { index -> getString(index) }
        } finally {
            close()
        }


    override fun clearAll() {
        databaseWrapper
            .runCatching { delete(APMWebViewTraceEntry.TABLE_NAME, null, null) }
            .onFailure(::reportAndLog)
    }

    private fun reportAndLog(throwable: Throwable) {
        val errorMessage = "APM WebViewTraces Database error"
        logger.logSDKErrorProtected(errorMessage, throwable)
        IBGDiagnostics.reportNonFatal(throwable, errorMessage)
    }
}