package com.instabug.apm.handler.experiment

import com.instabug.apm.cache.handler.experiment.ExperimentCacheHandler
import com.instabug.apm.cache.handler.session.SessionMetaDataCacheHandler
import com.instabug.apm.configuration.APMConfigurationProvider
import com.instabug.apm.logger.internal.Logger
import com.instabug.library.core.InstabugCore
import com.instabug.library.diagnostics.IBGDiagnostics
import com.instabug.library.util.threading.PoolProvider
import com.instabug.library.util.threading.ReturnableSingleThreadExecutor

class ExperimentHandlerImpl(
    private val experimentsCacheHandler: ExperimentCacheHandler,
    private val metaDataCacheHandler: SessionMetaDataCacheHandler,
    private val configurationProvider: APMConfigurationProvider,
    private val logger: Logger
) : ExperimentHandler {

    companion object {
        private const val experimentsExecutorIdentifier = "ApmExperiments"
        private const val experimentsDisabledLogMessage =
            "experiments weren't synced as Experiments seems to be disabled for your Instabug " +
                    "company account. Please contact support for more information."
    }

    private val experimentsExecutor: ReturnableSingleThreadExecutor =
        PoolProvider.getReturnableSingleThreadExecutor(experimentsExecutorIdentifier)

    override fun storeExperimentsAsync(sessionId: String) {
        experimentsExecutor.execute {
            try {
                InstabugCore.getExperiments(1.0f)?.ifEmpty { null }?.let { experiments ->
                    val experimentsEnabled = configurationProvider.isExperimentsFeatureEnabled
                    if (experimentsEnabled) {
                        val experimentsTotalCount = experiments.size
                        val experimentsTrimmed = trimExperiments(experiments)
                        experimentsCacheHandler.addExperiments(experimentsTrimmed, sessionId)
                        metaDataCacheHandler.setExperimentsTotalCount(
                            sessionId,
                            experimentsTotalCount
                        )
                    } else {
                        logger.d(experimentsDisabledLogMessage)
                    }
                }
            } catch (exception: Exception) {
                logger.logSDKError("Failed to store experiments", exception)
                IBGDiagnostics.reportNonFatal(exception, "Failed to store experiments")
            }
        }
    }

    override fun getSessionExperimentsAsync(sessionId: String): List<String>? =
        experimentsExecutor.executeAndGet {
            experimentsCacheHandler.getExperiments(sessionId).ifEmpty {
                null
            }
        }

    override fun clearAllExperimentsAsync() {
        experimentsExecutor.execute {
            experimentsCacheHandler.clear()
            metaDataCacheHandler.resetExperimentsCount()
        }
    }

    private fun trimExperiments(experiments: List<String>): List<String> {
        val limitPerRequest = configurationProvider.experimentsLimitPerRequest
        return if (experiments.size <= limitPerRequest) {
            experiments
        } else {
            val listIndexExclusive = experiments.size
            val startIndex = listIndexExclusive - limitPerRequest
            experiments.subList(startIndex, listIndexExclusive)
        }
    }
}