package com.instabug.apm.fragment

import androidx.annotation.VisibleForTesting
import androidx.fragment.app.Fragment
import com.instabug.apm.di.ServiceLocator
import com.instabug.apm.model.EventTimeMetricCapture
import com.instabug.library.diagnostics.IBGDiagnostics
import com.instabug.library.transform.TransformationClass
import com.instabug.library.transform.TransformationMethod
import java.util.*
import java.util.concurrent.Executor

@TransformationClass
class FragmentEventDispatcher {
    companion object {

        @JvmStatic
        @VisibleForTesting
        val listeners =
            Collections.synchronizedSet(HashSet<FragmentLifecycleEventListener>())

        @JvmStatic
        private inline val executor: Executor
            get() = ServiceLocator.getFragmentSpansOrderedExecutor()

        private val logger = ServiceLocator.getApmLogger()

        @JvmStatic
        fun addListener(listener: FragmentLifecycleEventListener) {
            listeners.add(listener)
        }

        @JvmStatic
        fun removeListener(listener: FragmentLifecycleEventListener) {
            listeners.remove(listener)
        }

        @TransformationMethod
        @JvmStatic
        fun onFragmentPreAttach(fragment: Fragment) = dispatchIOEvent(
            fragment,
            "onFragmentPreAttach",
            FragmentLifecycleEventListener::onFragmentPreAttach
        )

        @TransformationMethod
        @JvmStatic
        fun onFragmentPostAttach(fragment: Fragment) = dispatchIOEvent(
            fragment,
            "onFragmentPostAttach",
            FragmentLifecycleEventListener::onFragmentPostAttach
        )

        @TransformationMethod
        @JvmStatic
        fun onFragmentPreCreate(fragment: Fragment) = dispatchIOEvent(
            fragment,
            "onFragmentPreCreate",
            FragmentLifecycleEventListener::onFragmentPreCreate
        )

        @TransformationMethod
        @JvmStatic
        fun onFragmentPostCreate(fragment: Fragment) = dispatchIOEvent(
            fragment,
            "onFragmentPostCreate",
            FragmentLifecycleEventListener::onFragmentPostCreate
        )

        @TransformationMethod
        @JvmStatic
        fun onFragmentPreCreateView(fragment: Fragment) = dispatchIOEvent(
            fragment,
            "onFragmentPreCreateView",
            FragmentLifecycleEventListener::onFragmentPreCreateView
        )

        @TransformationMethod
        @JvmStatic
        fun onFragmentPostCreateView(fragment: Fragment) = dispatchIOEvent(
            fragment,
            "onFragmentPostCreateView",
            FragmentLifecycleEventListener::onFragmentPostCreateView
        )

        @TransformationMethod
        @JvmStatic
        fun onFragmentPreViewCreated(fragment: Fragment) = dispatchIOEvent(
            fragment,
            "onFragmentPreViewCreated",
            FragmentLifecycleEventListener::onFragmentPreViewCreated
        )

        @TransformationMethod
        @JvmStatic
        fun onFragmentPostViewCreated(fragment: Fragment) = dispatchIOEvent(
            fragment,
            "onFragmentPostViewCreated",
            FragmentLifecycleEventListener::onFragmentPostViewCreated
        )

        @TransformationMethod
        @JvmStatic
        fun onFragmentPreActivityCreated(fragment: Fragment) = dispatchIOEvent(
            fragment,
            "onFragmentPreActivityCreated",
            FragmentLifecycleEventListener::onFragmentPreActivityCreated
        )

        @TransformationMethod
        @JvmStatic
        fun onFragmentPostActivityCreated(fragment: Fragment) = dispatchIOEvent(
            fragment,
            "onFragmentPostActivityCreated",
            FragmentLifecycleEventListener::onFragmentPostActivityCreated
        )

        @TransformationMethod
        @JvmStatic
        fun onFragmentPreViewStateRestore(fragment: Fragment) = dispatchIOEvent(
            fragment,
            "onFragmentPreViewStateRestore",
            FragmentLifecycleEventListener::onFragmentPreViewStateRestore
        )

        @TransformationMethod
        @JvmStatic
        fun onFragmentPostViewStateRestore(fragment: Fragment) = dispatchIOEvent(
            fragment,
            "onFragmentPostViewStateRestore",
            FragmentLifecycleEventListener::onFragmentPostViewStateRestore
        )

        @TransformationMethod
        @JvmStatic
        fun onFragmentPreStart(fragment: Fragment) = dispatchIOEvent(
            fragment,
            "onFragmentPreStart",
            FragmentLifecycleEventListener::onFragmentPreStart
        )

        @TransformationMethod
        @JvmStatic
        fun onFragmentPostStart(fragment: Fragment) = dispatchIOEvent(
            fragment,
            "onFragmentPostStart",
            FragmentLifecycleEventListener::onFragmentPostStart
        )

        @TransformationMethod
        @JvmStatic
        fun onFragmentPreResume(fragment: Fragment) = dispatchIOEvent(
            fragment,
            "onFragmentPreResume",
            FragmentLifecycleEventListener::onFragmentPreResume
        )

        @TransformationMethod
        @JvmStatic
        fun onFragmentPostResume(fragment: Fragment) = dispatchIOEvent(
            fragment,
            "onFragmentPostResume",
            FragmentLifecycleEventListener::onFragmentPostResume
        )

        @TransformationMethod
        @JvmStatic
        fun onFragmentPreDeAttach(fragment: Fragment) = dispatchIOEvent(
            fragment,
            "onFragmentPreDeAttach",
            FragmentLifecycleEventListener::onFragmentPreDeAttach
        )

        @TransformationMethod
        @JvmStatic
        fun onFragmentPostDeAttach(fragment: Fragment) = dispatchIOEvent(
            fragment,
            "onFragmentPostDeAttach",
            FragmentLifecycleEventListener::onFragmentPostDeAttach
        )

        private inline fun dispatchIOEvent(
            fragment: Fragment,
            eventName: String,
            crossinline event: ((FragmentLifecycleEventListener, Fragment, EventTimeMetricCapture) -> Unit)
        ) {
            val timeMetric = EventTimeMetricCapture()
            executor.execute { dispatchFragmentEventEvent(event, fragment, timeMetric, eventName) }
        }

        private inline fun dispatchFragmentEventEvent(
            crossinline event: (FragmentLifecycleEventListener, Fragment, EventTimeMetricCapture) -> Unit,
            fragment: Fragment,
            timeMetric: EventTimeMetricCapture,
            eventName: String
        ) = synchronized(listeners) {
            listeners.runCatching {
                forEach { listener -> event.invoke(listener, fragment, timeMetric) }
            }.onFailure {
                val message =
                    "error occurred while handling fragment event $eventName: ${it.message}"
                IBGDiagnostics.reportNonFatal(it, message)
                logger.logSDKProtected(message)
            }
        }
    }
}