package com.instabug.apm.appflow

import com.instabug.apm.appflow.model.AppFlowCacheModel
import com.instabug.apm.cache.model.ExecutionTraceCacheModel
import com.instabug.apm.cache.model.SessionCacheModel
import com.instabug.apm.cache.model.SessionMetaData
import com.instabug.apm.networking.mapping.sessions.SessionFeatureJsonFiller
import com.instabug.apm.networking.mapping.sessions.SessionFeatureJsonFiller.Companion.getDropCount
import com.instabug.library.map.Mapper
import com.instabug.library.util.extenstions.plus
import org.json.JSONArray
import org.json.JSONObject

class SessionAppFlowAndExecutionTraceJsonFiller(
    private val appFlowMapper: Mapper<List<AppFlowCacheModel>?, JSONArray?>,
    private val executionTraceMapper: Mapper<List<ExecutionTraceCacheModel>?, JSONArray?>
) : SessionFeatureJsonFiller {

    companion object {
        const val TRACE_OBJECT_KEY = "ts"
        const val TRACE_LIST_KEY = "tl"
        const val DROP_COUNT_STORE_LIMIT_KEY = "dcsl"
        const val DROP_COUNT_REQUEST_LIMIT_KEY = "dcrl"
    }

    override fun addToJsonObject(model: SessionCacheModel, sessionJsonObject: JSONObject) {
        model.sessionMetaData
            ?.takeIf { hasAppFlowsOrExecutionTraces(it, model.appFlows?.size ?: 0) }
            ?.let { metaData ->
                val (executionTraceDropStoreLimit, executionTraceDropRequestLimit) = getDropCount(
                    model::getExecutionTraces,
                    metaData::getExecutionTracesTotalCount,
                    metaData::getExecutionTracesDroppedCount
                )
                val (appFlowDropStoreLimit, appFlowDropRequestLimit) = getDropCount(
                    model::getAppFlows,
                    metaData::getAppFlowTotalCount,
                    metaData::getAppFlowDroppedCountRequestLimit
                )
                val dropCountStoreLimit = executionTraceDropStoreLimit + appFlowDropStoreLimit
                val dropCountRequestLimit = executionTraceDropRequestLimit + appFlowDropRequestLimit
                val tracesArray =
                    executionTraceMapper.map(model.executionTraces) + appFlowMapper.map(model.appFlows)

                val tracesJsonObject = JSONObject().apply {
                    dropCountStoreLimit.takeIf { it > 0 }
                        ?.let { put(DROP_COUNT_STORE_LIMIT_KEY, it) }
                    dropCountRequestLimit.takeIf { it > 0 }
                        ?.let { put(DROP_COUNT_REQUEST_LIMIT_KEY, it) }
                    tracesArray?.takeIf { it.length() > 0 }
                        ?.let { put(TRACE_LIST_KEY, it) }
                }
                sessionJsonObject.put(TRACE_OBJECT_KEY, tracesJsonObject)
            }
    }

    private fun hasAppFlowsOrExecutionTraces(metaData: SessionMetaData, cachedAppFlowsCount: Int) =
        cachedAppFlowsCount + metaData.appFlowTotalCount + metaData.executionTracesTotalCount > 0
}