package com.instabug.apm.webview.webview_trace.handler

import com.instabug.apm.cache.handler.session.SessionMetaDataCacheHandler
import com.instabug.apm.logger.internal.Logger
import com.instabug.apm.webview.webview_trace.configuration.WebViewTraceConfigurationProvider
import com.instabug.apm.webview.webview_trace.model.WebViewCacheModel

interface WebViewTraceHandler {
    fun save(model: WebViewCacheModel, uiTraceId: Long)
    fun clearAll()
}

class WebViewTraceHandlerImpl(
    private val webViewsCacheHandler: APMWebViewTraceCacheHandler,
    private val metaDataCacheHandler: SessionMetaDataCacheHandler,
    private val configurations: WebViewTraceConfigurationProvider,
    private val logger: Logger
) : WebViewTraceHandler {

    override fun save(model: WebViewCacheModel, uiTraceId: Long) {
        if (!configurations.enabled) return
        webViewsCacheHandler.getSessionIdForUiTrace(uiTraceId)?.also { sessionId ->
            webViewsCacheHandler.insert(model, uiTraceId)
                .takeIf { it != -1L }
                ?.let {
                    metaDataCacheHandler.addToWebViewTracesTotalCount(sessionId, 1)
                    val droppedRequestLimit =
                        webViewsCacheHandler.trim(sessionId, configurations.requestLimit)
                    webViewsCacheHandler.trimAll(configurations.storeLimit)
                    droppedRequestLimit
                }?.takeIf { it > 0 }
                ?.also { droppedCount ->
                    metaDataCacheHandler.addToWebViewTracesDroppedCount(sessionId, droppedCount)
                    logger.d("$WEB_VIEW_TRACE_DROPPED_COUNT $droppedCount")
                }
        } ?: let { logger.e(TRACE_NOT_SAVED_NO_SESSION_FOUND) }
    }

    override fun clearAll() {
        webViewsCacheHandler.clearAll()
        metaDataCacheHandler.resetWebViewTracesCounts()
    }

    companion object {
        private const val WEB_VIEW_TRACE_DROPPED_COUNT = "Web Views spans dropped count:"
        private const val TRACE_NOT_SAVED_NO_SESSION_FOUND =
            "No sessions found, skipping saving WebView trace"
    }
}