@file:TransformationClass

package com.instabug.apm.webview.dispatch.util

import android.annotation.SuppressLint
import android.os.Build
import android.webkit.WebView
import android.webkit.WebViewClient
import androidx.annotation.RequiresApi
import com.instabug.apm.webview.dispatch.IbgWebViewClient
import com.instabug.library.transform.TransformationClass
import com.instabug.library.transform.TransformationMethod

@TransformationMethod
fun WebView.ibgSetWebViewClient(client: WebViewClient?) {
    if (isSupportedAndroidAPI()) {
        @SuppressLint("WebViewApiAvailability") val currentClient: WebViewClient? = webViewClient
        if (currentClient is IbgWebViewClient) {
            currentClient.delegate = client
        } else {
            WebViewDelegator.setWebViewClient(this, IbgWebViewClient(client))
        }
    } else {
        WebViewDelegator.setWebViewClient(this, client)
    }
}

@TransformationMethod
fun WebView.ibgLoadUrl(url: String?) {
    injectAndRunOnWebViewClientIfPossible { it.onLoadUrl(this, url) }
    WebViewDelegator.loadUrl(this, url)
}

@TransformationMethod
fun WebView.ibgLoadUrl(url: String?, additionalHttpHeaders: Map<String, String>?) {
    injectAndRunOnWebViewClientIfPossible { it.onLoadUrl(this, url, additionalHttpHeaders) }
    WebViewDelegator.loadUrl(this, url, additionalHttpHeaders)
}

@TransformationMethod
fun WebView.ibgPostUrl(url: String?, postData: ByteArray?) {
    injectAndRunOnWebViewClientIfPossible { it.onPostUrl(this, url, postData) }
    WebViewDelegator.postUrl(this, url, postData)
}

@TransformationMethod
fun WebView.ibgLoadData(data: String?, mimeType: String?, encoding: String?) {
    injectAndRunOnWebViewClientIfPossible { it.onLoadData(this, data, mimeType, encoding) }
    WebViewDelegator.loadData(this, data, mimeType, encoding)
}

@TransformationMethod
fun WebView.ibgLoadDataWithBaseURL(
    baseUrl: String?,
    data: String?,
    mimeType: String?,
    encoding: String?,
    historyUrl: String?
) {
    injectAndRunOnWebViewClientIfPossible {
        it.onLoadDataWithBaseUrl(this, baseUrl, data, mimeType, encoding)
    }
    WebViewDelegator.loadDataWithBaseURL(this, baseUrl, data, mimeType, encoding, historyUrl)
}

private inline fun WebView.injectAndRunOnWebViewClientIfPossible(action: (IbgWebViewClient) -> Unit) {
    if (isSupportedAndroidAPI()) action(injectClient())
}

@SuppressLint("AnnotateVersionCheck")
private fun isSupportedAndroidAPI(): Boolean = Build.VERSION.SDK_INT >= Build.VERSION_CODES.O

@RequiresApi(Build.VERSION_CODES.O)
private fun WebView.injectClient(): IbgWebViewClient {
    val currentClient = webViewClient
    return if (currentClient is IbgWebViewClient) currentClient
    else IbgWebViewClient(currentClient).also { webViewClient = it }
}