package com.instabug.apm.uitrace.manager

import android.content.Context
import android.os.Build
import androidx.annotation.RequiresApi
import com.instabug.apm.configuration.APMConfigurationProvider
import com.instabug.apm.di.Provider
import com.instabug.apm.logger.internal.Logger
import com.instabug.apm.model.EventTimeMetricCapture
import com.instabug.apm.uitrace.activitycallbacks.APMUiTraceActivityCallbacks
import com.instabug.apm.uitrace.activitycallbacks.CompositeApmUiTraceActivityCallbacks
import com.instabug.apm.uitrace.handler.CPAutomaticUiTracesHandlerImpl
import com.instabug.apm.uitrace.repo.UiTracesRepo
import com.instabug.apm.uitrace.util.getUiTraceEndParams
import com.instabug.apm.util.device.APMDeviceStateProvider
import com.instabug.apm.util.runOrReportAPMError
import com.instabug.library.Platform
import com.instabug.library.settings.SettingsManager
import com.instabug.library.tracking.InstabugInternalTrackingDelegate
import java.util.concurrent.Executor

interface UiTracesManager {

    /**
     * Registers the required handlers to start capturing of automatic ui traces if the feature is enabled
     */
    fun start()

    /**
     * If feature is enabled registers the required handlers to start capturing of automatic ui traces
     *
     * Else it unregisters the handlers, cleanup in memory traces and clears database
     */
    fun onStateChanged()

    /**
     * Registers the required handlers to start capturing of automatic ui traces on the same thread without checking feature availability
     */
    fun startSynchronous()

    /**
     * End and clean all active ui traces executed on the caller thread
     */
    fun endAll()

    /**
     * Report screen change from cp
     */
    fun onScreenChanged(
        screenName: String?,
        timeStampMicro: Long,
        uiTraceId: Long
    )
}

@RequiresApi(Build.VERSION_CODES.JELLY_BEAN)
class UiTracesManagerImpl(
    private val compositeApmUiTraceActivityCallbacks: CompositeApmUiTraceActivityCallbacks,
    private val nativeAutomaticUiTraceHandlerProvider: Provider<APMUiTraceActivityCallbacks>,
    private val cpAutomaticUiTraceHandlerProvider: Provider<CPAutomaticUiTracesHandlerImpl>,
    private val customUiTracesHandlerActivityCallbacksProvider: Provider<APMUiTraceActivityCallbacks?>,
    private val configurationProvider: APMConfigurationProvider,
    private val executor: Executor,
    private val repo: UiTracesRepo,
    private val internalTrackingDelegate: InstabugInternalTrackingDelegate,
    private val deviceStateProvider: APMDeviceStateProvider,
    private val settingsManager: SettingsManager,
    private val logger: Logger,
    private val contextProvider: Provider<Context?>
) : UiTracesManager {

    private val isNativeApp
        get() = settingsManager.getEarlyCurrentPlatform(contextProvider()) == Platform.ANDROID

    private val canCaptureCustomUiTraces
        get() = isNativeApp && configurationProvider.isUiHangsFeatureEnabled

    private var nativeAutomaticUiTraceHandler: APMUiTraceActivityCallbacks? = null
    private var cpAutomaticUiTraceHandler: CPAutomaticUiTracesHandlerImpl? = null

    override fun start() = executor.execute { startSynchronous() }

    override fun startSynchronous() {
        runOrReportAPMError("Error while initializing Ui traces feature", logger) {
            if (canCaptureCustomUiTraces) customUiTracesHandlerActivityCallbacksProvider()?.start()
            if (configurationProvider.isUiTraceEnabled) startAutomaticUiTraces()
        }
    }

    override fun onStateChanged() = executor.execute {
        runOrReportAPMError("Error while handling UiTrace feature state changed", logger) {
            onAutomaticUiTracesStateChanged()
            onCustomUiTraceStateChanged()
        }
    }

    override fun endAll() {
        val endTimeInMicros = getEndTimeInMicros()
        executor.execute {
            runOrReportAPMError("An error occurred while ending all UiTraces", logger) {
                internalTrackingDelegate
                    .currentActivity
                    ?.run { repo.endAll(getUiTraceEndParams(deviceStateProvider, endTimeInMicros)) }
            }
        }
    }

    override fun onScreenChanged(
        screenName: String?,
        timeStampMicro: Long,
        uiTraceId: Long
    ) {
        cpAutomaticUiTraceHandler?.onScreenChanged(
            screenName,
            timeStampMicro,
            uiTraceId
        )
    }

    private fun onAutomaticUiTracesStateChanged() {
        if (configurationProvider.isUiTraceEnabled) {
            startAutomaticUiTraces()
            clearUiHangsIfNotEnabled()
        } else {
            stopAutomaticUiTraces()
        }
    }

    private fun startAutomaticUiTraces() =
        if (isNativeApp) {
            registerNativeAutomaticUiTraceHandler()
        } else {
            registerCPAutomaticUiTraceHandler()
        }

    private fun clearUiHangsIfNotEnabled() =
        repo.takeUnless { configurationProvider.isAutoUiHangsEnabled }?.clearUiHangs()

    private fun stopAutomaticUiTraces() {
        unregisterNativeAutomaticUiTraceHandler()
        unregisterCPAutomaticUiTraceHandler()
        repo.clearAll()
    }

    private fun onCustomUiTraceStateChanged() =
        customUiTracesHandlerActivityCallbacksProvider()?.run {
            if (canCaptureCustomUiTraces) start() else stop()
        }

    private fun registerNativeAutomaticUiTraceHandler() {
        if (nativeAutomaticUiTraceHandler == null) {
            nativeAutomaticUiTraceHandler =
                nativeAutomaticUiTraceHandlerProvider().start()
        }
    }

    private fun unregisterNativeAutomaticUiTraceHandler() {
        nativeAutomaticUiTraceHandler?.stop()
        nativeAutomaticUiTraceHandler = null
    }

    private fun registerCPAutomaticUiTraceHandler() {
        if (cpAutomaticUiTraceHandler == null) {
            cpAutomaticUiTraceHandler = cpAutomaticUiTraceHandlerProvider()
                .apply { start() }
        }
    }

    private fun unregisterCPAutomaticUiTraceHandler() {
        cpAutomaticUiTraceHandler?.stop()
        cpAutomaticUiTraceHandler = null
    }

    private fun APMUiTraceActivityCallbacks.start() = apply {
        observeAPMSessions()
        compositeApmUiTraceActivityCallbacks.add(this)
    }

    private fun APMUiTraceActivityCallbacks.stop() = apply {
        stopObservingAPMSessions()
        compositeApmUiTraceActivityCallbacks.remove(this)
    }

    private fun getEndTimeInMicros() = EventTimeMetricCapture().run {
        if (isNativeApp) getMicroTime() else getTimeStampMicro()
    }
}