package com.instabug.apm.screenloading.repo

import androidx.annotation.VisibleForTesting
import com.instabug.apm.cache.handler.uitrace.UiLoadingMetricCacheHandler
import com.instabug.apm.cache.model.UiLoadingModel
import com.instabug.apm.constants.UiLoadingMetric
import com.instabug.apm.di.ServiceLocator
import com.instabug.apm.handler.uitrace.uiloading.UiLoadingMetricHandler
import com.instabug.apm.model.EventTimeMetricCapture
import com.instabug.apm.sanitization.Validator
import com.instabug.apm.screenloading.UiLoadingMetricHandlerFactory
import com.instabug.apm.screenloading.di.ScreenLoadingServiceLocator
import com.instabug.library.factory.Factory
import com.instabug.library.util.collections.LimitedLinkedHashmap

data class ScreenLoadingWrapper(
    val uiLoadingMetricHandler: UiLoadingMetricHandler,
    var uiTraceId: Long
)

interface NativeScreenLoadingRepo {
    fun start(screenName: String, uiTraceId: Long)
    fun startIfNotStarted(screenName: String, uiTraceId: Long)
    fun end(screenName: String)
    fun setEndScreenLoadingStage(screenName: String, timeMetric: EventTimeMetricCapture): Boolean
    fun clearAll()
    fun endAll()
    fun addStage(
        screenName: String,
        timeMetric: EventTimeMetricCapture,
        @UiLoadingMetric.ActivityEvents activityEvent: Int
    )

    fun addResumedPostRunStage(screenName: String, timeMetric: EventTimeMetricCapture)
}

class NativeScreenLoadingRepoImpl(
    private val uiLoadingMetricHandlerFactory: Factory<UiLoadingMetricHandler>,
    private val screenLoadingCacheHandler: UiLoadingMetricCacheHandler,
    private val nativeScreenLoadingStagesValidator: Validator<UiLoadingModel>
) : NativeScreenLoadingRepo {

    @VisibleForTesting
    val screenLoadingHandlers = LimitedLinkedHashmap<String, ScreenLoadingWrapper>(5)

    override fun start(screenName: String, uiTraceId: Long) {
        screenLoadingHandlers[screenName] = ScreenLoadingWrapper(
            uiLoadingMetricHandlerFactory.create(),
            uiTraceId
        )
    }

    override fun startIfNotStarted(screenName: String, uiTraceId: Long) {
        if (screenLoadingHandlers[screenName] == null) {
            start(screenName, uiTraceId)
        }
    }

    override fun end(screenName: String) {
        screenLoadingHandlers[screenName]?.also(::insertModelIfValid)
        screenLoadingHandlers.remove(screenName)
    }

    override fun endAll() {
        screenLoadingHandlers.onEach { (_, wrapper) -> insertModelIfValid(wrapper) }
            .clear()
    }

    private fun insertModelIfValid(wrapper: ScreenLoadingWrapper) = wrapper
        .uiLoadingMetricHandler
        .uiLoadingModel
        ?.takeIf { nativeScreenLoadingStagesValidator.isValid(it) }
        ?.let { model -> screenLoadingCacheHandler.insert(model, wrapper.uiTraceId) }

    override fun clearAll() {
        screenLoadingHandlers.clear()
    }

    override fun addStage(
        screenName: String,
        timeMetric: EventTimeMetricCapture,
        activityEvent: Int
    ) = doRepoOperation(screenName) { it?.addActivityEvent(activityEvent, timeMetric) }

    override fun addResumedPostRunStage(screenName: String, timeMetric: EventTimeMetricCapture) =
        doRepoOperation(screenName) {
            it?.takeIf {
                it.hasEventNeverBeenReportedBefore(UiLoadingMetric.ON_ACTIVITY_RESUMED_POST_RUN)
            }?.run {
                addActivityEvent(UiLoadingMetric.ON_ACTIVITY_RESUMED_POST_RUN, timeMetric)
            }
        }

    override fun setEndScreenLoadingStage(
        screenName: String,
        timeMetric: EventTimeMetricCapture
    ): Boolean {
        var endedSuccessfully = false
        doRepoOperation(screenName) {
            endedSuccessfully = it?.onScreenLoadingEnded(timeMetric) != null
        }
        return endedSuccessfully
    }

    private inline fun doRepoOperation(
        screenName: String,
        operation: (UiLoadingMetricHandler?) -> Unit
    ) = operation(screenLoadingHandlers[screenName]?.uiLoadingMetricHandler)
}

class NativeScreenLoadingRepoFactory : Factory<NativeScreenLoadingRepo> {
    override fun create(): NativeScreenLoadingRepo = NativeScreenLoadingRepoImpl(
        UiLoadingMetricHandlerFactory(),
        ServiceLocator.getUiLoadingMetricCacheHandler(),
        ScreenLoadingServiceLocator.nativeScreenLoadingStagesValidator
    )

}